#pragma once

#include "data_access.h"
#include "dataset_holder.h"
#include "design_registry.h"
#include "permission_checker.h"
#include "profiler.h"
#include "tools.h"

#include <maps/wikimap/mapspro/services/mrc/libs/blackbox_client/include/blackbox_client.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/revision_loader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/privacy/include/region_privacy.h>
#include <maps/wikimap/mapspro/services/mrc/libs/ugc_event_logger/include/logger.h>

#include <maps/infra/yacare/include/metrics.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/renderer/libs/design/include/designs.h>
#include <maps/renderer/libs/style2_renderer/include/render_params.h>
#include <yandex/maps/mds/mds.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <chrono>
#include <filesystem>
#include <memory>
#include <optional>
#include <shared_mutex>
#include <string>
#include <variant>

namespace maps::mrc::browser {

const std::string DEFAULT_DESIGN_ICONS_PATH = "/usr/share/yandex/maps/mrc/browser/icons";
const std::string DEFAULT_GEOID_COVERAGE_PATH = "/var/lib/yandex/maps/ecstatic/data/yandex-maps-coverage5-geoid";
const std::string DEFAULT_DESIGN_PATH = "/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-design";
const std::string ROAD_GRAPH_DATASET_NAME = "road_graph";
const std::string PEDESTRIAN_GRAPH_DATASET_NAME = "pedestrian_graph";
const std::string ROAD_PHOTO_TO_EDGE_DATASET_NAME = "road_photo_to_edge";
const std::string PEDESTRIAN_PHOTO_TO_EDGE_DATASET_NAME = "pedestrian_photo_to_edge";
const std::string REGION_PRIVACY_DATASET_NAME = "region_privacy";
const std::string MRC_DATASET_NAME = "mrc";
// alias for service in tvmtool config
const std::string SELF_TVM_SERVICE_ALIAS = "maps-core-nmaps-mrc-browser";

inline const std::string LAYER_AGE = "mrc_age";
inline const std::string LAYER_AGE_PEDESTRIAN = "mrc_age_p";
inline const std::string LAYER_AGE_ROAD = "mrc_age_r";
inline const std::string LAYER_SIGNAL_SEQUENCES = "mrcss";
inline const std::string LAYER_EDGES = "mrce";
inline const std::string LAYER_EDGES_COVERAGE = "mrce_cov";
inline const std::string LAYER_PEDESTRIAN_EDGES = "mrcpe";
inline const std::string LAYER_PEDESTRIAN_EDGES_NIGHT = "mrcpent";
inline const std::string LAYER_PHOTO_AUTOMOTIVE = "mrcauto";
inline const std::string LAYER_PHOTO_PEDESTRIAN = "mrcpdst";

using namespace std::chrono_literals;

using IBlackboxClientHolder = std::unique_ptr<blackbox_client::IBlackboxClient>;

class Configuration;
using ConfigurationPtr = std::shared_ptr<Configuration>;

#define DATASET_AGE_MAX_METRIC "dataset_age_max"
#define DATASET_AGE_DIFF_METRIC "dataset_age_diff"

YCR_CREATE_METRIC(DATASET_AGE_MAX_METRIC, YCR_HGRAM, 0, 4320, 4320);
YCR_CREATE_METRIC(DATASET_AGE_DIFF_METRIC, YCR_HGRAM, 0, 4320, 4320);

class Configuration {

public:
    Configuration(
            const common::Config& mrcCfg,
            const wiki::common::ExtendedXmlDoc& wikiCfg,
            const std::string& graphFolder,
            const std::string& pedestrianGraphFolder,
            const std::string& photoToEdgeFolder,
            const std::string& photoToPedestrianEdgeFolder,
            const std::string& geoIdCoveragePath,
            const std::optional<std::string>& mrcDataFolder,
            const std::string& designIconsPath,
            const std::string& designPath,
            IBlackboxClientHolder blackboxClient,
            IPermissionCheckerHolder permissionChecker)
        : blackboxClient_(std::move(blackboxClient))
        , mrcSharedPool_(mrcCfg.makeSharedPool())
        , wikiPoolHolder_(wikiCfg, "core", "tasks")
        , mds_(mrcCfg.makeMdsClient())
        , publicMds_(mrcCfg.makePublicMdsClient())
        , permissionChecker_(std::move(permissionChecker))
        , hiddenFeatureRedirectUrl_(mrcCfg.externals().mrcBrowserProUrl())
        , s3MdsUrl_(mrcCfg.externals().s3MdsUrl())
        , stvdescrUrl_(mrcCfg.externals().stvdescrUrl())
        , roadGraph_(
                std::in_place_type<ReloadableDatasetHolder<Graph>>,
                ROAD_GRAPH_DATASET_NAME,
                [](const std::string& path) {
                    TimeProfiler timer("Load dataset " + ROAD_GRAPH_DATASET_NAME, 2000);
                    return std::make_shared<RoadGraph>(path, EMappingMode::Locked);
                },
                graphFolder)
        , pedestrianGraph_(
                std::in_place_type<ReloadableDatasetHolder<Graph>>,
                PEDESTRIAN_GRAPH_DATASET_NAME,
                [](const std::string& path) {
                    TimeProfiler timer("Load dataset " + PEDESTRIAN_GRAPH_DATASET_NAME, 2000);
                    return std::make_shared<PedestrianGraph>(path, EMappingMode::Locked);
                },
                pedestrianGraphFolder)
        , roadPhotoToEdge_(
                std::in_place_type<ReloadableDatasetHolder<PhotoToEdge>>,
                ROAD_PHOTO_TO_EDGE_DATASET_NAME,
                [](const std::string& path) {
                    TimeProfiler timer("Load dataset " + ROAD_PHOTO_TO_EDGE_DATASET_NAME, 2000);
                    return std::make_shared<RoadPhotoToEdge>(path, EMappingMode::Locked);
                },
                photoToEdgeFolder)
        , pedestrianPhotoToEdge_(
                std::in_place_type<ReloadableDatasetHolder<PhotoToEdge>>,
                PEDESTRIAN_PHOTO_TO_EDGE_DATASET_NAME,
                [](const std::string& path) {
                    TimeProfiler timer("Load dataset " + PEDESTRIAN_PHOTO_TO_EDGE_DATASET_NAME, 2000);
                    return std::make_shared<PedestrianPhotoToEdge>(path, EMappingMode::Locked);
                },
                photoToPedestrianEdgeFolder)
        , regionPrivacy_(
                std::in_place_type<ReloadableDatasetHolder<privacy::RegionPrivacy>>,
                REGION_PRIVACY_DATASET_NAME,
                [this](const std::string& coveragePath)
                {
                    TimeProfiler timer("Load dataset " + REGION_PRIVACY_DATASET_NAME, 2000);
                    auto objectLoader = mrc::object::makeRevisionLoader(wikiPoolHolder_.pool().slaveTransaction());
                    return privacy::makeCachingRegionPrivacy(*objectLoader, coveragePath, privacy::LockMemory::Yes);
                },
                geoIdCoveragePath
            )
        , dataAccess_(makeDataAccess(mrcDataFolder))
        , designRegistry_(designIconsPath, designPath)
        , switchingToTargetStateWorker_{[&] {
            using namespace std::chrono_literals;
            do {
                try {
                    switchToTargetStateWorkCycle();
                } catch (const maps::Exception& e) {
                    ERROR() << "Switching to target state worker: " << e;
                } catch (const std::exception& e) {
                    ERROR() << "Switching to target state worker: " << e.what();
                }
            } while (!semaphore_.waitForStop(1s));
            INFO() << "The worker has finished";
        }}
    {}

    ~Configuration()
    {
        INFO() << "Configuration stopping";
        semaphore_.stop();
        switchingToTargetStateWorker_.join();
        INFO() << "Configuration stopped";
    }

    static ConfigurationPtr instance() {
        auto configuration = Configuration::rwInstance();
        REQUIRE(configuration, "Configuration is not defined!");
        return configuration;
    }

    static ConfigurationPtr swap(ConfigurationPtr configuration) {
        std::swap(rwInstance(), configuration);
        return configuration;
    }

    blackbox_client::IBlackboxClient& blackboxClient() {
        ASSERT(blackboxClient_);
        return *blackboxClient_;
    }

    pgpool3::Pool& mrcPool() {
        return mrcSharedPool_->pool();
    }

    common::SharedPool mrcSharedPool() {
        return mrcSharedPool_;
    }

    pgpool3::Pool& wikiPool() {
        return wikiPoolHolder_.pool();
    }

    maps::mds::Mds& mds() { return mds_; }

    const maps::mds::Mds& publicMds() { return publicMds_; }

    IPermissionChecker& permissionChecker() { return *permissionChecker_; }

    http::URL& hiddenFeatureRedirectUrl() { return hiddenFeatureRedirectUrl_; }

    const http::URL& s3MdsUrl() const { return s3MdsUrl_; }

    const http::URL& stvdescrUrl() const { return stvdescrUrl_; }

    std::shared_ptr<IGraph> roadGraph()
    {
        std::shared_lock lock{guard_};
        if (auto graphPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&roadGraph_)) {
            return graphPtr->dataset();
        } else if (auto graphPtr = std::get_if<IGraphPtr>(&roadGraph_)) {
            return *graphPtr;
        }
        REQUIRE(false, "Could not get roadGraph");
    }

    std::shared_ptr<IGraph> pedestrianGraph()
    {
        std::shared_lock lock{guard_};
        if (auto graphPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&pedestrianGraph_)) {
            return graphPtr->dataset();
        } else if (auto graphPtr = std::get_if<IGraphPtr>(&pedestrianGraph_)) {
            return *graphPtr;
        }
        REQUIRE(false, "Could not get pedestrianGraph");
    }

    IPhotoToEdgePtr roadPhotoToEdge()
    {
        auto lock = std::shared_lock{guard_};
        if (auto ptr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&roadPhotoToEdge_)) {
            return ptr->dataset();
        } else if (auto ptr = std::get_if<IPhotoToEdgePtr>(&roadPhotoToEdge_)) {
            return *ptr;
        }
        REQUIRE(false, "Could not get roadPhotoToEdge");
    }

    IPhotoToEdgePtr pedestrianPhotoToEdge()
    {
        auto lock = std::shared_lock{guard_};
        if (auto ptr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&pedestrianPhotoToEdge_)) {
            return ptr->dataset();
        } else if (auto ptr = std::get_if<IPhotoToEdgePtr>(&pedestrianPhotoToEdge_)) {
            return *ptr;
        }
        REQUIRE(false, "Could not get pedestrianPhotoToEdge");
    }

    std::shared_ptr<IGraph> graphByType(db::GraphType graphType)
    {
        if (graphType == db::GraphType::Road) {
            return roadGraph();
        } else if (graphType == db::GraphType::Pedestrian) {
            return pedestrianGraph();
        }
        REQUIRE(false, "Unsupported value of graphType " << graphType);
    }

    std::shared_ptr<IGraph> graphByTileLayer(const std::string& layer)
    {
        static const std::map<std::string, db::GraphType> layerToGraphType{
            {LAYER_AGE_PEDESTRIAN, db::GraphType::Pedestrian},
            {LAYER_AGE_ROAD, db::GraphType::Road},
            {LAYER_EDGES, db::GraphType::Road},
            {LAYER_EDGES_COVERAGE, db::GraphType::Road},
            {LAYER_PEDESTRIAN_EDGES, db::GraphType::Pedestrian},
            {LAYER_PEDESTRIAN_EDGES_NIGHT, db::GraphType::Road},
            {LAYER_PHOTO_AUTOMOTIVE, db::GraphType::Road},
            {LAYER_PHOTO_PEDESTRIAN, db::GraphType::Pedestrian}};

        const auto iter = layerToGraphType.find(layer);
        REQUIRE(
            iter != layerToGraphType.end(),
            "Layer to graph mapping is not defined");
        return graphByType(iter->second);
    }

    std::shared_ptr<privacy::RegionPrivacy> regionPrivacy()
    {
        std::shared_lock lock{guard_};
        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<privacy::RegionPrivacy>>(&regionPrivacy_)) {
            return datasetPtr->dataset();
        } else if (auto privacyPtr = std::get_if<std::shared_ptr<privacy::RegionPrivacy>>(&regionPrivacy_)) {
            return *privacyPtr;
        }
        REQUIRE(false, "Could not get regionPrivacy");
    }

    std::shared_ptr<ugc_event_logger::Logger> ugcEventLogger() {
        return ugcEventLoggerPtr_;
    }

    std::shared_ptr<IDataAccess> dataAccess()
    {
        std::shared_lock lock{guard_};
        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<FbDataAccess>>(&dataAccess_)) {
            return datasetPtr->dataset();
        } else if (auto datasetPtr = std::get_if<std::shared_ptr<IDataAccess>>(&dataAccess_)) {
            return *datasetPtr;
        }
        REQUIRE(false, "Could not get dataAccess");
    }

    bool areAllDatasetsInTargetState() const
    {
        return isRoadGraphInTargetState() &&
            isPedestrianGraphInTargetState() &&
            isRoadPhotoToEdgeInTargetState() &&
            isPedestrianPhotoToEdgeInTargetState() &&
            isRegionPrivacyInTargetState() &&
            isDataAccessInTargetState();
    }

    bool areAllDatasetsLoaded() const
    {
        return isRoadGraphLoaded() &&
            isPedestrianGraphLoaded() &&
            isRoadPhotoToEdgeLoaded() &&
            isPedestrianPhotoToEdgeLoaded() &&
            isRegionPrivacyLoaded() &&
            isDataAccessLoaded();
    }

    void setRoadGraphDatasetTargetPath(
        const std::optional<std::string>& path,
        std::optional<SteadyTimePoint> timeToStartSwitchToTargetState =
            std::nullopt)
    {
        std::shared_lock lock{guard_};

        if (auto graphPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&roadGraph_)) {
            graphPtr->setTargetDatasetPath(path, timeToStartSwitchToTargetState);
            semaphore_.notifyAll();
            return;
        }

        REQUIRE(false, "Could not get roadGraph");
    }

    void setPedestrianGraphDatasetTargetPath(
        const std::optional<std::string>& path,
        std::optional<SteadyTimePoint> timeToStartSwitchToTargetState =
            std::nullopt)
    {
        std::shared_lock lock{guard_};

        if (auto graphPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&pedestrianGraph_)) {
            graphPtr->setTargetDatasetPath(path, timeToStartSwitchToTargetState);
            semaphore_.notifyAll();
            return;
        }

        REQUIRE(false, "Could not get pedestrianGraph");
    }

    void setRoadPhotoToEdgeDatasetTargetPath(
        const std::optional<std::string>& path,
        std::optional<SteadyTimePoint> timeToStartSwitchToTargetState = std::nullopt)
    {
        auto lock = std::shared_lock{guard_};

        if (auto ptr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&roadPhotoToEdge_)) {
            ptr->setTargetDatasetPath(path, timeToStartSwitchToTargetState);
            semaphore_.notifyAll();
            return;
        }

        REQUIRE(false, "Could not get roadPhotoToEdge");
    }

    void setPedestrianPhotoToEdgeDatasetTargetPath(
        const std::optional<std::string>& path,
        std::optional<SteadyTimePoint> timeToStartSwitchToTargetState = std::nullopt)
    {
        auto lock = std::shared_lock{guard_};

        if (auto ptr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&pedestrianPhotoToEdge_)) {
            ptr->setTargetDatasetPath(path, timeToStartSwitchToTargetState);
            semaphore_.notifyAll();
            return;
        }

        REQUIRE(false, "Could not get pedestrianPhotoToEdge");
    }

    void setRegionPrivacyDatasetTargetPath(
        const std::optional<std::string>& path,
        std::optional<SteadyTimePoint> timeToStartSwitchToTargetState =
            std::nullopt)
    {
        std::shared_lock lock{guard_};
        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<privacy::RegionPrivacy>>(&regionPrivacy_)) {
            datasetPtr->setTargetDatasetPath(path, timeToStartSwitchToTargetState);
            semaphore_.notifyAll();
            return;
        }
        REQUIRE(false, "Could not get regionPrivacy");
    }

    void setDataAccessDatasetTargetPath(
        const std::optional<std::string>& path,
        std::optional<SteadyTimePoint> timeToStartSwitchToTargetState =
            std::nullopt)
    {
        std::shared_lock lock{guard_};
        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<FbDataAccess>>(&dataAccess_)) {
            datasetPtr->setTargetDatasetPath(path, timeToStartSwitchToTargetState);
            semaphore_.notifyAll();
            return;
        }
        REQUIRE(false, "Could not get dataAccess");
    }

    void setUgcEventLogger(std::shared_ptr<ugc_event_logger::Logger> ugcEventLogger)
    {
        ugcEventLoggerPtr_ = std::move(ugcEventLogger);
    }

    /// These fucntions are used only in tests

    IBlackboxClientHolder swapBlackboxClient(IBlackboxClientHolder client)
    {
        ASSERT(client);
        std::swap(client, blackboxClient_);
        return client;
    }

    IPermissionCheckerHolder swapPermissionChecker(IPermissionCheckerHolder checker)
    {
        ASSERT(checker);
        std::swap(checker, permissionChecker_);
        return checker;
    }

    void setRegionPrivacy(privacy::RegionPrivacyPtr regionPrivacy)
    {
        ASSERT(regionPrivacy);
        std::unique_lock lock{guard_};
        regionPrivacy_ = regionPrivacy;
    }

    void setRoadGraph(IGraphPtr graphPtr)
    {
        ASSERT(graphPtr);
        std::unique_lock lock{guard_};
        roadGraph_ = graphPtr;
    }

    void setPedestrianGraph(IGraphPtr graphPtr)
    {
        ASSERT(graphPtr);
        std::unique_lock lock{guard_};
        pedestrianGraph_ = graphPtr;
    }

    void setRoadPhotoToEdge(IPhotoToEdgePtr ptr)
    {
        ASSERT(ptr);
        auto lock = std::unique_lock{guard_};
        roadPhotoToEdge_ = ptr;
    }

    void setPedestrianPhotoToEdge(IPhotoToEdgePtr ptr)
    {
        ASSERT(ptr);
        auto lock = std::unique_lock{guard_};
        pedestrianPhotoToEdge_ = ptr;
    }

    std::shared_ptr<renderer::image::ImageStorage> imageStorage() const
    {
        return designRegistry_.imageStorage();
    }

    void reloadDesign()
    {
        designRegistry_.reloadDesign();
    }

    renderer::design::Design design(
        const std::string& layer,
        const std::optional<std::string>& designExperimentId) const
    {
        return designRegistry_.design(layer, designExperimentId);
    }

    bool areCartographDesignsLoaded() const
    {
        return designRegistry_.areCartographDesignsLoaded();
    }

    static void logDatasetMetrics()
    {
        auto generatedAts = std::vector<chrono::TimePoint>{};
        auto getGeneratedAt = [&generatedAts](auto datasetPtr) {
            if (!datasetPtr) {
                return;
            }
            if (auto generatedAt = datasetPtr->getGeneratedAt()) {
                generatedAts.push_back(*generatedAt);
            }
        };

        auto that = instance();
        getGeneratedAt(that->dataAccess());
        getGeneratedAt(that->roadGraph());
        getGeneratedAt(that->pedestrianGraph());
        getGeneratedAt(that->roadPhotoToEdge());
        getGeneratedAt(that->pedestrianPhotoToEdge());
        auto [minIt, maxIt] =
            std::minmax_element(generatedAts.begin(), generatedAts.end());
        if (minIt != generatedAts.end() && maxIt != generatedAts.end()) {
            auto ageMax = std::chrono::duration_cast<std::chrono::minutes>(
                              chrono::TimePoint::clock::now() - *minIt)
                              .count();
            YCR_LOG_METRIC(DATASET_AGE_MAX_METRIC, ageMax);
            auto diff = std::chrono::duration_cast<std::chrono::minutes>(
                            *maxIt - *minIt)
                            .count();
            YCR_LOG_METRIC(DATASET_AGE_DIFF_METRIC, diff);
        }
    }

private:
    static ConfigurationPtr& rwInstance() {
        static ConfigurationPtr instance_;
        return instance_;
    }

    std::variant<std::shared_ptr<IDataAccess>, ReloadableDatasetHolder<FbDataAccess>>
    makeDataAccess(const std::optional<std::string> datasetPath)
    {
        if (datasetPath.has_value()) {
            return std::variant<std::shared_ptr<IDataAccess>, ReloadableDatasetHolder<FbDataAccess>>(
                std::in_place_type<ReloadableDatasetHolder<FbDataAccess>>,
                MRC_DATASET_NAME,
                [](const std::string& path) {
                    return std::make_shared<FbDataAccess>(path);
                },
                datasetPath.value()
            );
        }
        return std::make_shared<DbDataAccess>(mrcSharedPool());
    }

    bool isRegionPrivacyInTargetState() const
    {
        std::shared_lock lock{guard_};
        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<privacy::RegionPrivacy>>(&regionPrivacy_)) {
            return datasetPtr->isInTargetState();
        } else if (auto privacyPtr = std::get_if<std::shared_ptr<privacy::RegionPrivacy>>(&regionPrivacy_)) {
            return true;
        }
        REQUIRE(false, "Could not get regionPrivacy");
    }

    bool isDataAccessInTargetState() const
    {
        std::shared_lock lock{guard_};
        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<FbDataAccess>>(&dataAccess_)) {
            return datasetPtr->isInTargetState();
        } else if (auto datasetPtr = std::get_if<std::shared_ptr<IDataAccess>>(&dataAccess_)) {
            return true;
        }
        REQUIRE(false, "Could not get dataAccess");
    }

    bool isRoadGraphInTargetState() const
    {
        std::shared_lock lock{guard_};
        if (auto graphPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&roadGraph_)) {
            return graphPtr->isInTargetState();
        } else if (auto graphPtr = std::get_if<IGraphPtr>(&roadGraph_)) {
            return true;
        }
        REQUIRE(false, "Could not get roadGraph");
    }

    bool isPedestrianGraphInTargetState() const
    {
        std::shared_lock lock{guard_};
        if (auto graphPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&pedestrianGraph_)) {
            return graphPtr->isInTargetState();
        } else if (auto graphPtr = std::get_if<IGraphPtr>(&pedestrianGraph_)) {
            return true;
        }
        REQUIRE(false, "Could not get pedestrianGraph");
    }

    bool isRoadPhotoToEdgeInTargetState() const
    {
        auto lock = std::shared_lock{guard_};
        if (auto ptr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&roadPhotoToEdge_)) {
            return ptr->isInTargetState();
        } else if (auto ptr = std::get_if<IPhotoToEdgePtr>(&roadPhotoToEdge_)) {
            return true;
        }
        REQUIRE(false, "Could not get roadPhotoToEdge");
    }

    bool isPedestrianPhotoToEdgeInTargetState() const
    {
        auto lock = std::shared_lock{guard_};
        if (auto ptr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&pedestrianPhotoToEdge_)) {
            return ptr->isInTargetState();
        } else if (auto ptr = std::get_if<IPhotoToEdgePtr>(&pedestrianPhotoToEdge_)) {
            return true;
        }
        REQUIRE(false, "Could not get pedestrianPhotoToEdge");
    }

    bool isRegionPrivacyLoaded() const
    {
        std::shared_lock lock{guard_};
        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<privacy::RegionPrivacy>>(&regionPrivacy_)) {
            return datasetPtr->isLoaded();
        } else if (auto privacyPtr = std::get_if<std::shared_ptr<privacy::RegionPrivacy>>(&regionPrivacy_)) {
            return true;
        }
        REQUIRE(false, "Could not get regionPrivacy");
    }

    bool isDataAccessLoaded() const
    {
        std::shared_lock lock{guard_};
        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<FbDataAccess>>(&dataAccess_)) {
            return datasetPtr->isLoaded();
        } else if (auto datasetPtr = std::get_if<std::shared_ptr<IDataAccess>>(&dataAccess_)) {
            return true;
        }
        REQUIRE(false, "Could not get dataAccess");
    }

    bool isRoadGraphLoaded() const
    {
        std::shared_lock lock{guard_};
        if (auto graphPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&roadGraph_)) {
            return graphPtr->isLoaded();
        } else if (auto graphPtr = std::get_if<IGraphPtr>(&roadGraph_)) {
            return true;
        }
        REQUIRE(false, "Could not get roadGraph");
    }

    bool isPedestrianGraphLoaded() const
    {
        std::shared_lock lock{guard_};
        if (auto graphPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&pedestrianGraph_)) {
            return graphPtr->isLoaded();
        } else if (auto graphPtr = std::get_if<IGraphPtr>(&pedestrianGraph_)) {
            return true;
        }
        REQUIRE(false, "Could not get pedestrianGraph");
    }

    bool isRoadPhotoToEdgeLoaded() const
    {
        auto lock = std::shared_lock{guard_};
        if (auto ptr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&roadPhotoToEdge_)) {
            return ptr->isLoaded();
        } else if (auto ptr = std::get_if<IPhotoToEdgePtr>(&roadPhotoToEdge_)) {
            return true;
        }
        REQUIRE(false, "Could not get roadPhotoToEdge");
    }

    bool isPedestrianPhotoToEdgeLoaded() const
    {
        auto lock = std::shared_lock{guard_};
        if (auto ptr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&pedestrianPhotoToEdge_)) {
            return ptr->isLoaded();
        } else if (auto ptr = std::get_if<IPhotoToEdgePtr>(&pedestrianPhotoToEdge_)) {
            return true;
        }
        REQUIRE(false, "Could not get pedestrianPhotoToEdge");
    }

    void switchToTargetStateWorkCycle()
    {
        std::shared_lock lock{guard_};

        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&roadGraph_)) {
            datasetPtr->switchToTargetState();
        }

        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<Graph>>(&pedestrianGraph_)) {
            datasetPtr->switchToTargetState();
        }

        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&roadPhotoToEdge_)) {
            datasetPtr->switchToTargetState();
        }

        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<PhotoToEdge>>(&pedestrianPhotoToEdge_)) {
            datasetPtr->switchToTargetState();
        }

        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<privacy::RegionPrivacy>>(&regionPrivacy_)) {
            datasetPtr->switchToTargetState();
        }

        if (auto datasetPtr = std::get_if<ReloadableDatasetHolder<FbDataAccess>>(&dataAccess_)) {
            datasetPtr->switchToTargetState();
        }
    }

    IBlackboxClientHolder blackboxClient_;
    common::SharedPool mrcSharedPool_;
    maps::wiki::common::PoolHolder wikiPoolHolder_;
    maps::mds::Mds mds_;
    maps::mds::Mds publicMds_;
    IPermissionCheckerHolder permissionChecker_;
    http::URL hiddenFeatureRedirectUrl_;
    http::URL s3MdsUrl_;
    http::URL stvdescrUrl_;

    mutable std::shared_mutex guard_;
    std::variant<ReloadableDatasetHolder<Graph>, IGraphPtr> roadGraph_;
    std::variant<ReloadableDatasetHolder<Graph>, IGraphPtr> pedestrianGraph_;
    std::variant<ReloadableDatasetHolder<PhotoToEdge>, IPhotoToEdgePtr> roadPhotoToEdge_;
    std::variant<ReloadableDatasetHolder<PhotoToEdge>, IPhotoToEdgePtr> pedestrianPhotoToEdge_;
    std::variant<ReloadableDatasetHolder<privacy::RegionPrivacy>, privacy::RegionPrivacyPtr> regionPrivacy_;
    std::variant<std::shared_ptr<IDataAccess>, ReloadableDatasetHolder<FbDataAccess>> dataAccess_;

    std::shared_ptr<ugc_event_logger::Logger> ugcEventLoggerPtr_;

    DesignRegistry designRegistry_;

    Semaphore semaphore_;
    std::thread switchingToTargetStateWorker_;
};

} // namespace maps::mrc::browser
