#pragma once

#include <maps/libs/common/include/exception.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/libs/road_graph/include/types.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/sequenced_lifetime_guard.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/object_in_photo.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/walk_object.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/features_reader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/walk_objects_reader.h>

#include <memory>
#include <unordered_map>

namespace maps::mrc::browser {

const uint64_t YANDEX_UID = 0;

class IDataAccess {
public:
    using IdToPointMap = std::unordered_map<db::TId, geolib3::Point2>;
    using IdToUidMap = std::unordered_map<db::TId, uint64_t>;

    virtual ~IDataAccess() = default;

    virtual db::Feature getFeatureById(db::TId featureId) = 0;

    virtual db::Features getFeaturesByIds(const db::TIds& featureIds) = 0;

    virtual std::optional<db::Feature> tryGetFeatureById(db::TId featureId) = 0;

    virtual db::Features getFeaturesByBbox(
        const geolib3::BoundingBox& geoBbox) = 0;

    virtual db::Features getWalkFeaturesByWalkObjectsBbox(
        const geolib3::BoundingBox& geoBbox) = 0;

    virtual db::Features getNearestFeatures(
        const geolib3::Point2& geoPos,
        size_t limit) = 0;

    virtual bool existFeaturesInBbox(const geolib3::BoundingBox& geoBbox) = 0;

    virtual db::Features getFeaturesSequence(
        db::TId baseFeatureId,
        size_t beforeLimit,
        size_t afterLimit) = 0;

    virtual db::ObjectsInPhoto getObjectsInPhotoByFeatureId(
        db::TId featureId) = 0;

    virtual IdToPointMap getWalkObjectsByFeatures(
        const db::Features& features) = 0;

    virtual uint64_t getUid(const db::Feature& feature) = 0;

    virtual IdToUidMap getUids(const db::Features& features) = 0;

    virtual std::string getVersion() = 0;

    virtual bool isUnpublished(db::TId) const = 0;

    virtual void removeUnpublishedFeatures(
        std::vector<fb::CoveredSubPolyline>&) const {}

    virtual std::optional<chrono::TimePoint> getGeneratedAt() const = 0;
};

using IDataAccessPtr = std::shared_ptr<IDataAccess>;

class DbDataAccess : public IDataAccess {
public:
    explicit DbDataAccess(common::SharedPool);

    db::Feature getFeatureById(db::TId featureId) override;

    db::Features getFeaturesByIds(const db::TIds& featureIds) override;

    std::optional<db::Feature> tryGetFeatureById(db::TId featureId) override;

    db::Features getFeaturesByBbox(const geolib3::BoundingBox& geoBbox) override;

    db::Features getWalkFeaturesByWalkObjectsBbox(
        const geolib3::BoundingBox& geoBbox) override;

    db::Features getNearestFeatures(
        const geolib3::Point2& geoPos,
        size_t limit) override;

    bool existFeaturesInBbox(const geolib3::BoundingBox& geoBbox) override;

    db::Features getFeaturesSequence(
        db::TId baseFeatureId,
        size_t beforeLimit,
        size_t afterLimit) override;

    db::ObjectsInPhoto getObjectsInPhotoByFeatureId(db::TId featureId) override;

    IdToPointMap getWalkObjectsByFeatures(const db::Features& features) override;

    uint64_t getUid(const db::Feature& feature) override;

    IdToUidMap getUids(const db::Features& features) override;

    std::string getVersion() override;

    bool isUnpublished(db::TId featureId) const override;

    std::optional<chrono::TimePoint> getGeneratedAt() const override { return {}; }

private:
    common::SharedPool pgPool_;
};

class IdSet;
class Semaphore;

class FbObjectNotFoundException : public RuntimeError {
public:
    using RuntimeError::RuntimeError;
};

class FbDataAccess : common::SequencedLifetimeGuard<FbDataAccess>,
                     public IDataAccess {
public:
    explicit FbDataAccess(const std::string& dataDir);

    db::Feature getFeatureById(db::TId featureId) override;

    db::Features getFeaturesByIds(const db::TIds& featureIds) override;

    std::optional<db::Feature> tryGetFeatureById(db::TId featureId) override;

    db::Features getFeaturesByBbox(
        const geolib3::BoundingBox& geoBbox) override;

    db::Features getWalkFeaturesByWalkObjectsBbox(
        const geolib3::BoundingBox& geoBbox) override;

    db::Features getNearestFeatures(const geolib3::Point2& geoPos,
                                    size_t limit) override;

    bool existFeaturesInBbox(const geolib3::BoundingBox& geoBbox) override;

    db::Features getFeaturesSequence(db::TId baseFeatureId,
                                     size_t beforeLimit,
                                     size_t afterLimit) override;

    db::ObjectsInPhoto getObjectsInPhotoByFeatureId(db::TId featureId) override;

    IdToPointMap getWalkObjectsByFeatures(
        const db::Features& features) override;

    uint64_t getUid(const db::Feature& feature) override;

    IdToUidMap getUids(const db::Features& features) override;

    std::string getVersion() override;

    bool isUnpublished(db::TId featureId) const override;

    void removeUnpublishedFeatures(
        std::vector<fb::CoveredSubPolyline>&) const override;

    std::optional<chrono::TimePoint> getGeneratedAt() const override { return generatedAt_; }

private:
    fb::FeaturesReader features_;
    fb::WalkObjectsReader walkObjects_;
    std::optional<chrono::TimePoint> generatedAt_;
};

} // namespace maps::mrc::browser
