#include "data_access.h"
#include "tools.h"

#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/version.h>

#include <boost/lexical_cast.hpp>

namespace maps::mrc::browser {

namespace {

auto intersectFn(IDataAccess* dataAccess)
{
    return [dataAccess](const geolib3::BoundingBox& box, fb_rtree::Id id) {
        auto feature = dataAccess->tryGetFeatureById(id);
        return feature && geolib3::spatialRelation(
                              box, feature->geodeticPos(), geolib3::Intersects);
    };
}

auto intersectFn(fb::WalkObjectsReader* walkObjects)
{
    return [walkObjects](const geolib3::BoundingBox& box, fb_rtree::Id id) {
        auto pos = walkObjects->posByWalkObjectId(id);
        return pos.has_value() &&
               geolib3::spatialRelation(box, pos.value(), geolib3::Intersects);
    };
}

template <class Iterator>
void copyFeatures(Iterator first,
                  Iterator last,
                  size_t limit,
                  IDataAccess* dataAccess,
                  db::Features& result)
{
    for (; first != last && limit; ++first) {
        if (auto feature = dataAccess->tryGetFeatureById(first->featureId)) {
            result.push_back(feature.value());
            --limit;
        }
    }
}

template <class Iterator>
void reverseCopyFeatures(Iterator first,
                         Iterator last,
                         size_t limit,
                         IDataAccess* dataAccess,
                         db::Features& result)
{
    for (; first != last && limit; --last) {
        auto prev = std::prev(last);
        if (auto feature = dataAccess->tryGetFeatureById(prev->featureId)) {
            result.push_back(feature.value());
            --limit;
        }
    }
}

}  // namespace

FbDataAccess::FbDataAccess(const std::string& dataDir)
    : features_(dataDir, EMappingMode::Locked)
    , walkObjects_(dataDir, EMappingMode::Locked)
{
    REQUIRE(features_.version() == walkObjects_.version(),
            "inconsistent versions of features ("
                << features_.version() << "), walk objects ("
                << walkObjects_.version() << ")");
    if (!features_.version().empty()) {
        generatedAt_ = fb::parseVersion(std::string{features_.version()});
    }
}

db::Feature FbDataAccess::getFeatureById(db::TId featureId)
{
    if (auto result = tryGetFeatureById(featureId)) {
        return result.value();
    }
    throw FbObjectNotFoundException() << "missing feature " << featureId;
}

db::Features FbDataAccess::getFeaturesByIds(const db::TIds& featureIds)
{
    auto result = db::Features{};
    result.reserve(featureIds.size());
    for (auto featureId : featureIds) {
        if (auto feature = tryGetFeatureById(featureId)) {
            result.push_back(feature.value());
        }
    }
    return result;
}

std::optional<db::Feature> FbDataAccess::tryGetFeatureById(db::TId featureId)
{
    if (isUnpublished(featureId)) {
        return std::nullopt;
    }
    return features_.featureById(featureId);
}

db::Features FbDataAccess::getFeaturesByBbox(
    const geolib3::BoundingBox& geoBbox)
{
    auto result = db::Features{};
    auto rng = features_.rtree().allIdsInWindow(geoBbox, intersectFn(this));
    for (db::TId featureId : rng) {
        if (auto feature = tryGetFeatureById(featureId)) {
            result.push_back(feature.value());
        }
    }
    return result;
}

db::Features FbDataAccess::getWalkFeaturesByWalkObjectsBbox(
    const geolib3::BoundingBox& geoBbox)
{
    auto result = db::Features{};
    auto walkObjectIds = walkObjects_.rtree().allIdsInWindow(
        geoBbox, intersectFn(&walkObjects_));
    for (db::TId walkObjectId : walkObjectIds) {
        auto featureIds =
            features_.lookupFeatureIdsByWalkObjectId(walkObjectId);
        for (db::TId featureId : featureIds) {
            if (auto feature = tryGetFeatureById(featureId)) {
                result.push_back(feature.value());
            }
        }
    }
    return result;
}

db::Features FbDataAccess::getNearestFeatures(const geolib3::Point2& geoPos,
                                              size_t limit)
{
    auto distance = fb_rtree::SquaredDistance{};
    auto distanceById = [&](const geolib3::Point2& point, fb_rtree::Id id) {
        auto feature = tryGetFeatureById(id);
        return feature ? distance(point, feature->geodeticPos())
                       : std::numeric_limits<double>::max();
    };
    auto rng = fb_rtree::nearestIds(
        features_.rtree(), geoPos, distance, distanceById, limit);
    auto result = db::Features{};
    for (db::TId featureId : rng) {
        if (auto feature = tryGetFeatureById(featureId)) {
            result.push_back(feature.value());
        }
    }
    return result;
}

bool FbDataAccess::existFeaturesInBbox(const geolib3::BoundingBox& geoBbox)
{
    auto rng = features_.rtree().allIdsInWindow(geoBbox, intersectFn(this));
    for (db::TId featureId : rng) {
        if (tryGetFeatureById(featureId)) {
            return true;
        }
    }
    return false;
}

db::Features FbDataAccess::getFeaturesSequence(db::TId baseFeatureId,
                                               size_t beforeLimit,
                                               size_t afterLimit)
{
    auto baseFeature = getFeatureById(baseFeatureId);
    if (baseFeature.sourceId() == db::feature::NO_SOURCE_ID) {
        return {baseFeature};
    }
    auto rng = features_.photoTimes(baseFeature.sourceId());
    auto photoTime = fb::TPhotoTime{.time = baseFeature.timestamp(),
                                    .featureId = baseFeature.id()};
    auto [lowerBound, upperBound] =
        std::equal_range(rng.begin(), rng.end(), photoTime);
    REQUIRE(std::find(lowerBound, upperBound, photoTime) != upperBound,
            "missing photo time " << baseFeatureId);

    auto result = db::Features{};
    reverseCopyFeatures(rng.begin(), lowerBound, beforeLimit, this, result);
    copyFeatures(lowerBound, upperBound, upperBound - lowerBound, this, result);
    copyFeatures(upperBound, rng.end(), afterLimit, this, result);
    std::sort(result.begin(),
              result.end(),
              [](const db::Feature& lhs, const db::Feature& rhs) {
                  return std::make_tuple(lhs.timestamp(), lhs.id()) <
                         std::make_tuple(rhs.timestamp(), rhs.id());
              });
    return result;
}

db::ObjectsInPhoto FbDataAccess::getObjectsInPhotoByFeatureId(db::TId featureId)
{
    return features_.objectsInPhotoByFeatureId(featureId);
}

FbDataAccess::IdToPointMap FbDataAccess::getWalkObjectsByFeatures(
    const db::Features& features)
{
    auto result = IdToPointMap{};
    for (const auto& feature : features) {
        if (auto objId = feature.walkObjectId()) {
            if (auto pos = walkObjects_.posByWalkObjectId(objId.value())) {
                result.emplace(feature.id(), pos.value());
            }
        }
    }
    return result;
}

void FbDataAccess::removeUnpublishedFeatures(
    std::vector<fb::CoveredSubPolyline>& coveredSubpolylines) const
{
    coveredSubpolylines.erase(
        std::remove_if(
            coveredSubpolylines.begin(),
            coveredSubpolylines.end(),
            [this](const fb::CoveredSubPolyline& coveredSubpolyline) {
                return isUnpublished(coveredSubpolyline.featureId());
            }),
        coveredSubpolylines.end());
}

bool FbDataAccess::isUnpublished(db::TId /*featureId*/) const
{
    return false;
}

uint64_t FbDataAccess::getUid(const db::Feature& feature)
{
    auto userId = feature.userId();
    return userId ? boost::lexical_cast<uint64_t>(userId.value()) : YANDEX_UID;
}

DbDataAccess::IdToUidMap FbDataAccess::getUids(const db::Features& features)
{
    auto result = IdToUidMap{};
    result.reserve(features.size());
    for (const auto& feature : features) {
        result.emplace(feature.id(), getUid(feature));
    }
    return result;
}

std::string FbDataAccess::getVersion()
{
    return std::string{features_.version()};
}

}  // namespace maps::mrc::browser
