#pragma once

#include <maps/libs/chrono/include/time_point.h>

#include <maps/libs/log8/include/log8.h>

#include <chrono>
#include <shared_mutex>
#include <string>
#include <thread>


namespace maps::mrc::browser {

using SteadyTimePoint = std::chrono::time_point<std::chrono::steady_clock, std::chrono::nanoseconds>;

template<typename T>
class ReloadableDatasetHolder {
public:
    using DatasetLoader = std::function<std::shared_ptr<T>(const std::string&)>;

    /// @param name -- dataset name, used for logging
    ReloadableDatasetHolder(std::string name, DatasetLoader loader, std::optional<std::string> targetDatasetPath)
        : name_(std::move(name))
        , loader_(std::move(loader))
        , targetDatasetPath_(std::move(targetDatasetPath))
    {}

    ReloadableDatasetHolder(const ReloadableDatasetHolder&) = delete;
    ReloadableDatasetHolder& operator=(const ReloadableDatasetHolder&) = delete;

    ReloadableDatasetHolder(ReloadableDatasetHolder&& other) = default;
    ReloadableDatasetHolder& operator=(ReloadableDatasetHolder&&) = default;

    std::optional<std::string> currentDatasetPath() const
    {
        std::shared_lock lock{guard_};
        return currentDatasetPath_;
    }

    std::optional<std::string> targetDatasetPath() const
    {
        std::shared_lock lock{guard_};
        return targetDatasetPath_;
    }

    bool isInTargetState() const
    {
        std::shared_lock lock{guard_};
        return currentDatasetPath_ == targetDatasetPath_;
    }

    bool isLoaded() const
    {
        std::shared_lock lock{guard_};
        return currentDatasetPath_.has_value();
    }

    std::shared_ptr<T> dataset() const
    {
        std::shared_lock lock{guard_};
        std::shared_ptr<T> result = sharedDataset_;
        REQUIRE(result, "Dataset " << name_ << " is not loaded");
        return result;
    }

    void setTargetDatasetPath(std::optional<std::string> path, std::optional<SteadyTimePoint> timeToStartSwitchToTargetState)
    {
        std::unique_lock lock{guard_};
        timeToStartSwitchToTargetState_ = timeToStartSwitchToTargetState;
        targetDatasetPath_ = std::move(path);
    }

    /// This method should not be called from different threads simultaneously
    void switchToTargetState()
    {
        std::shared_lock readLock{guard_};
        if (currentDatasetPath_ == targetDatasetPath_) {
            return;
        }
        std::optional<std::string> targetDatasetPath = targetDatasetPath_;
        auto timeToStartSwitchToTargetState = timeToStartSwitchToTargetState_;
        readLock.unlock();

        std::string datasetStr = targetDatasetPath.has_value() ? ("'" + targetDatasetPath.value() + "'"): "<none>";

        INFO() << "Switching dataset " << name_ << " to " << datasetStr;


        if (timeToStartSwitchToTargetState.has_value()) {
            const auto now = std::chrono::steady_clock::now();
            if (now < timeToStartSwitchToTargetState.value()) {
                const auto sleepDuration = timeToStartSwitchToTargetState.value() - now;
                INFO() << "Waiting for " << std::chrono::duration_cast<std::chrono::seconds>(sleepDuration).count() << " sec";
                std::this_thread::sleep_for(sleepDuration);
                INFO() << "Done waiting";
            }
        }

        std::shared_ptr<T> dataset;
        {
            std::unique_lock writeLock{guard_};
            if (sharedDataset_) {
                dataset.swap(sharedDataset_);
                currentDatasetPath_.reset();
            }
        }
        dataset.reset();
        if (targetDatasetPath.has_value()) {
            INFO() << "Dataset " << name_ << " loading from '" << targetDatasetPath.value() << "'";
            dataset = loader_(targetDatasetPath.value());
        }
        std::unique_lock writeLock{guard_};
        sharedDataset_ = std::move(dataset);
        currentDatasetPath_ = targetDatasetPath;

        INFO() << "Switched dataset " << name_ << " to " << datasetStr;
    }

private:
    const std::string name_;
    DatasetLoader loader_;
    mutable std::shared_mutex guard_;
    std::shared_ptr<T> sharedDataset_;
    /// If sharedDataset_ is defined currentDatasetPath_ points to it's path
    std::optional<std::string> currentDatasetPath_;
    std::optional<std::string> targetDatasetPath_;
    std::optional<SteadyTimePoint> timeToStartSwitchToTargetState_;
};


} // namespace maps::mrc::browser
