#include "configuration.h"
#include "undistort.h"
#include "tools.h"
#include "tile_renderer.h"
#include "yacare_params.h"

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/opencv.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/yacare_helpers.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/visibility.h>
#include <maps/wikimap/mapspro/services/mrc/libs/proto/include/common.h>

#include <maps/libs/concurrent/include/lru_cache.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <yandex/maps/proto/mobile_config/mapkit2/layers.pb.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/http/include/etag.h>
#include <maps/libs/tile/include/geometry.h>
#include <maps/infra/yacare/include/yacare.h>
#include <maps/infra/yacare/include/params/tile.h>
#include <maps/infra/yacare/include/params/scale.h>
#include <maps/infra/yacare/include/params/tvm.h>
#include <maps/infra/yacare/include/limit_rate.h>

#include <boost/algorithm/cxx11/any_of.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <boost/range/algorithm_ext/erase.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/filesystem.hpp>

#include <optional>

namespace maps::mrc::browser {

namespace {

using FixedVersion =
    yandex::maps::proto::mobile_config::mapkit2::layers::FixedVersion;

const std::string ETAG_HEADER = "ETag";
const std::string HOST_KEY = "HTTP_HOST";
const common::Size FULLHD_LANDSCAPE{1920, 1080};
const common::Size FULLHD_PORTRAIT {1080, 1920};
static constexpr int CACHE_ZOOM_UPPER_BOUND = 11;
static constexpr double THRESHOLD_FOR_ONE_PATH = 70.;

yacare::ThreadPool& mdsThreadPool()
{
    static yacare::ThreadPool pool(
        "mds_thread_pool",
        yacare::ThreadPool::CPUCOUNT * 5,
        4096
    );
    return pool;
}

cv::Mat loadFeatureImage(const db::Feature& feature)
{
    auto data = Configuration::instance()->mds().get(feature.mdsKey());
    return common::decodeImage(data);
}

FixedVersion makeFixedVersion(const std::string& version)
{
    FixedVersion versionProto;
    versionProto.set_value(TString(version));
    return versionProto;
}

db::FeaturePrivacy evalMinFeaturePrivacy(const geolib3::BoundingBox& geoBox)
{
    const auto cfg = Configuration::instance();
    return cfg->regionPrivacy()->evalMinFeaturePrivacy(geoBox);
}

void blurSignalqOverlay(cv::Mat& image)
{
    static const struct { double minX, minY, maxX, maxY; } relativeBoxes[] {
        {0.0,   0.916, 0.146, 1.0}, // Bottom-left area with timestamp and coordinates
        {0.896, 0.926, 1.0,   1.0}  // Bottom-right area with SignalQ logo
    };

    for (const auto& relativeBox : relativeBoxes) {
        mrc::common::ellipticalBlur(image, mrc::common::ImageBox{
            static_cast<size_t>(relativeBox.minX * image.cols),
            static_cast<size_t>(relativeBox.minY * image.rows),
            static_cast<size_t>(relativeBox.maxX * image.cols),
            static_cast<size_t>(relativeBox.maxY * image.rows)});
    }
}

void blurPrivateDataInImage(const db::Feature& feature, cv::Mat& image)
{
    const auto objects = Configuration::instance()->dataAccess()
        ->getObjectsInPhotoByFeatureId(feature.id());

    for (const auto& object : objects) {
        if (object.type() == db::ObjectInPhotoType::LicensePlate
            || object.type() == db::ObjectInPhotoType::Face) {
            mrc::common::ellipticalBlur(image, object.imageBox());
        }
    }

    if (feature.dataset() == db::Dataset::TaxiSignalQ2) {
        blurSignalqOverlay(image);
    }
}

template<typename TFunctor>
void processFeatureImageRequest(
    std::optional<blackbox_client::Uid> maybeUserId,
    const yacare::Request& request,
    yacare::Response& response,
    const std::string& featureIdStr,
    const std::string& /*suffix*/,
    TFunctor modifier,
    std::optional<common::ImageOrientation> orientation = std::nullopt,
    int quality = 0)
{
    maps::http::ETag etag(featureIdStr);
    if (!request.preconditionSatisfied(etag)) {
        response.setStatus(yacare::HTTPStatus::NotModified);
        return;
    }
    response.setHeader(ETAG_HEADER, boost::lexical_cast<std::string>(etag));

    const auto featureId = boost::lexical_cast<db::TId>(featureIdStr);
    auto configuration = Configuration::instance();
    db::Feature feature = configuration->dataAccess()->getFeatureById(featureId);

    if (!userMayViewPhotos(maybeUserId, feature.privacy())) {
        throw yacare::errors::Forbidden();
    }

    auto image = loadFeatureImage(feature);

    // Blur privacy in the images before rotation to normal orientation and
    // resizing to avoid bboxes translation to the transformed image coordinates.
    blurPrivateDataInImage(feature, image);

    if (orientation.has_value()) {
        image = transformByImageOrientation(image, orientation.value());
    } else {
        image = transformByImageOrientation(image, feature.orientation());
    }

    if (db::isStandalonePhotosDataset(feature.dataset())) {
        image = (image.rows < image.cols)
            ? common::fitImageTo(image, FULLHD_LANDSCAPE)
            : common::fitImageTo(image, FULLHD_PORTRAIT);
    }

    mrc::common::equalizeHistogramBGR(image);

    logUgcPhotoViewEvent(maybeUserId, {feature.id()}, request);

    auto encodedImage = mrc::common::encodeImage(modifier(image), quality);
    response.setHeader(CONTENT_TYPE_HEADER, CONTENT_TYPE_JPEG);
    response.write(reinterpret_cast<const char*>(encodedImage.data()),
                   encodedImage.size());
}

db::Features
loadNearestFeatures(Configuration& configuration,
                    const geolib3::Point2& geoPos,
                    size_t limit,
                    std::optional<db::GraphType> graphType)
{
    if (graphType.has_value() && graphType.value() == db::GraphType::Pedestrian) {
        auto graph = configuration.graphByType(graphType.value());
        auto edgeIds =
            graph->getNearestCoveredEdges(geoPos, limit, 10 /*maxFc*/);
        auto featureIds = std::set<db::TId>{};
        for (auto edgeId : edgeIds) {
            boost::range::copy(graph->getFeatureIdsByEdgeId(edgeId),
                               std::inserter(featureIds, featureIds.end()));
        }
        return configuration.dataAccess()->getFeaturesByIds(
            {featureIds.begin(), featureIds.end()});
    }
    return configuration.dataAccess()->getNearestFeatures(geoPos, limit);
}

bool isNmapsLayer(const std::string& layer)
{
    static const auto LAYERS = {LAYER_AGE,
                                LAYER_AGE_PEDESTRIAN,
                                LAYER_AGE_ROAD,
                                LAYER_SIGNAL_SEQUENCES};
    return std::find(LAYERS.begin(), LAYERS.end(), layer) != LAYERS.end();
}

template<typename T>
std::optional<T> makeOptional(bool defined, T value)
{
    if (!defined) {
        return std::nullopt;
    }
    return std::make_optional(value);
}

bool isVectorFormat(const std::string& contentType)
{
    return contentType == CONTENT_TYPE_PROTOBUF ||
           contentType == CONTENT_TYPE_TEXT_PROTOBUF;
}

template<typename ResultFormatT>
ResultFormatT resultFormat(const std::string& contentType)
{
    if (contentType == CONTENT_TYPE_TEXT_PROTOBUF)
        return ResultFormatT::Text;
    return ResultFormatT::Proto;
}

bool lessEdgeId(const fb::PhotoToEdgePair& lhs, const fb::PhotoToEdgePair& rhs)
{
    return lhs.edgeId() < rhs.edgeId();
}

void removeIfNoSharedEdge(const IPhotoToEdge& photoToEdge,
                          const db::Feature& baseFeature,
                          db::Features& result)
{
    auto baseRelations = photoToEdge.lookupByFeatureId(baseFeature.id());
    if (baseRelations.empty()) {
        result.clear();
        return;
    }
    std::sort(baseRelations.begin(), baseRelations.end(), lessEdgeId);
    boost::range::remove_erase_if(result, [&](const db::Feature& feature) {
        auto relations = photoToEdge.lookupByFeatureId(feature.id());
        std::sort(relations.begin(), relations.end(), lessEdgeId);
        auto intersection = std::vector<fb::PhotoToEdgePair>{};
        set_intersection(baseRelations.begin(),
                         baseRelations.end(),
                         relations.begin(),
                         relations.end(),
                         std::back_inserter(intersection),
                         lessEdgeId);
        return intersection.empty();
    });
}

} // namespace

YCR_RESPOND_TO("GET /tiles", tile, l = {browser::LAYER_SIGNAL_SEQUENCES},
    maybeUserId, contentType, scale = 1.0, zmin = {}, zmax = {}, v = {}, vec_protocol,
    YCR_ARGUMENT_METRICS({"l", {LAYER_SIGNAL_SEQUENCES, LAYER_EDGES, LAYER_PEDESTRIAN_EDGES}}),
    experimental_mrc_design = std::nullopt, theme,
    experimental_equidistant_as_style = 0)
{
    Configuration::logDatasetMetrics();

    static concurrent::LruCache<TileRequestCacheKey,
                                std::optional<std::vector<char>>,
                                introspection::Hasher> lruCache{20000};

    static const std::vector<std::string> SUPPORTED_CONTENT_TYPES = {
        CONTENT_TYPE_PNG,
        CONTENT_TYPE_PROTOBUF,
        CONTENT_TYPE_TEXT_PROTOBUF
    };

    if (l.empty()) {
        throw yacare::errors::BadRequest("wrong l param");
    }

    std::string responseContentType =
        yacare::bestContentType(contentType, SUPPORTED_CONTENT_TYPES);

    const std::string layer = l.at(0);
    auto configuration = Configuration::instance();
    auto filter = FeatureFilter::fromRequest(request);
    if (isNmapsLayer(layer)) {
        // only in this layer non-public photos should be shown
        filter.uid = maybeUserId;
    } else {
        filter.cameraDeviation = db::CameraDeviation::Front;
    }

    auto tileRequest =
        TileRequestCacheKey(layer, tile.x(), tile.y(), tile.z(), filter)
            .setScale(scale)
            .setContentType(responseContentType)
            .setMultiZoom(makeOptional(has(zmin), zmin),
                          makeOptional(has(zmax), zmax),
                          tile.z())
            .setVecProtocolVersion(vec_protocol);
        // WARN: MAPSMRC-3123 dataset version is ignored for now

    if (theme && responseContentType == CONTENT_TYPE_PNG)
        tileRequest.setTheme(*theme);

    const auto design = configuration->design(layer, experimental_mrc_design);
    if (design.revision)
        tileRequest.setDesignVersion(*design.revision);

    // TODO: After VECTOR-778 use Equidistants::AsStyle unconditionally
    if (experimental_equidistant_as_style == 1)
        tileRequest.equidistantFormat = renderer::vecdata::Equidistants::AsStyle;

    auto renderTileFunc =
        [&](const TileRequestCacheKey& tileRequest) {
            if (isVectorFormat(responseContentType)) {
                if (vec_protocol == 2u) {
                    return browser::renderVectorTile(
                        tileRequest,
                        configuration->dataAccess(),
                        configuration->roadGraph(),
                        configuration->pedestrianGraph(),
                        resultFormat<renderer::vecdata::ResultFormat>(responseContentType),
                        configuration->imageStorage(),
                        design
                    );
                } else {
                    return browser::renderVectorTile(
                        tileRequest,
                        configuration->dataAccess(),
                        configuration->roadGraph(),
                        configuration->pedestrianGraph(),
                        resultFormat<renderer::vecdata3::ResultFormat>(responseContentType),
                        configuration->imageStorage(),
                        design
                    );
                }
            } else if (responseContentType == CONTENT_TYPE_PNG) {
                return browser::renderRasterTile(
                    tileRequest,
                    configuration->dataAccess(),
                    configuration->roadGraph(),
                    configuration->pedestrianGraph(),
                    configuration->imageStorage(),
                    design
                );
            }
            throw yacare::errors::BadRequest() << "Unsupported Content-Type";
        };

    std::optional<std::vector<char>> optImage;

    if (!isNmapsLayer(layer) && tile.z() < CACHE_ZOOM_UPPER_BOUND) {
        optImage = *lruCache.getOrEmplace(tileRequest, renderTileFunc);
    } else {
        optImage = renderTileFunc(tileRequest);
    }

    common::handleYandexOrigin(request, response);

    if (!optImage.has_value()) {
        response.setStatus(yacare::HTTPStatus::NoContent);
        return;
    }

    auto etag = maps::http::ETag::fromMD5Hash(optImage.value());
    if (!request.preconditionSatisfied(etag)) {
        response.setStatus(yacare::HTTPStatus::NotModified);
        return;
    }

    response.setHeader(ETAG_HEADER, etag.toString());
    response.setHeader(CONTENT_TYPE_HEADER, responseContentType);
    response.write(optImage.value().data(), optImage.value().size());
}

YCR_RESPOND_TO("OPTIONS /tiles",
               l = {browser::LAYER_SIGNAL_SEQUENCES},
               YCR_ARGUMENT_METRICS({"l",
                                     {LAYER_SIGNAL_SEQUENCES,
                                      LAYER_EDGES,
                                      LAYER_PEDESTRIAN_EDGES}}))
{
    if (common::handleYandexOrigin(request, response)) {
        response.setHeader(ALLOW_METHODS_HEADER, METHODS_GET_OPTIONS);
    }
}

YCR_RESPOND_TO("GET /features/hotspot", tile, maybeUserId, l = {browser::LAYER_SIGNAL_SEQUENCES}, callback = {},
        with_authors = false, experimental_mrc_design = std::nullopt)
{
    Configuration::logDatasetMetrics();

    static concurrent::LruCache<TileRequestCacheKey, std::string, introspection::Hasher> lruCache{20000};

    if (l.empty()) {
        throw yacare::errors::BadRequest("wrong l param");
    }

    const std::string layer = l.at(0);
    auto filter = FeatureFilter::fromRequest(request);
    if (isNmapsLayer(layer)) {
        // only in this layer non-public photos should be shown
        filter.uid = maybeUserId;
    } else {
        filter.cameraDeviation = db::CameraDeviation::Front;
    }

    auto configuration = Configuration::instance();

    TileRequestCacheKey tileRequest(layer, tile.x(), tile.y(), tile.z(), filter);
    tileRequest.setWithAuthors(with_authors);

    const auto design = configuration->design(layer, experimental_mrc_design);
    if (design.revision)
        tileRequest.setDesignVersion(*design.revision);

    auto renderTileFunc =
        [&](const TileRequestCacheKey tileRequest) {
            return renderHotspotTile(
                tileRequest,
                configuration->dataAccess(),
                configuration->roadGraph(),
                configuration->pedestrianGraph(),
                baseUrl(request),
                callback,
                configuration->imageStorage(),
                design
            );
        };

    std::string json;

    if (!isNmapsLayer(layer) && tile.z() < CACHE_ZOOM_UPPER_BOUND) {
        json = *lruCache.getOrEmplace(tileRequest, renderTileFunc);
    } else {
        json = renderTileFunc(tileRequest);
    }

    common::handleYandexOrigin(request, response);
    if (has(callback)) {
        response[CONTENT_TYPE_HEADER] = CONTENT_TYPE_JAVASCRIPT;
    } else {
        response[CONTENT_TYPE_HEADER] = CONTENT_TYPE_JSON;
    }
    response << json;
}

YCR_RESPOND_TO("OPTIONS /features/hotspot")
{
    if (common::handleYandexOrigin(request, response)) {
        response.setHeader(ALLOW_METHODS_HEADER, METHODS_GET_OPTIONS);
    }
}


/*
 * This handle is meant for inter-service communication
 * MAPSMRC-3927
 */
YCR_RESPOND_TO("/internal/feature/$/image",
    YCR_IN_POOL(mdsThreadPool()),
    YCR_USING(yacare::Tvm2ServiceRequire(SELF_TVM_SERVICE_ALIAS)),
    YCR_LIMIT_RATE(resource("maps_core_nmaps_mrc_browser_/internal_feature_image")))
{
    const auto featureId = boost::lexical_cast<db::TId>(argv[0]);
    auto configuration = Configuration::instance();
    db::Feature feature = configuration->dataAccess()->getFeatureById(featureId);

    auto image = loadFeatureImage(feature);

    // Blur privacy in the images before rotation to normal orientation and
    // resizing to avoid bboxes translation to the transformed image coordinates.
    blurPrivateDataInImage(feature, image);
    image = transformByImageOrientation(image, feature.orientation());

    auto encodedImage = mrc::common::encodeImage(image);
    response.setHeader(CONTENT_TYPE_HEADER, CONTENT_TYPE_JPEG);
    response.write(reinterpret_cast<const char*>(encodedImage.data()),
                   encodedImage.size());
}

YCR_RESPOND_TO("GET /feature/$/image",
               boxes = {},
               orientation = std::nullopt,
               maybeUserId,
               size_name = proto::SIZE_NAME_ORIGINAL,
               YCR_IN_POOL(mdsThreadPool()))
{
    Configuration::logDatasetMetrics();

    constexpr int QUALITY = 40;
    const auto draw = [&](cv::Mat& image) {
        if (has(boxes)) {
            browser::drawImageBoxes(image, boxes);
        }
        if (size_name == proto::SIZE_NAME_THUMBNAIL) {
            image = browser::toThumbnail(image);
        }
        return image;
    };

    processFeatureImageRequest(maybeUserId, request, response, argv[0],
                               "image", draw, orientation, QUALITY);
}

YCR_RESPOND_TO("GET /feature/$/thumbnail", maybeUserId, YCR_IN_POOL(mdsThreadPool()))
{
    Configuration::logDatasetMetrics();

    processFeatureImageRequest(maybeUserId, request, response, argv[0],
                               "thumbnail", browser::toThumbnail);
}

YCR_RESPOND_TO("GET /feature/$/undistorted", maybeUserId, YCR_IN_POOL(mdsThreadPool()))
{
    Configuration::logDatasetMetrics();

    processFeatureImageRequest(maybeUserId, request, response, argv[0],
                               "undistorted", browser::undistort);
}

YCR_RESPOND_TO("GET /features/$/path",
               with_authors = false,
               snap_to_road_graph = false,
               before_limit = 100,
               after_limit = 100,
               maybeUserId)
{
    Configuration::logDatasetMetrics();

    ResponseOptions responseOptions;
    responseOptions.withAuthors = with_authors;
    if (snap_to_road_graph) {
        responseOptions.snapToGraph =
            parseOptionalGraphType(request).value_or(db::GraphType::Road);
    }

    const auto baseFeatureId = boost::lexical_cast<db::TId>(argv[0]);

    db::Features features = Configuration::instance()->dataAccess()
        ->getFeaturesSequence(baseFeatureId, before_limit, after_limit);

    auto filter = FeatureFilter::fromRequest(request);
    filter.uid = maybeUserId;
    boost::range::remove_erase_if(features, filter);

    common::handleYandexOrigin(request, response);
    response << YCR_JSON(obj)
    {
        toPathJson(baseFeatureId,
                   std::move(features),
                   baseUrl(request),
                   responseOptions,
                   obj);
    };
}

YCR_RESPOND_TO("OPTIONS /features/$/path")
{
    if (common::handleYandexOrigin(request, response)) {
        response.setHeader(ALLOW_METHODS_HEADER, METHODS_GET_OPTIONS);
    }
}

YCR_RESPOND_TO("GET /features/$/similar",
               with_authors = false,
               snap_to_road_graph = false,
               maybeUserId)
{
    Configuration::logDatasetMetrics();

    namespace geo = maps::geolib3;
    auto configuration = Configuration::instance();
    auto dataAccess = configuration->dataAccess();
    auto baseFeatureId = boost::lexical_cast<db::TId>(argv[0]);
    auto maybeGraphType = parseOptionalGraphType(request);

    ResponseOptions responseOptions;
    responseOptions.withAuthors = with_authors;
    if (snap_to_road_graph) {
        responseOptions.snapToGraph = maybeGraphType.value_or(db::GraphType::Road);
    }


    auto optBaseFeature = dataAccess->tryGetFeatureById(baseFeatureId);
    if (!optBaseFeature) {
        throw yacare::errors::NotFound();
    }
    if (!userMayViewPhotos(maybeUserId, optBaseFeature->privacy())) {
        throw yacare::errors::Forbidden();
    }

    auto bbox = db::expandBbox(optBaseFeature->geodeticPos().boundingBox(),
                               SIMILARITY_RADIUS_METERS);

    auto features = dataAccess->getFeaturesByBbox(bbox);

    auto filter = FeatureFilter::fromRequest(request);
    filter.uid = maybeUserId;
    boost::range::remove_erase_if(features, filter);

    if (isStandalonePhotosDataset(optBaseFeature->dataset())) {
        boost::range::remove_erase_if(features, [](const db::Feature& feature) {
            return !isStandalonePhotosDataset(feature.dataset());
        });
    }
    else if (optBaseFeature->graph() == db::GraphType::Road) {
        removeIfNoSharedEdge(
            *configuration->roadPhotoToEdge(), *optBaseFeature, features);
    }
    else if (optBaseFeature->graph() == db::GraphType::Pedestrian) {
        removeIfNoSharedEdge(
            *configuration->pedestrianPhotoToEdge(), *optBaseFeature, features);
    }

    // input photo must be returned
    auto isBase = [baseFeatureId](const db::Feature& feature) {
        return baseFeatureId == feature.id();
    };
    if (std::none_of(features.begin(), features.end(), isBase)) {
        features.push_back(*optBaseFeature);
    }

    IdToPointMap targetPoints = dataAccess->getWalkObjectsByFeatures(features);

    common::handleYandexOrigin(request, response);
    response << YCR_JSON(obj)
    {
        toSimilarJson(baseFeatureId, std::move(features),
                      targetPoints, baseUrl(request), responseOptions, obj);
    };
}

YCR_RESPOND_TO("OPTIONS /features/$/similar")
{
    if (common::handleYandexOrigin(request, response)) {
        response.setHeader(ALLOW_METHODS_HEADER, METHODS_GET_OPTIONS);
    }
}

YCR_RESPOND_TO("GET /features/coverage", bbox, maybeUserId)
{
    Configuration::logDatasetMetrics();

    auto configuration = Configuration::instance();

    bool result = false;
    if (userMayViewPhotos(maybeUserId, evalMinFeaturePrivacy(bbox))) {
        result = configuration->dataAccess()->existFeaturesInBbox(bbox);
    }

    response << YCR_JSON(obj) { obj["coverage"] = result; };
}

YCR_RESPOND_TO("GET /features/stamp.xml")
{
    auto configuration = Configuration::instance();
    response[CONTENT_TYPE_HEADER] = CONTENT_TYPE_XML;
    response <<
R"(<?xml version="1.0" encoding="utf-8"?>
<ymaps xmlns="http://maps.yandex.ru/ymaps/1.x" xmlns:gml="http://www.opengis.net/gml" xmlns:lm="http://maps.yandex.ru/layers/1.x">
  <GeoObjectCollection>
    <gml:metaDataProperty>
      <lm:LayerMetaData>
        <lm:version>)" << configuration->dataAccess()->getVersion() << R"(</lm:version>
        <lm:Zoom>
          <lm:min>5</lm:min>
          <lm:max>22</lm:max>
        </lm:Zoom>
      </lm:LayerMetaData>
    </gml:metaDataProperty>
  </GeoObjectCollection>
</ymaps>)";
}


YCR_RESPOND_TO("GET /features/stampPedestrian.xml")
{
    auto configuration = Configuration::instance();
    response[CONTENT_TYPE_HEADER] = CONTENT_TYPE_XML;
    response <<
R"(<?xml version="1.0" encoding="utf-8"?>
<ymaps xmlns="http://maps.yandex.ru/ymaps/1.x" xmlns:gml="http://www.opengis.net/gml" xmlns:lm="http://maps.yandex.ru/layers/1.x">
  <GeoObjectCollection>
    <gml:metaDataProperty>
      <lm:LayerMetaData>
        <lm:version>)" << configuration->dataAccess()->getVersion() << R"(</lm:version>
        <lm:Zoom>
          <lm:min>5</lm:min>
          <lm:max>22</lm:max>
        </lm:Zoom>
      </lm:LayerMetaData>
    </gml:metaDataProperty>
  </GeoObjectCollection>
</ymaps>)";
}

YCR_RESPOND_TO("GET /features/near",
               ll,
               maybeUserId,
               with_authors = false,
               snap_to_road_graph = false,
               z = 23)
{
    Configuration::logDatasetMetrics();

    namespace geo = maps::geolib3;
    auto configuration = Configuration::instance();
    auto dataAccess = configuration->dataAccess();
    using LongEdgeId = maps::road_graph::LongEdgeId;
    std::optional<geo::Direction2> dir;
    if (auto longEdgeId = request.optParam<LongEdgeId::ValueType>(FIELD_EDGE)) {
        const auto graph = configuration->roadGraph();
        if (auto edgeId = graph->persistentIndex().findShortId(
                LongEdgeId{longEdgeId.value()})) {
            auto edge = geolib3::Polyline2(graph->graph().edgeData(*edgeId).geometry());
            auto segmentIdx = edge.closestPointSegmentIndex(ll);
            auto segment = edge.segmentAt(segmentIdx);
            dir = geo::Direction2{geo::convertGeodeticToMercator(segment)};
        }
        else {
            throw yacare::errors::BadRequest()
                << "invalid edge: " << longEdgeId.value();
        }
    }

    auto maybeGraphType = parseOptionalGraphType(request);

    ResponseOptions responseOptions;
    responseOptions.withAuthors = with_authors;
    if (snap_to_road_graph) {
        responseOptions.snapToGraph = maybeGraphType.value_or(db::GraphType::Road);
    }

    const auto DEFAULT_RADUIS_METERS = 40.;
    const auto DEFAULT_RADIUS_PIXELS = 5.;
    const auto LIMIT_PHOTOS = 100;
    auto radiusPixels = DEFAULT_RADIUS_PIXELS * tile::zoomToResolution(z);
    auto ratio = geolib3::MercatorRatio::fromGeodeticPoint(ll);
    auto radiusMeters =
        std::max(DEFAULT_RADUIS_METERS, ratio.toMeters(radiusPixels));
    auto features =
        loadNearestFeatures(*configuration, ll, LIMIT_PHOTOS, maybeGraphType);

    auto filter = FeatureFilter::fromRequest(request);
    filter.uid = maybeUserId;
    boost::range::remove_erase_if(features, filter);

    IdToPointMap targetPoints = dataAccess->getWalkObjectsByFeatures(features);

    common::handleYandexOrigin(request, response);
    response << YCR_JSON(obj)
    {
        toNearJson(ll,
                   dir,
                   radiusMeters,
                   std::move(features),
                   targetPoints,
                   baseUrl(request),
                   responseOptions,
                   obj);
    };
}

YCR_RESPOND_TO("OPTIONS /features/near")
{
    if (common::handleYandexOrigin(request, response)) {
        response.setHeader(ALLOW_METHODS_HEADER, METHODS_GET_OPTIONS);
    }
}

YCR_RESPOND_TO("GET /features/find", ids, maybeUserId, with_authors = false)
{
    Configuration::logDatasetMetrics();

    auto dataAccess = Configuration::instance()->dataAccess();
    auto features = dataAccess->getFeaturesByIds(ids);
    auto filter = FeatureFilter::fromRequest(request);
    filter.uid = maybeUserId;
    boost::range::remove_erase_if(features, filter);
    auto targetPoints = dataAccess->getWalkObjectsByFeatures(features);
    common::handleYandexOrigin(request, response);
    response << YCR_JSON(obj)
    {
        toFeaturesResponseJson(std::move(features),
                               targetPoints,
                               baseUrl(request),
                               ResponseOptions{.withAuthors = with_authors},
                               obj);
    };
}

YCR_RESPOND_TO("OPTIONS /features/find")
{
    if (common::handleYandexOrigin(request, response)) {
        response.setHeader(ALLOW_METHODS_HEADER, METHODS_GET_OPTIONS);
    }
}

YCR_RESPOND_TO("POST /features/tour_along_route",
               with_authors = false,
               graphType = db::GraphType::Road)
{
    Configuration::logDatasetMetrics();

    ResponseOptions responseOptions;
    responseOptions.withAuthors = with_authors;

    geolib3::Polyline2 geoLine;
    try {
        geoLine = toTrack(request.body());
    } catch (const std::exception&) {
        throw yacare::errors::BadRequest() << "Invalid track";
    }

    auto optCameraDeviation = parseOptionalCameraDeviation(request);
    if (!optCameraDeviation.has_value()) {
        optCameraDeviation = db::CameraDeviation::Front;
    }

    auto configuration = Configuration::instance();

    auto features = loadFeaturesTourAlongLine(
        configuration->dataAccess(),
        configuration->graphByType(graphType),
        geoLine,
        optCameraDeviation.value());

    auto resultFeatures = leaveSameFeaturesByThreshold(features, THRESHOLD_FOR_ONE_PATH);

    common::handleYandexOrigin(request, response);

    response << YCR_JSON(obj)
    {
        toFeaturesResponseJson(std::move(resultFeatures), {}, baseUrl(request),
                               responseOptions, obj);
    };
}

YCR_RESPOND_TO("OPTIONS /features/tour_along_route")
{
    if (common::handleYandexOrigin(request, response)) {
        response.setHeader(ALLOW_METHODS_HEADER, METHODS_GET_OPTIONS);
    }
}

YCR_RESPOND_TO("GET /version", contentType)
{
    static const std::vector<std::string> SUPPORTED_CONTENT_TYPES = {
        CONTENT_TYPE_PLAIN,
        CONTENT_TYPE_PROTOBUF,
        CONTENT_TYPE_TEXT_PROTOBUF
    };

    std::string responseContentType =
        yacare::bestContentType(contentType, SUPPORTED_CONTENT_TYPES);

    auto configuration = Configuration::instance();

    std::string version = configuration->dataAccess()->getVersion();
    if (responseContentType == CONTENT_TYPE_PLAIN) {
        response << version;
    } else if (responseContentType == CONTENT_TYPE_PROTOBUF) {
        response << makeFixedVersion(version);
    } else if (responseContentType == CONTENT_TYPE_TEXT_PROTOBUF) {
        response << makeFixedVersion(version).DebugString();
    } else {
        throw yacare::errors::BadRequest() << "Unsupported Content-Type";
    }

    response.setHeader(CONTENT_TYPE_HEADER, responseContentType);
}

YCR_RESPOND_TO("GET /features/directions",
               ll,
               dir,
               with_authors = false,
               graphType = db::GraphType::Road)
{
    Configuration::logDatasetMetrics();

    if (!has(dir) || *dir < 0. || *dir >= 360.) {
        throw yacare::errors::BadRequest() << "wrong dir param";
    }
    auto baseRay = db::Ray{ll, geolib3::Direction2(geolib3::Heading(*dir))};
    auto data = Configuration::instance()->dataAccess();
    auto graph = Configuration::instance()->graphByType(graphType);
    auto features = evalCoveredRoadGraphDirections(
        baseRay, *data, *graph, FeatureFilter::fromRequest(request));
    common::handleYandexOrigin(request, response);
    response << YCR_JSON(obj)
    {
        toDirectionsResponseJson(
            baseRay, features, baseUrl(request), with_authors, obj);
    };
}

YCR_RESPOND_TO("OPTIONS /features/directions")
{
    if (common::handleYandexOrigin(request, response)) {
        response.setHeader(ALLOW_METHODS_HEADER, METHODS_GET_OPTIONS);
    }
}

} // namespace maps::mrc::browser
