#include "feature_author.h"
#include "configuration.h"

#include <maps/libs/json/include/builder.h>
#include <maps/wikimap/mapspro/services/mrc/libs/blackbox_client/include/blackbox_client.h>

#include <boost/lexical_cast.hpp>

namespace maps::mrc::browser {

namespace {

const std::string YANDEX_NAME{"Yandex"};
const std::string DEFAULT_AVATAR_URL_ID{"0/0-0"};

} // namespace

const auto YANDEX_USER_INFO = [] {
    auto result = auth::UserInfo{};
    result.setDisplayName(auth::DisplayName{.publicName = YANDEX_NAME,
                                            .name = YANDEX_NAME,
                                            .avatarId = DEFAULT_AVATAR_URL_ID});
    result.setLogin(std::to_string(YANDEX_UID));
    result.setPublicId(std::to_string(YANDEX_UID));
    result.setUid(std::to_string(YANDEX_UID));
    return result;
}();

blackbox_client::UserInfo getBlackboxUserInfo(const db::Feature& feature)
{
    const auto& cfg = Configuration::instance();
    auto uid = cfg->dataAccess()->getUid(feature);

    if (uid != YANDEX_UID) {
        auto uidToUserInfo = cfg->blackboxClient().uidToUserInfoMap(
            {uid}, blackbox_client::QueryParams{}.requestDisplayName());
        auto itr = uidToUserInfo.find(uid);
        if (itr != uidToUserInfo.end()) {
            return itr->second;
        }
    }
    return YANDEX_USER_INFO;
}

FeatureIdToUserInfo getBlackboxUserInfos(const db::Features& features)
{
    FeatureIdToUserInfo result;

    const auto& cfg = Configuration::instance();
    auto featureIdToUid = cfg->dataAccess()->getUids(features);

    std::vector<blackbox_client::Uid> uids;
    for (const auto& [featureId, uid] : featureIdToUid) {
        if (uid != YANDEX_UID) {
            uids.push_back(uid);
        }
    }

    if (uids.empty()) {
        return result;
    }

    auto uidToUserInfo = cfg->blackboxClient().uidToUserInfoMap(
        uids, blackbox_client::QueryParams{}.requestDisplayName());
    for (const auto& [featureId, uid] : featureIdToUid) {
        auto itr = uidToUserInfo.find(uid);
        if (itr != uidToUserInfo.end()) {
            result.emplace(featureId, itr->second);
        }
        else {
            result.emplace(featureId, YANDEX_USER_INFO);
        }
    }

    for (const auto& feature : features) {
        result.emplace(feature.id(), YANDEX_USER_INFO);
    }
    return result;
}

void toFeatureAuthorJson(
    const json::ObjectBuilder& builder,
    const blackbox_client::UserInfo& userInfo)
{
    builder["author"] = [&](json::ObjectBuilder b) {
        b["uid"] = userInfo.uid();
        b["login"] = userInfo.login();
        auto displayName = getDisplayName(userInfo);
        b["name"] = displayName.name;
        b["avatarUrl"] = getAvatarUrl(*displayName.avatarId, "{size}");
    };
}

void toFeatureAuthorJson(
    const json::ObjectBuilder& builder,
    db::TId featureId,
    const FeatureIdToUserInfo& featureIdToUserInfo)
{
    blackbox_client::UserInfo userInfo = YANDEX_USER_INFO;

    auto itr = featureIdToUserInfo.find(featureId);
    if (itr != featureIdToUserInfo.end()) {
        userInfo = itr->second;
    }
    toFeatureAuthorJson(builder, userInfo);
}

auth::DisplayName getDisplayName(const blackbox_client::UserInfo& userInfo)
{
    auto result =
        userInfo.displayName().value_or(YANDEX_USER_INFO.displayName().value());
    result.avatarId = result.avatarId.value_or(DEFAULT_AVATAR_URL_ID);
    return result;
}

std::string getAvatarUrl(std::string_view avatarId, std::string_view sizeName)
{
    /// @see https://docs.yandex-team.ru/blackbox/methods/userinfo
    auto os = std::ostringstream{};
    os << "https://avatars.mds.yandex.net/get-yapic/" << avatarId << "/"
       << sizeName;
    return os.str();
}

} // namespace maps::mrc::browser
