#include "configuration.h"
#include "tools.h"
#include "yacare_params.h"

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/opencv.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/panorama_utils.h>

#include <maps/libs/http/include/http.h>
#include <maps/infra/yacare/include/yacare.h>

#include <boost/lexical_cast.hpp>

#include <cmath>

namespace maps::mrc::browser {

namespace {

const std::string ETAG_HEADER = "ETag";

void validateParameters(
    const common::PanoCutParams& cutParams, const common::PanoCropOut& cropOut)
{
    if (cropOut.width == 0 || cropOut.height == 0
        || cutParams.totalWidth <= cropOut.x
        || cutParams.totalWidth < cropOut.width
        || cutParams.totalHeight < cropOut.y + cropOut.height) {
        throw yacare::errors::BadRequest {"Invalid crop out rectangle"};
    }

    if (cropOut.scale < 0.f || 1.f < cropOut.scale ||
        common::round<std::uint32_t>(cropOut.width * cropOut.scale) == 0 ||
        common::round<std::uint32_t>(cropOut.height * cropOut.scale) == 0) {
        throw yacare::errors::BadRequest {
            "Invalid image scale"};
    }
}

template <typename TFunctor>
void processPanoImageRequest(const yacare::Request& request,
                             yacare::Response& response,
                             const std::string& mdsKey,
                             const common::PanoCutParams& cutParams,
                             const common::PanoCropOut& cropOut,
                             TFunctor modifier)
{
    static const auto s3MdsUrl =
        Configuration::instance()->s3MdsUrl().toString();

    validateParameters(cutParams, cropOut);

    maps::http::ETag etag(mdsKey);
    if (!request.preconditionSatisfied(etag)) {
        response.setStatus(yacare::HTTPStatus::NotModified);
        return;
    }
    response.setHeader(ETAG_HEADER, boost::lexical_cast<std::string>(etag));

    const auto tiles = common::makeAsyncPanoTilesDownloader(s3MdsUrl, mdsKey)
                           ->download(cutParams, cropOut);

    cv::Mat image =
        common::concatAndCropTiles(cutParams, cropOut, tiles);

    auto encodedImage = mrc::common::encodeImage(modifier(image));
    response.setHeader(CONTENT_TYPE_HEADER, CONTENT_TYPE_JPEG);
    response.write(reinterpret_cast<const char*>(encodedImage.data()),
        encodedImage.size());

}

std::string toString(const std::string& arg)
{
    return arg;
}

template<typename T>
std::string toString(T&& arg)
{
    return std::to_string(std::forward<T>(arg));
}

template<typename... Args>
std::string concat(Args&&... args)
{
    return ((toString(std::forward<Args>(args)) + "_") + ...);
}
} // namespace

YCR_RESPOND_TO("GET /pano/$/image", cut_params, crop_out, boxes = {})
{
    const auto postprocess = [&](cv::Mat& image) {
        image = common::fitImageTo(image,
            {common::round<std::size_t>(crop_out.width * crop_out.scale),
             common::round<std::size_t>(crop_out.height * crop_out.scale)});

        if (has(boxes)) {
            browser::drawImageBoxes(image, boxes);
        }
        return image;
    };

    processPanoImageRequest(
        request, response, argv[0], cut_params, crop_out, postprocess);
}

YCR_RESPOND_TO("GET /pano/$/thumbnail", cut_params, crop_out)
{
    processPanoImageRequest(request, response, argv[0],
        cut_params, crop_out, browser::toThumbnail);
}

YCR_RESPOND_TO("GET /v2/pano/$/image", heading, tilt, hfov, size)
{
    const std::string& panoramaId = argv[0];

    maps::http::ETag etag(
        concat(panoramaId, heading, tilt, hfov, size.first, size.second));

    if (!request.preconditionSatisfied(etag)) {
        response.setStatus(yacare::HTTPStatus::NotModified);
        return;
    }
    response.setHeader(ETAG_HEADER, boost::lexical_cast<std::string>(etag));

    const auto loadPanoramaProjection = [&] {
        try {
            return common::loadPanoramaProjection(
                Configuration::instance()->stvdescrUrl(),
                panoramaId,
                geolib3::Heading{heading},
                geolib3::Degrees{tilt},
                geolib3::Degrees{hfov},
                common::Size{size.first, size.second});
        } catch (const std::exception& ex) {
            ERROR() << ex.what();
            throw yacare::errors::NotFound("No panorama with such ID");
        }
    };
    const std::string encodedImage = loadPanoramaProjection();

    response.setHeader(CONTENT_TYPE_HEADER, CONTENT_TYPE_JPEG);
    response.write(reinterpret_cast<const char*>(encodedImage.data()),
        encodedImage.size());
}

} // namespace maps::mrc::browser
