#include "permission_checker.h"
#include "common.h"

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>

#include <exception>
#include <unordered_set>

namespace maps::mrc::browser {

const std::map<std::string, db::FeaturePrivacy>
WikiAclPermissionChecker::permissionPathToPrivacy{
    {"mpro/social/feedback/task/hidden", db::FeaturePrivacy::Restricted},
    {"mpro/mrc/photos/view-hidden", db::FeaturePrivacy::Secret}
};

bool WikiAclPermissionChecker::userMayViewPhotos(
    std::optional<Uid> uid, db::FeaturePrivacy privacy)
{
    if (privacy == db::FeaturePrivacy::Public) {
        return true;
    }

    if (uid) {
        try {
            auto permissions = loadUserPermissions(*uid);
            ASSERT(permissions);
            return std::find(
                       permissions->allowedPermissions.begin(),
                       permissions->allowedPermissions.end(),
                       privacy) != permissions->allowedPermissions.end();
        } catch (const std::exception& ex) {
            WARN() << "Failed to get permissions for user " << *uid << ": "
                   << ex.what();
        }
    }
    return false;
}

std::shared_ptr<WikiAclPermissionChecker::TimedPermissions>
WikiAclPermissionChecker::loadUserPermissions(IPermissionChecker::Uid uid)
{
    auto now = chrono::TimePoint::clock::now();

    auto evalTimedPermissions = [&](Uid uid) {
        INFO() << "loading permissions of user " << uid;
        return TimedPermissions{evalUserPermissions(uid), now};
    };

    auto value = cache_.getOrEmplace(uid, evalTimedPermissions);

    if (value) {
        if (now - value->created < expirationPeriod_) {
            return value;
        } else {
            INFO() << "cached permissions of user " << uid << " has expired";
        }
    }

    return cache_.put(uid, evalTimedPermissions(uid));
}

std::string WikiAclPermissionChecker::formatPathsParam() const
{
    json::Builder bld;
    bld <<
        [&](json::ArrayBuilder bld) {
            for (const auto& [path, _]: permissionPathToPrivacy) {
                bld << path;
            }
        };

    return bld.str();
}

http::URL WikiAclPermissionChecker::makeCheckPermissionsUrl(Uid uid) const
{
    std::stringstream sstr;
    sstr << "http://" << host_ << "/users/" << uid << "/check-permissions";
    http::URL url(sstr.str());
    url.addParam("paths", formatPathsParam());

    return url;
}

std::vector<db::FeaturePrivacy>
WikiAclPermissionChecker::evalUserPermissions(Uid uid)
{
    std::vector<db::FeaturePrivacy> result;
    http::Request request(client_, http::GET, makeCheckPermissionsUrl(uid));
    INFO() << "Performing request to url " << request.url();
    request.addHeader(maps::auth::SERVICE_TICKET_HEADER, tvmTicketProvider_());
    auto response = request.perform();

    REQUIRE(response.status() == 200, "Got unexpected response status " << response.status());
    auto responseJson = json::Value::fromStream(response.body());
    const std::vector<std::string> paths(responseJson["paths"]);

    for (const auto& path : paths) {
        if (permissionPathToPrivacy.count(path)) {
            result.push_back(permissionPathToPrivacy.at(path));
        } else {
            WARN() << "Got unexpected path " << path;
        }
    }
    return result;
}

} // namespace maps::mrc::browser
