#pragma once

#include "data_access.h"
#include "graph.h"
#include "types.h"

#include <maps/infra/yacare/include/request.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/tile/include/tile.h>
#include <maps/wikimap/mapspro/services/mrc/libs/blackbox_client/include/blackbox_client.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/geometry.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/visibility.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/export_gen/lib/graph.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>

#include <boost/optional.hpp>

#include <functional>
#include <shared_mutex>
#include <string>
#include <unordered_map>
#include <vector>

namespace maps::mrc::browser {

using FeaturesRange = boost::iterator_range<db::Features::iterator>;

const std::string ALLOW_METHODS_HEADER = "Access-Control-Allow-Methods";
const std::string CONTENT_TYPE_HEADER = "Content-Type";

const std::string CONTENT_TYPE_JAVASCRIPT = "application/javascript";
const std::string CONTENT_TYPE_JPEG = "image/jpeg";
const std::string CONTENT_TYPE_JSON = "application/json";
const std::string CONTENT_TYPE_PNG = "image/png";
const std::string CONTENT_TYPE_XML = "text/xml";
const std::string CONTENT_TYPE_PLAIN = "text/plain";
const std::string CONTENT_TYPE_PROTOBUF = "application/x-protobuf";
const std::string CONTENT_TYPE_TEXT_PROTOBUF = "text/x-protobuf";
const std::string METHODS_GET_OPTIONS = "GET, OPTIONS";

static const std::string JSON_FILE_EXT = ".json";

static const std::string FIELD_ANGLE = "angle";
static const std::string FIELD_ID = "id";
static const std::string FIELD_CAMERA_DIRECTION = "cameraDirection";
static const std::string FIELD_DAYS = "days";
static const std::string FIELD_DISTANCE = "distance";
static const std::string FIELD_GEOMETRY = "geometry";
static const std::string FIELD_EDGE = "edge";
static const std::string FIELD_HEADING = "heading";
static const std::string FIELD_IMAGE_PREVIEW = "imagePreview";
static const std::string FIELD_IMAGE_FULL = "imageFull";
static const std::string FIELD_PAGE = "page";
static const std::string FIELD_POS = "pos";
static const std::string FIELD_PROPERTIES = "properties";
static const std::string FIELD_SOURCE_ID = "sourceId";
static const std::string FIELD_TARGET_GEOMETRY = "targetGeometry";
static const std::string FIELD_TIMESTAMP = "timestamp";
static const std::string FIELD_TYPE = "type";

static const std::string STYLE_NAME_NMAPS = "nmaps";
static const std::string STYLE_NAME_PANO = "pano";
static const std::string VALUE_FORWARD = "forward";
static const std::string VALUE_BACKWARD = "backward";

enum class Direction { Forward, Backward };

constexpr size_t DEFAULT_PHOTO_WIDTH_PX = 1600;
constexpr size_t DEFAULT_PHOTO_HEIGHT_PX = 1200;
constexpr double SIMILARITY_RADIUS_METERS = 40.;
constexpr double DIRECT_SPATIAL_CONNECTION_DISTANCE_METERS = 30.;
constexpr double TURNAROUND_SPATIAL_CONNECTION_DISTANCE_METERS = 10.;
constexpr double HISTORICAL_CONNECTION_DISTANCE_METERS =
    DIRECT_SPATIAL_CONNECTION_DISTANCE_METERS / 2.0;
constexpr size_t SIMILAR_LIMIT = 20;
constexpr int MAX_ZOOM = 21;
using geolib3::PI;

enum class LoopControl { Break, Continue };

inline std::string baseUrl(const yacare::Request& request)
{
    return "https://" + request.env("HTTP_HOST") + "/";
}

std::string formatToContentType(const std::string& format);

using IdToPointMap = std::unordered_map<db::TId, geolib3::Point2>;

Direction toDirection(const std::string& direction);

std::string toCameraDirection(const db::Feature& feature);

struct ImagePreviewManipulator {
    const db::Feature& feature;
    const std::string& url;

    void json(json::ObjectBuilder builder) const;
};

ImagePreviewManipulator imagePreview(const db::Feature& feature,
                                     const std::string& url);

std::vector<std::string> split(const std::string& source, const char* delimeter);

geolib3::Polyline2 toTrack(const std::string& requestBody);

void formatFeatureHotspotProperties(const std::string& url, json::ObjectBuilder obj, const db::Feature& dbFeature, const IdToPointMap& targetPoints);

void toFeaturesResponseJson(db::Features features /* to move */,
                            const IdToPointMap& targetPoints,
                            const std::string& url,
                            const ResponseOptions& responseOptions,
                            json::ObjectBuilder builder);

void toPathJson(db::TId baseFeatureId,
                db::Features features /* to move */,
                const std::string& url,
                const ResponseOptions& responseOptions,
                json::ObjectBuilder builder);

void toSimilarJson(db::TId baseFeatureId,
                   db::Features features /* to move */,
                   const IdToPointMap& targetPoints,
                   const std::string& url,
                   const ResponseOptions& responseOptions,
                   json::ObjectBuilder builder);

void toNearJson(const geolib3::Point2& geodeticPos,
                const std::optional<geolib3::Direction2>& dir,
                double radiusMeters,
                db::Features features /* to move */,
                const IdToPointMap& targetPoints,
                const std::string& url,
                const ResponseOptions& responseOptions,
                json::ObjectBuilder builder);

std::string makePath(const std::string& dir, const std::string& file);

bool userMayViewPhotos(std::optional<blackbox_client::Uid>, db::FeaturePrivacy);

struct FeatureFilter {
    std::vector<db::Dataset> datasets;
    std::optional<db::CameraDeviation> cameraDeviation;
    std::optional<blackbox_client::Uid> uid;
    std::optional<chrono::TimePoint> actualizedAfter;
    std::optional<chrono::TimePoint> actualizedBefore;

    static FeatureFilter fromRequest(const yacare::Request&);

    // returns true if feature should be excluded from response
    bool operator()(const db::Feature&) const;

    auto introspect() const
    {
        return std::tie(datasets,
                        cameraDeviation,
                        uid,
                        actualizedAfter,
                        actualizedBefore);
    }
};

/**
 * loading with margins to avoid cutting signals along the edges of the tiles
 */
db::Features loadFeaturesWithMargins(IDataAccessPtr dataAccess,
                                     const geolib3::BoundingBox& mercatorBbox,
                                     double marginInMetricMeters,
                                     const FeatureFilter& filter);

db::Features loadFeaturesWithMargins(IDataAccessPtr dataAccess,
                                     const geolib3::BoundingBox& mercatorBbox,
                                     const FeatureFilter& filter);

db::Features loadFeaturesTourAlongLine(IDataAccessPtr dataAccess,
                                       const IGraphPtr graph,
                                       const geolib3::Polyline2& geoLine,
                                       db::CameraDeviation);

db::Features leaveSameFeaturesByThreshold(const db::Features& features, double thresholdMeters);

void orderByPassages(FeaturesRange features);


cv::Mat toThumbnail(const cv::Mat& image);

void drawImageBoxes(cv::Mat& image, const common::ImageBoxes& boxes);


/// area in which a long time between shots (traffic light, jam)
static constexpr double GAP_STOP_LIMIT_METERS{30};
static constexpr std::chrono::seconds GAP_STOP_LIMIT_TIME{120};

/// regular driving
static constexpr double GAP_DRIVING_LIMIT_METERS{80};
static constexpr std::chrono::seconds GAP_DRIVING_LIMIT_TIME{15};


enum class DistanceCheck { On, Off };

template <DistanceCheck distancePolicy>
struct Gap {
    /// check if photos taken in different passages
    bool operator()(const db::Feature& lhs, const db::Feature& rhs) const
    {
        if (db::isStandalonePhotosDataset(lhs.dataset()) ||
            lhs.dataset() != rhs.dataset() ||
            lhs.sourceId() == db::feature::NO_SOURCE_ID ||
            lhs.sourceId() != rhs.sourceId()) {
            return true;
        }

        auto timeInterval = abs(lhs.timestamp() - rhs.timestamp());
        if (timeInterval > GAP_STOP_LIMIT_TIME)
            return true;

        auto diffOfMeters = geolib3::fastGeoDistance(lhs.geodeticPos(), rhs.geodeticPos());
        if constexpr (distancePolicy == DistanceCheck::On) {
            if (diffOfMeters > GAP_DRIVING_LIMIT_METERS)
                return true;
        }

        return diffOfMeters > GAP_STOP_LIMIT_METERS
               && timeInterval > GAP_DRIVING_LIMIT_TIME;
    };
};


template <DistanceCheck distancePolicy = DistanceCheck::On, typename Consumer>
void forEachPassage(FeaturesRange features, const Consumer& consumer)
{
    orderByPassages(features);
    for (auto begin = features.begin(); begin != features.end();) {
        auto end = std::adjacent_find(begin, features.end(), Gap<distancePolicy>{});
        if (end != features.end())
            end = std::next(end);
        if (consumer(begin, end) == LoopControl::Break)
            break;
        begin = end;
    }
}

using SnapLossFunc =
    std::function<std::optional<double>(const db::Ray&, const db::Ray&)>;

SnapLossFunc makeSnapLossFunc(
    Meters snapDistanceThreshold, geolib3::Radians snapAngleThreshold);

std::optional<road_graph::EdgeId> snapToGraph(
    const IGraph& graph,
    db::Ray& rayToChange,
    SnapLossFunc lossFunc =
        makeSnapLossFunc(Meters{export_gen::COVERAGE_METERS_SNAP_THRESHOLD},
                         export_gen::COVERAGE_ANGLE_DIFF_THRESHOLD));

std::optional<road_graph::EdgeId> snapToGraph(
    const IGraph& graph,
    db::Feature& featureToChange,
    SnapLossFunc lossFunc =
        makeSnapLossFunc(Meters{export_gen::COVERAGE_METERS_SNAP_THRESHOLD},
                         export_gen::COVERAGE_ANGLE_DIFF_THRESHOLD));

class Semaphore {
    std::mutex guard_;
    std::condition_variable notifier_;
    bool stop_{false};

public:
    template<class Duration>
    bool waitForStop(const Duration& duration)
    {
        std::unique_lock lock{guard_};
        notifier_.wait_for(lock, duration, [this] { return stop_; });
        return stop_;
    }

    void notifyAll()
    {
        notifier_.notify_all();
    }

    void stop()
    {
        {
            std::lock_guard lock{guard_};
            stop_ = true;
        }
        notifier_.notify_all();
    }
};

class IdSet {
    std::shared_mutex guard_;
    std::unordered_set<db::TId> ids_;

public:
    void reset(std::unordered_set<db::TId> ids)
    {
        std::unique_lock lock{guard_};
        ids_ = std::move(ids);
    }

    bool contains(db::TId id)
    {
        std::shared_lock lock{guard_};
        return ids_.count(id);
    }

    auto skip()
    {
        return boost::adaptors::filtered(
            [this](db::TId id) { return !contains(id); });
    }
};

std::optional<db::Feature> findClosestEdgeCoveringFeature(
    IDataAccess& data,
    const IGraph& graph,
    road_graph::EdgeId baseEdgeId,
    const geolib3::Point2& pos,
    const std::optional<double>& minDistance = std::nullopt,
    const std::optional<double>& maxDistance = std::nullopt,
    const FeatureFilter& skip = {});

// Snap ray on road graph and follow along it to collect snapped features
// within a hardcoded distance from the base ray (see
// DIRECT_SPATIAL_CONNECTION_DISTANCE_METERS). It collectes nearby features on
// graph such that it is possible to step from one to another along graph
// edge. I.e. there is a part of features connected to another part of
// features with directed graph edges. There is also might be single
// "turnaround" feature within TURNAROUND_SPATIAL_CONNECTION_DISTANCE_METERS
// distance which is snapped to an edge in reverse direction Note: If there is
// feature within 1 meter on an edge it is not included in the result.
db::Features evalCoveredRoadGraphDirections(db::Ray baseRay,
                                            IDataAccess& data,
                                            const IGraph& graph,
                                            const FeatureFilter& skip = {});

void toDirectionsResponseJson(const db::Ray&,
                              const db::Features&,
                              const std::string& url,
                              bool withAuthors,
                              json::ObjectBuilder result);

geolib3::PointsVector makeGridPoints(const geolib3::BoundingBox&,
                                     size_t rows,
                                     size_t cols);

void logUgcPhotoViewEvent(
    const std::optional<uint64_t>& userId,
    const db::TIds& featureIds,
    const yacare::Request& request);

Meters fastGeoDistanceAlong(const geolib3::Polyline2& polyline,
                            common::geometry::PolylinePosition lhs,
                            common::geometry::PolylinePosition rhs);

db::Ray reverseRay(const db::Ray& ray);

} // namespace maps::mrc::browser
