#include "common.h"
#include "fixture.h"
#include "mocks.h"

#include <yandex/maps/mrc/unittest/utils.h>

#include <maps/wikimap/mapspro/services/mrc/browser/lib/configuration.h>
#include <maps/wikimap/mapspro/services/mrc/browser/lib/feature_author.h>
#include <maps/wikimap/mapspro/services/mrc/libs/blackbox_client/include/blackbox_client.h>

#include <maps/libs/introspection/include/comparison.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <yandex/maps/proto/mobile_config/mapkit2/layers.pb.h>

#include <maps/libs/http/include/test_utils.h>
#include <maps/libs/concurrent/include/scoped_guard.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/infra/yacare/include/yacare.h>
#include <maps/infra/yacare/include/test_utils.h>

#include <maps/libs/json/include/value.h>
#include <maps/libs/img/include/algorithm.h>
#include <maps/libs/img/include/raster.h>

#include <library/cpp/testing/gmock_in_unittest/gmock.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <map>
#include <memory>
#include <string>
#include <sstream>

using namespace std::literals;
using namespace testing;

using FixedVersion =
    yandex::maps::proto::mobile_config::mapkit2::layers::FixedVersion;


namespace maps::mrc::blackbox_client {

using maps::introspection::operator==;

} // namespace maps::mrc::blackbox_client

namespace maps::mrc::browser::tests {

const std::string HTTP_HOST = "localhost";
const std::string REMOTE_ADDR = "127.0.0.1";


namespace {

cv::Mat expectedTestImage()
{
    static auto blob
        = maps::common::readFileToString(SRC_("expected/fisheye.jpeg"));
    return mrc::common::decodeImage(blob);
}

cv::Mat expectedTestImageRotated()
{
    static auto blob
        = maps::common::readFileToString(SRC_("expected/fisheye.orientation_180degree.jpeg"));
    return mrc::common::decodeImage(blob);
}

cv::Mat expectedTestImageBlurred()
{
    static auto blob
        = maps::common::readFileToString(SRC_("expected/fisheye.blurred.jpeg"));
    return mrc::common::decodeImage(blob);
}

cv::Mat expectedSignalqImageBlurred()
{
    static auto blob
        = maps::common::readFileToString(SRC_("expected/signalq.blurred.jpeg"));
    return mrc::common::decodeImage(blob);
}

cv::Mat expectedTestThumbnail()
{
    static auto blob
        = maps::common::readFileToString(SRC_("expected/fisheye.thumb.jpeg"));
    return mrc::common::decodeImage(blob);
}

cv::Mat expectedTestThumbnailBlurred()
{
    static auto blob
        = maps::common::readFileToString(SRC_("expected/fisheye.thumb.blurred.jpeg"));
    return mrc::common::decodeImage(blob);
}

} // namespace


Y_UNIT_TEST_SUITE(feature_api_should) {

Y_UNIT_TEST(test_get_tiles)
{
    Fixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/tiles")
                .setParams("x=79222&y=41104&z=17&l=mrcss"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    EXPECT_EQ(
        img::RasterImage::fromPngBlob(common::toBytes(resp.body)),
        img::RasterImage::fromPngFile(SRC_("expected/tiles_mrcss_79222_41104_17.png"))
    );
}

Y_UNIT_TEST(test_get_tiles_not_500)
{
    /// @see https://paste.yandex-team.ru/897511
    UserMayViewPhotosFixture<false> fixture;
    http::MockRequest rq(
        http::GET,
        http::URL("http://localhost/tiles")
            .setParams(
                "l=mrce&x=63&y=15&z=6&scale=1&camera-direction=front"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_NE(resp.status, 500);
}

Y_UNIT_TEST(test_get_features_similar_not_500)
{
    /// @see https://st.yandex-team.ru/MAPSMRC-2498
    Fixture fixture;
    constexpr db::TId FEATURE_ID = 2;
    {
        auto txn = fixture.txnHandle();
        auto gtw = db::FeatureGateway{*txn};
        auto feature = gtw.loadById(FEATURE_ID);
        feature.setPrivacy(db::FeaturePrivacy::Restricted);
        gtw.update(feature, db::UpdateFeatureTxn::Yes);
        txn->commit();
    }

    http::MockRequest rq(http::GET,
                         http::URL("http://localhost/features/" +
                                   std::to_string(FEATURE_ID) + "/similar")
                             .setParams("camera-direction=front&"
                                        "snap_to_road_graph=true&"
                                        "with_authors=true"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_NE(resp.status, 500);
}

Y_UNIT_TEST(get_feature_400_response_for_bad_orientation)
{
    Fixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/feature/1/image")
                .setParams("orientation=9"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 400);
}

Y_UNIT_TEST(get_feature_full_response_for_request_with_orientation)
{
    Fixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/feature/1/image")
                .setParams("orientation=3"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    const cv::Mat image = mrc::common::decodeImage(resp.body);
    EXPECT_TRUE(unittest::areImagesEq(image, expectedTestImageRotated()));
}

Y_UNIT_TEST(test_get_version)
{
    const std::string ACCEPT_HEADER = "Accept";
    FixtureFb fixture;
    http::URL url("http://localhost/version");

    http::MockRequest plainRq(http::GET, url);
    auto plainResp = yacare::performTestRequest(plainRq);
    EXPECT_EQ(plainResp.status, 200);
    EXPECT_EQ(plainResp.headers["Content-Type"], "text/plain");
    std::string version = plainResp.body;
    EXPECT_NE(version, "");

    http::MockRequest protoRq(http::GET, url);
    protoRq.headers.emplace(ACCEPT_HEADER, "application/x-protobuf");
    auto protoResp = yacare::performTestRequest(protoRq);
    EXPECT_EQ(protoResp.status, 200);
    EXPECT_EQ(protoResp.headers["Content-Type"], "application/x-protobuf");

    FixedVersion protoVersion;
    Y_PROTOBUF_SUPPRESS_NODISCARD protoVersion.ParseFromArray(protoResp.body.data(), protoResp.body.size());
    EXPECT_TRUE(protoVersion.IsInitialized());
    EXPECT_EQ(protoVersion.value(), TString(version));
}


} // Y_UNIT_TEST_SUITE(feature_api_should, Fixture)


Y_UNIT_TEST_SUITE(feature_api_privacy_should) {

Y_UNIT_TEST(get_tiles_seq_empty_response_for_regular_user_in_unpublished_region)
{
    RegularUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/tiles")
                .setParams("x=79222&y=41104&z=17&l=mrcss"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 204);
    EXPECT_EQ(resp.body.size(), 0u);
}

Y_UNIT_TEST(get_tiles_seq_full_response_for_trusted_user_in_unpublished_region)
{
    TrustedUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/tiles")
                .setParams("x=79222&y=41104&z=17&l=mrcss"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    EXPECT_EQ(
        img::RasterImage::fromPngBlob(common::toBytes(resp.body)),
        img::RasterImage::fromPngFile(SRC_("expected/tiles_mrcss_79222_41104_17.png"))
    );
}

Y_UNIT_TEST(get_tiles_seq_full_response_for_regular_user_in_published_region)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/tiles")
                .setParams("x=79222&y=41104&z=17&l=mrcss"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    EXPECT_EQ(
        img::RasterImage::fromPngBlob(common::toBytes(resp.body)),
        img::RasterImage::fromPngFile(SRC_("expected/tiles_mrcss_79222_41104_17.png"))
    );
}

Y_UNIT_TEST(get_tiles_edges_empty_response_for_regular_user_in_unpublished_region)
{
    RegularUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/tiles")
                .setParams("x=79222&y=41104&z=17&l=mrcss"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 204);
    EXPECT_EQ(resp.body.size(), 0u);
}

Y_UNIT_TEST(get_tiles_edges_full_response_for_trusted_user_in_unpublished_region)
{
    TrustedUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/tiles")
                .setParams("x=79222&y=41104&z=17&l=mrcss"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    EXPECT_EQ(
        img::RasterImage::fromPngBlob(common::toBytes(resp.body)),
        img::RasterImage::fromPngFile(SRC_("expected/tiles_mrcss_79222_41104_17.png"))
    );
}

Y_UNIT_TEST(get_tiles_edges_full_response_for_regular_user_in_published_region)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/tiles")
                .setParams("x=79222&y=41104&z=17&l=mrcss"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    // maps::common::writeFile("tiles_mrcss_9902_5138_14.png", resp.body);
    EXPECT_EQ(
        img::RasterImage::fromPngBlob(common::toBytes(resp.body)),
        img::RasterImage::fromPngFile(SRC_("expected/tiles_mrcss_79222_41104_17.png"))
    );
}

Y_UNIT_TEST(get_feature_403_response_for_regular_user_in_unpublished_region)
{
    RegularUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/1/image"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 403);
}

Y_UNIT_TEST(get_feature_403_response_for_hidden_feature_regular_user)
{
    RegularUserAccessPublicRegionFixture fixture;
    setFeaturesPrivacy(fixture.pgPool(), db::FeaturePrivacy::Restricted);
    concurrent::ScopedGuard onExit([&]{ setFeaturesPrivacy(fixture.pgPool(), db::FeaturePrivacy::Public); });

    http::MockRequest rq(http::GET,
        http::URL("http://localhost/feature/1/image")
            .addParam(yacare::tests::USER_ID_HEADER, fixture.UID)
    );

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 403);
}

Y_UNIT_TEST(get_feature_response_for_trusted_user_in_unpublished_region)
{
    TrustedUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/1/image"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
}

Y_UNIT_TEST(get_feature_response_for_hidden_feature_trusted_user)
{
    TrustedUserAccessPublicRegionFixture fixture;
    setFeaturesPrivacy(fixture.pgPool(), db::FeaturePrivacy::Restricted);
    concurrent::ScopedGuard onExit([&]{ setFeaturesPrivacy(fixture.pgPool(), db::FeaturePrivacy::Public); });

    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/1/image"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
}

Y_UNIT_TEST(get_feature_full_response_for_regular_user_in_published_region)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/1/image"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    const cv::Mat image = mrc::common::decodeImage(resp.body);
    EXPECT_TRUE(unittest::areImagesEq(image, expectedTestImage()));
}

Y_UNIT_TEST(get_feature_thumbnail_403_response_for_regular_user_in_unpublished_region)
{
    RegularUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/1/thumbnail"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 403);
}

Y_UNIT_TEST(get_feature_thumbnail_response_for_trusted_user_in_unpublished_region)
{
    TrustedUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/1/thumbnail"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
}

Y_UNIT_TEST(get_feature_thumbnail_full_response_for_regular_user_in_published_region)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/1/thumbnail"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    EXPECT_EQ(resp.body.size(), 16562u);
}

Y_UNIT_TEST(get_feature_image_with_blurred_privacy)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/2/image"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    const cv::Mat image = mrc::common::decodeImage(resp.body);
    EXPECT_TRUE(unittest::areImagesEq(image, expectedTestImageBlurred()));
}

Y_UNIT_TEST(get_signalq_feature_image_with_blurred_privacy)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/19/image"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    const cv::Mat image = mrc::common::decodeImage(resp.body);
    EXPECT_TRUE(unittest::areImagesEq(image, expectedSignalqImageBlurred()));
}

Y_UNIT_TEST(get_feature_thumbnail_plain)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/1/thumbnail"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    const cv::Mat image = mrc::common::decodeImage(resp.body);
    EXPECT_TRUE(unittest::areImagesEq(image, expectedTestThumbnail()));
}

Y_UNIT_TEST(get_feature_thumbnail_with_blurred_privacy)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET, http::URL("http://localhost/feature/2/thumbnail"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    const cv::Mat image = mrc::common::decodeImage(resp.body);
    EXPECT_TRUE(unittest::areImagesEq(image, expectedTestThumbnailBlurred()));
}

Y_UNIT_TEST(get_features_hotspot_empty_response_for_regular_user_in_unpublished_region)
{
    RegularUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/hotspot")
                .setParams("x=79222&y=41104&z=17"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    EXPECT_EQ(resp.body, EMPTY_HOTSPOT_RESPONSE);
}

Y_UNIT_TEST(get_features_hotspot_empty_response_for_hidden_feature_regular_user)
{
    RegularUserAccessPublicRegionFixture fixture;
    setFeaturesPrivacy(fixture.pgPool(), db::FeaturePrivacy::Restricted);
    concurrent::ScopedGuard onExit([&]{ setFeaturesPrivacy(fixture.pgPool(), db::FeaturePrivacy::Public); });

    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/hotspot")
                .setParams("x=79222&y=41104&z=17"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    EXPECT_EQ(resp.body, EMPTY_HOTSPOT_RESPONSE);
}

Y_UNIT_TEST(get_features_hotspot_full_response_for_trusted_user_in_unpublished_region)
{
    TrustedUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/hotspot")
                .setParams("x=79222&y=41104&z=17"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 8u);
}


Y_UNIT_TEST(get_features_hotspot_full_response_for_regular_user_in_published_region)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/hotspot")
                .setParams("x=79222&y=41104&z=17"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 8u);
}

Y_UNIT_TEST(get_features_hotspot_full_response_for_walk_feature)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/hotspot")
                .setParams("x=652445&y=325766&z=20"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 2u);
    for (const auto& feature : json["data"]["features"]) {
        EXPECT_TRUE(
                feature["properties"]["properties"].hasField("targetGeometry"));
    }
}

Y_UNIT_TEST(get_features_path_empty_response_for_regular_user_in_unpublished_region)
{
    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_path.response.schema.json");
    const std::string RESPONSE_PATH = "path.json";

    RegularUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/2/path"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);

    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);

    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 0u);
}

Y_UNIT_TEST(get_features_path_empty_response_for_hidden_feature_regular_user)
{
    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_path.response.schema.json");
    const std::string RESPONSE_PATH = "path.json";

    RegularUserAccessPublicRegionFixture fixture;
    setFeaturesPrivacy(fixture.pgPool(), db::FeaturePrivacy::Restricted);
    concurrent::ScopedGuard onExit([&]{ setFeaturesPrivacy(fixture.pgPool(), db::FeaturePrivacy::Public); });

    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/2/path"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);

    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);

    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 0u);
}

Y_UNIT_TEST(get_features_path_full_response_for_trusted_user_in_unpublished_region)
{
    TrustedUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/2/path"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 4u);
}

Y_UNIT_TEST(get_features_path_full_response_for_regular_user_in_published_region)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/2/path"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 4u);
}

Y_UNIT_TEST(get_features_similar_403_response_for_regular_user_in_unpublished_region)
{
    RegularUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/2/similar"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 403);
}

Y_UNIT_TEST(get_features_similar_full_response_for_trusted_user_in_unpublished_region)
{
    TrustedUserAccessSecretRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/2/similar"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 2u);
}

Y_UNIT_TEST(get_features_similar_full_response_for_regular_user_in_published_region)
{
    RegularUserAccessPublicRegionFixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/2/similar"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 2u);
}

Y_UNIT_TEST(get_features_near_empty_response_for_regular_user_in_unpublished_region)
{
    RegularUserAccessSecretRegionFixtureFb fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/near?ll=43.9992,56.3218"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 0u);
}

Y_UNIT_TEST(get_features_near_full_response_for_trusted_user_in_unpublished_region)
{
    TrustedUserAccessSecretRegionFixtureFb fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/near?ll=43.9992,56.3218"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 0u);
}

Y_UNIT_TEST(get_features_near_full_response_for_regular_user_in_published_region)
{
    RegularUserAccessPublicRegionFixtureFb fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/near?ll=43.9992,56.3218"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 2u);
}

template <class Fixture>
bool testCoverage()
{
    Fixture fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/coverage")
                .addParam("bbox", "43.998,56.3218~44.0,56.322"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
    auto json = json::Value::fromString(resp.body);
    return json["coverage"].as<bool>();
}

Y_UNIT_TEST(get_features_coverage)
{
    EXPECT_FALSE(testCoverage<RegularUserAccessSecretRegionFixture>());
    EXPECT_FALSE(testCoverage<OutsourcerUserAccessSecretRegionFixture>());
    EXPECT_TRUE(testCoverage<TrustedUserAccessSecretRegionFixture>());
    EXPECT_FALSE(testCoverage<RegularUserAccessRestrictedRegionFixture>());
    EXPECT_TRUE(testCoverage<OutsourcerUserAccessRestrictedRegionFixture>());
    EXPECT_TRUE(testCoverage<TrustedUserAccessRestrictedRegionFixture>());
    EXPECT_TRUE(testCoverage<RegularUserAccessPublicRegionFixture>());
    EXPECT_TRUE(testCoverage<OutsourcerUserAccessPublicRegionFixture>());
    EXPECT_TRUE(testCoverage<TrustedUserAccessPublicRegionFixture>());
}


Y_UNIT_TEST(get_features_stamp)
{
    FixtureFb fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/stamp.xml"));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
}


Y_UNIT_TEST(get_features_stamp_pedestrian)
{
    FixtureFb fixture;
    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/stampPedestrian.xml"));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);
}


} // Y_UNIT_TEST_SUITE(feature_api_privacy_should)

Y_UNIT_TEST_SUITE(feature_api_author_should) {

const std::string PUBLIC_ID = "123";
const std::string AVATAR_ID = "123/avatar-id";
const std::string LOGIN = "test_login";
const std::string NAME = "test_name";
const std::string UID = "123";

using namespace blackbox_client;

const auto USER_INFO = [] {
    auto result = auth::UserInfo{};
    result.setDisplayName(auth::DisplayName{
        .publicName = NAME, .name = NAME, .avatarId = AVATAR_ID});
    result.setLogin(LOGIN);
    result.setPublicId(UID);
    result.setUid(UID);
    return result;
}();

std::string makeAvatarUrl(const std::string& avatarId)
{
    return "https://avatars.mds.yandex.net/get-yapic/" + avatarId + "/{size}";
}

bool isExpectedAuthor(const json::Value& author, const UserInfo& expected)
{
    return author["uid"].as<std::string>() == expected.uid()
        && author["login"].as<std::string>() == expected.login()
        && author["name"].as<std::string>() == expected.displayName()->name
        && author["avatarUrl"].as<std::string>() == makeAvatarUrl(*expected.displayName()->avatarId);
}

inline auto makeNiceMockBlackboxClient() {
    return std::make_unique<NiceMock<MockBlackboxClient>>();
}

Y_UNIT_TEST(test_hotspot_request_with_authors)
{
    Fixture fixture;
    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_hotspots.response.schema.json");
    const std::string RESPONSE_PATH = "hotspot.json";

    http::MockRequest rq(http::GET,
        http::URL("http://localhost/features/hotspot")
            .setParams("x=81555&y=40720&z=17&with_authors=true"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);

    EXPECT_EQ(resp.status, 200);
    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);
    auto receivedJson = json::Value::fromString(resp.body);

    EXPECT_EQ(receivedJson["data"]["features"].size(), 2u);
    for (const auto& feature : receivedJson["data"]["features"]) {
        EXPECT_TRUE(isExpectedAuthor(feature["properties"]["properties"]["author"], YANDEX_USER_INFO));
    }
}

Y_UNIT_TEST(test_hotspot_request_walk_features_with_authors)
{
    Fixture fixture;

    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_hotspots.response.schema.json");
    const std::string RESPONSE_PATH = "hotspot_walks.json";

    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/hotspot")
                .setParams("x=652445&y=325766&z=20")
                .addParam("with_authors", "true"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);

    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);

    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 2u);
    for (const auto& feature : json["data"]["features"]) {
        EXPECT_TRUE(isExpectedAuthor(feature["properties"]["properties"]["author"], YANDEX_USER_INFO));
    }
}

Y_UNIT_TEST(test_hotspot_request_walk_features_without_objects)
{
    Fixture fixture;

    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_hotspots.response.schema.json");
    const std::string RESPONSE_PATH = "hotspot_walks.json";

    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/hotspot")
                .setParams("x=79206&y=41106&z=17"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    ASSERT_EQ(resp.status, 200);

    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);

    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), 3u);
}

Y_UNIT_TEST(test_path_request_with_authors)
{
    Fixture fixture;

    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_path.response.schema.json");
    const std::string RESPONSE_PATH = "path.json";

    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/2/path")
                .addParam("with_authors", "true"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);

    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);

    auto json = json::Value::fromString(resp.body);

    EXPECT_TRUE(json["data"].hasField("author"));
    EXPECT_TRUE(isExpectedAuthor(json["data"]["author"], YANDEX_USER_INFO));
}

Y_UNIT_TEST(test_similar_request_with_authors)
{
    Fixture fixture;

    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_similar.response.schema.json");
    const std::string RESPONSE_PATH = "similar.json";

    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/2/similar")
                .addParam("with_authors", "true"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);

    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);

    auto json = json::Value::fromString(resp.body);

    for (const auto& feature : json["data"]["features"]) {
        EXPECT_TRUE(feature.hasField("author"));
        EXPECT_TRUE(isExpectedAuthor(feature["author"], YANDEX_USER_INFO));
    }
}

Y_UNIT_TEST(test_similar_request_ride_with_authors)
{
    MockedGraphFixture fixture;

    auto roadRelations = std::vector{makeRelation(13), makeRelation(14)};
    auto pedestrianRelations = std::vector{makeRelation(15)};

    ON_CALL(*fixture.roadPhotoToEdgeMock, lookupByFeatureId(13))
        .WillByDefault(::testing::Return(roadRelations));
    ON_CALL(*fixture.roadPhotoToEdgeMock, lookupByFeatureId(14))
        .WillByDefault(::testing::Return(roadRelations));
    ON_CALL(*fixture.pedestrianPhotoToEdgeMock, lookupByFeatureId(15))
        .WillByDefault(::testing::Return(pedestrianRelations));

    auto blackboxClient = makeNiceMockBlackboxClient();
    EXPECT_CALL(*blackboxClient, uidToUserInfoMap(Eq(Uids{123}), _))
        .WillOnce(Return(std::unordered_map<Uid, UserInfo>{{123, USER_INFO}}));
    auto prevBlackboxClient = Configuration::instance()->swapBlackboxClient(
        std::move(blackboxClient));
    auto onExit = concurrent::ScopedGuard{[&] {
        Configuration::instance()->swapBlackboxClient(
            std::move(prevBlackboxClient));
    }};

    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_similar.response.schema.json");
    const std::string RESPONSE_PATH = "similar.json";

    const struct {
        db::TId featureId;
        size_t featuresNumber;
    } TESTS[] = {
        {13, 2u},
        {14, 2u},
        {15, 1u},
    };

    for (const auto& test : TESTS) {

    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/" +
                      std::to_string(test.featureId) + "/similar")
                .addParam("with_authors", "true"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);

    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);

    auto json = json::Value::fromString(resp.body);
    EXPECT_EQ(json["data"]["features"].size(), test.featuresNumber);
    for (const auto& feature : json["data"]["features"]) {
        EXPECT_TRUE(feature.hasField("author"));
        const auto& author = feature["author"];
        const auto& featureId = feature["properties"]["id"].as<std::string>();

        if (featureId == "15") {
            EXPECT_TRUE(isExpectedAuthor(author, USER_INFO));
        } else {
            EXPECT_TRUE(isExpectedAuthor(author, YANDEX_USER_INFO));
        }
    }

    }
}

Y_UNIT_TEST(test_near_request_with_authors)
{
    FixtureFb fixture;

    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_near.response.schema.json");
    const std::string RESPONSE_PATH = "near.json";

    http::MockRequest rq(http::GET,
            http::URL("http://localhost/features/near?ll=43.9992,56.3218")
                .addParam("with_authors", "true"));
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);

    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);

    auto json = json::Value::fromString(resp.body);

    for (const auto& feature : json["data"]["features"]) {
        EXPECT_TRUE(feature.hasField("author"));
        EXPECT_TRUE(isExpectedAuthor(feature["author"], YANDEX_USER_INFO));
    }
}

Y_UNIT_TEST(test_tour_along_route)
{
    FixtureFb fixture;

    const std::string SCHEMA_PATH =
        apiResponseSchemaPath("browser/get_tour_along_route.response.schema.json");
    const std::string RESPONSE_PATH = "tour_along_route.json";

    http::MockRequest rq(http::POST,
        http::URL("http://localhost/features/tour_along_route")
            .addParam("with_authors", "true")
            .addParam("graph_type", "pedestrian")
    );
    rq.headers.emplace(yacare::tests::USER_ID_HEADER, std::to_string(fixture.UID));

    rq.body = R"({"geometry": [[37.591518, 55.731716], [37.591172, 55.731579], [37.590729, 55.731702]]})";
    auto resp = yacare::performTestRequest(rq);
    EXPECT_EQ(resp.status, 200);

    maps::common::writeFile(RESPONSE_PATH, resp.body);
    validateJson(RESPONSE_PATH, SCHEMA_PATH);

    auto json = json::Value::fromString(resp.body);
    auto features = json["data"]["features"];
    EXPECT_FALSE(features.empty());
    for (const auto& feature : features) {
        EXPECT_TRUE(feature.hasField("author"));
        EXPECT_TRUE(isExpectedAuthor(feature["author"], YANDEX_USER_INFO));
    }
}

} // Y_UNIT_TEST_SUITE(feature_api_author_should)

} // namespace maps::mrc::browser::tests
