#include "common.h"
#include <maps/wikimap/mapspro/services/mrc/browser/tests/fixture.h>
#include <maps/wikimap/mapspro/services/mrc/browser/lib/permission_checker.h>

#include <maps/libs/http/include/test_utils.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/std.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/gmock_in_unittest/gmock.h>
#include <sys/types.h>

#include <thread>

namespace maps::mrc::browser::tests {

namespace {

const std::string HOST = "wiki-acl.yandex.ru";
const std::string TVM_SERVICE_TICKET = "12345";
const int64_t UID = 21;

const std::string RESTRICTED_ACCESS_PATH = "mpro/social/feedback/task/hidden";
const std::string SECRET_ACCESS_PATH = "mpro/mrc/photos/view-hidden";

std::string ticketProvider() {
    return TVM_SERVICE_TICKET;
}

} // namespace


std::string makeCheckPermissionUrl(int64_t uid)
{
    return "http://" + HOST + "/users/"
        + std::to_string(uid) + "/check-permissions";
}


auto makeWikiAclMock(int64_t uid, const std::vector<std::string>& allowedPaths)
{
    return http::addMock(makeCheckPermissionUrl(uid), [=](const http::MockRequest& request) {
        EXPECT_EQ(
            request.headers.at(maps::auth::SERVICE_TICKET_HEADER),
            TVM_SERVICE_TICKET);
        auto pathParam = request.url.optParam("paths");
        EXPECT_TRUE(pathParam.has_value());
        auto pathJson = json::Value::fromString(pathParam.value());
        std::vector<std::string> paths(pathJson);
        EXPECT_THAT(
            paths,
            ::testing::UnorderedElementsAre(
                RESTRICTED_ACCESS_PATH,
                SECRET_ACCESS_PATH
                ));

        json::Builder builder;
        builder << [&](json::ObjectBuilder builder) {
            builder["paths"] << allowedPaths;
        };

        return http::MockResponse(builder.str());
    });
}

auto makeFailingWikiAclMock(int64_t uid)
{
    return http::addMock(makeCheckPermissionUrl(uid),
        [=](const http::MockRequest& ) {
            http::MockResponse response;
            response.status = 500;
            return response;
    });
}

Y_UNIT_TEST_SUITE(permission_checker_should) {


Y_UNIT_TEST(basic_test)
{
    WikiAclPermissionChecker checker(HOST, ticketProvider, 10, std::chrono::seconds(100));

    auto mock = makeWikiAclMock(UID, {RESTRICTED_ACCESS_PATH, SECRET_ACCESS_PATH});

    EXPECT_EQ(
            checker.userMayViewPhotos(UID, db::FeaturePrivacy::Public), true);
    EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Restricted), true);
    EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Secret), true);
}

Y_UNIT_TEST(public_photos_are_checked_without_network_request)
{
    WikiAclPermissionChecker checker(HOST, ticketProvider, 10, std::chrono::seconds(100));
    auto mock = makeFailingWikiAclMock(UID);

    EXPECT_EQ(
            checker.userMayViewPhotos(UID, db::FeaturePrivacy::Public), true);
}

Y_UNIT_TEST(temporary_error_case)
{
    WikiAclPermissionChecker checker(HOST, ticketProvider, 10, std::chrono::seconds(100));
    {
        auto mock = makeFailingWikiAclMock(UID);

        EXPECT_EQ(
                checker.userMayViewPhotos(UID, db::FeaturePrivacy::Public), true);

        // Could not check permissions -- treat it as no permissions
        EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Restricted), false);
        EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Secret), false);
    }

    {
        auto mock = makeWikiAclMock(UID, {RESTRICTED_ACCESS_PATH, SECRET_ACCESS_PATH});
        EXPECT_EQ(
                checker.userMayViewPhotos(UID, db::FeaturePrivacy::Public), true);
        EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Restricted), true);
        EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Secret), true);
    }
}

Y_UNIT_TEST(check_caching_limiting_by_size)
{
    const int64_t UID2 = UID + 1;
    WikiAclPermissionChecker checker(HOST, ticketProvider, 1, std::chrono::seconds(100));

    {
        auto mock1 = makeWikiAclMock(UID, {RESTRICTED_ACCESS_PATH, SECRET_ACCESS_PATH});
        auto mock2 = makeWikiAclMock(UID2, {RESTRICTED_ACCESS_PATH, SECRET_ACCESS_PATH});
        EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Restricted), true);
        // After this call previous cached result should be eliminated from the cache
        EXPECT_EQ(checker.userMayViewPhotos(UID2, db::FeaturePrivacy::Secret), true);
    }

    {
        auto mock = makeFailingWikiAclMock(UID);
        EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Restricted), false);
    }
}


Y_UNIT_TEST(check_caching_limiting_by_time)
{
    WikiAclPermissionChecker checker(HOST, ticketProvider, 10, std::chrono::seconds(1));

    {
        auto mock = makeWikiAclMock(UID, {RESTRICTED_ACCESS_PATH, SECRET_ACCESS_PATH});
        EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Restricted), true);
    }
    std::this_thread::sleep_for(std::chrono::seconds(2));

    {
        auto mock = makeFailingWikiAclMock(UID);
        EXPECT_EQ(checker.userMayViewPhotos(UID, db::FeaturePrivacy::Restricted), false);
    }
}



} // Y_UNIT_TEST_SUITE(PermissionCheckerTests)

} // namespace maps::mrc::browser::tests
