import json
import logging
import hashlib
import requests
import os

from firmware_signature import get_firmware_signature

BASE_URL = 'http://core-firmware-updater.maps.yandex.net/firmware/1.x/target_state'
SIGNATURE_FILE = '/etc/yandex/firmware_signature.key'

logger = logging.getLogger('mrc-drive-updater')


class Firmware:
    def __init__(self, version, url, md5sum):
        self.version = version
        self.url = url
        self.md5sum = md5sum

    def __str__(self):
        return 'Firmware(version={}, url={}, md5={})'.format(self.version, self.url, self.md5sum)


def request_updates(base_url, user_agent, hardware_name, device_id, slot, signature_key):
    url = base_url + '?hardware={}&deviceid={}&slots={}'.format(hardware_name, device_id, slot)

    signature = get_firmware_signature(user_agent, url, signature_key)
    headers = {'X-YFirmware-Signature': signature, 'User-Agent': user_agent}

    logger.debug("Request updates from url {}".format(url))
    response = requests.get(url, headers=headers)

    firmwares = {}
    if response.status_code == 200:
        data = json.loads(response.content.decode('utf-8'))
        logger.debug('Response: {}'.format(data))
        for response_data in data['slots']:
            firmwares[response_data['name']] = Firmware(response_data['version'], response_data['url'], response_data['md5'])
    else:
        logger.warning('Response status code: {}'.format(response.status_code))

    return firmwares


def download_file(url, path, expected_md5_sum):
    response = requests.get(url, stream=True, timeout=180)
    md5_sum = hashlib.md5()
    chunk_size = 10 * 1024
    if response.status_code == 200:
        with open(path, 'wb') as file:
            for chunk in response.iter_content(chunk_size=chunk_size):
                md5_sum.update(chunk)
                file.write(chunk)
        if expected_md5_sum != md5_sum.hexdigest():
            raise RuntimeError('md5 sum is not correct for downloaded file')
    else:
        logger.error('Failed to download url {}. status code: {}'.format(url, response.status_code))
        raise requests.HTTPError(response.content)


def get_firmware_signature_key():
    if not os.path.isfile(SIGNATURE_FILE):
        raise RuntimeError('No signature file: {}'.format(SIGNATURE_FILE))
    return open(SIGNATURE_FILE, 'r').readline().strip('\n')


def get_installation_candidate(user_agent, hardware_name, device_id, slot):
    firmwares = request_updates(BASE_URL, user_agent, hardware_name, device_id, slot, get_firmware_signature_key())

    if len(firmwares) > 1:
        logger.warning('More than one firmware update received')

    if len(firmwares) > 0:
        return firmwares[slot]

    return None
