import json
import logging

from utils import run_command, get_command_output

RAUC_BIN = '/usr/bin/rauc'
logger = logging.getLogger('mrc-drive-updater')


class Slot:
    def __init__(self, slot_name, boot_status, state, slotclass, version):
        self.slot_name = slot_name
        self.boot_status = boot_status
        self.state = state
        self.slotclass = slotclass
        self.version = version


def parse_slot_info(slot_name, slot_info):
    version = None
    slot_status = slot_info.get('slot_status')
    if slot_status:
        bundle = slot_status.get('bundle')
        if bundle:
            version = bundle.get("version")

    return Slot(slot_name,
                slot_info.get('boot_status'),
                slot_info.get('state'),
                slot_info.get('class'),
                version)


def parse_rauc_slots(rauc_status_json):
    slots = []
    for slot in rauc_status_json['slots']:
        for (name, info) in slot.items():
            slots.append(parse_slot_info(name, info))
    return slots


def should_download_bundle(candidate_version, rauc_slots):
    for slot in rauc_slots:
        if slot.version == candidate_version:
            return False

    return True


def get_target_slot(candidate_version, rauc_slots):
    for slot in rauc_slots:
        if slot.version == candidate_version and slot.boot_status == 'good':
            return slot

    return None


def get_rauc_slots():
    stdout = get_command_output([RAUC_BIN, 'status', '--detailed', '--output-format=json'])
    return parse_rauc_slots(json.loads(stdout))


def get_booted_slot(rauc_slots):
    for slot in rauc_slots:
        if slot.state == 'booted':
            return slot
    raise RuntimeError('No booted slots')


def rauc_install(file_path):
    logger.info('Install firmware from {}'.format(file_path))
    return run_command([RAUC_BIN, 'install', file_path])


def switch_to_slot(slot):
    logger.info('Switch to slot {}'.format(slot.slot_name))
    return run_command([RAUC_BIN, 'status', 'mark-active', slot.slot_name])
