#pragma once

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/libs/vault_boy/include/secrets.h>
#include <maps/libs/xml/include/xml.h>

#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/pgpool3_helpers.h>

namespace maps::fw_updater::config {

using SharedXmlDoc = std::shared_ptr<xml3::Doc>;

class S3Config {
public:
    explicit S3Config(const xml3::Node& node);

    const std::string& host() const { return host_; }
    const std::string& publicReadHost() const { return publicReadHost_; }
    size_t connectTimeoutMs() const { return connectTimeoutMs_; }
    size_t requestTimeoutMs() const { return requestTimeoutMs_; }

    const std::string& bucket() const { return bucket_; }
    const std::string& accessKeyId() const { return accessKeyId_; }
    const std::string& accessKeySecret() const { return accessKeySecret_; }

private:
    std::string host_;
    std::string publicReadHost_;
    size_t connectTimeoutMs_;
    size_t requestTimeoutMs_;
    std::string bucket_;
    std::string accessKeyId_;
    std::string accessKeySecret_;
};

class StorageConfig {
public:
    struct Cli {
        explicit Cli(const xml3::Node& node);
        const std::string& oauthAppClientId() const { return oauthAppClientId_; }
        const std::string& oauthAppClientSecret() const { return oauthAppClientSecret_; }
    private:
        std::string oauthAppClientId_;
        std::string oauthAppClientSecret_;
    };

    explicit StorageConfig(const xml3::Node& node);
    const std::string& url() const { return url_; }
    const Cli& cli() const { return cli_; }
private:
    std::string url_;
    Cli cli_;
};

class UpdaterConfig {
public:
    explicit UpdaterConfig(const xml3::Node& node);
    const std::string& signatureKey() const { return signatureKey_; }
private:
    std::string signatureKey_;
};

class Config {
public:
    /// Reads config from file.
    explicit Config(const std::string& path);

    /// Reads config template from embedded resource and renders it.
    explicit Config(const vault_boy::Context& ctx);

    /// Reads config template from file and render it.
    explicit Config(const vault_boy::Context& ctx, const std::string& path);

    /// Constructs config from the given xml document.
    Config(SharedXmlDoc xml);

    /// Constructs config from the given string.
    static Config fromString(const std::string& config);

    std::string toString() const;

    wiki::common::PoolHolder makePoolHolder() const;

    const S3Config& s3() const { return s3_; }

    const StorageConfig& storage() const { return storage_; }

    const UpdaterConfig& updater() const { return updater_; }

private:
    SharedXmlDoc xml_;
    S3Config s3_;
    StorageConfig storage_;
    UpdaterConfig updater_;
};

/// Default directory with secrets for config templates.
/// `/etc/yandex/maps/maps-fw-updater-config/keys`
const std::string KEYS_PATH = vault_boy::defaultPath("maps-fw-updater-config");

inline Config templateConfigFromCmdPath(const cmdline::Option<std::string>& configPath) {
    vault_boy::DirectoryContext ctx{KEYS_PATH};
    if (configPath.defined()) {
        return Config{ctx, configPath};
    }
    return Config{ctx};
}

inline Config templateConfigFromCmdPath(const cmdline::Option<std::string>& secretVersion,
                                        const cmdline::Option<std::string>& configPath) {
    return secretVersion.defined()
        ? Config{maps::vault_boy::loadContextWithYaVault(secretVersion),
                configPath}
        : templateConfigFromCmdPath(configPath);
}

inline Config templateConfig() {
    return Config{vault_boy::DirectoryContext{KEYS_PATH}};
}

} // namespace maps::fw_updater::config
