#include "fixture.h"

#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/project_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/role_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/project_role_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/user_role_gateway.h>

#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/introspection/include/stream_output.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/unittest/env.h>

#include <algorithm>

namespace maps::fw_updater::db::idm {
using introspection::operator==;
using introspection::operator!=;
using introspection::operator<<;
} // namespace maps::fw_updater::db

namespace maps::fw_updater::db::idm::tests {

namespace {

const std::string MRC_PROJECT_KEY = "mrc-drive-nanopi-neo4";
const std::string AUTO_PROJECT_KEY = "auto-tesla-model-s";
const std::string ROLE_DEVELOPER = "developer";
const std::string ROLE_MANAGER = "manager";

using Fixture = maps::fw_updater::db::tests::Fixture;

} // namespace

TEST_F(Fixture, test_idm_projects_hierarchy)
{
    auto txn = txnHandle();
    ProjectGateway gtw{*txn};

    auto projects = gtw.load();
    EXPECT_EQ(projects.size(), 3u);

    auto root = gtw.loadOne(table::Project::key == "_root");
    EXPECT_EQ(root.nameRu(), "Система");
    EXPECT_EQ(root.nameEn(), "System");
    EXPECT_EQ(root.slug(), "projects");
    EXPECT_EQ(root.slugNameRu(), "Проекты");
    EXPECT_EQ(root.slugNameEn(), "Projects");

    auto childProjects = gtw.load(
        table::Project::parentId == root.id(),
        sql_chemistry::orderBy(table::Project::id));
    ASSERT_EQ(childProjects.size(), 2u);
    EXPECT_EQ(childProjects[0].id(), 2);
    EXPECT_EQ(childProjects[0].key(), MRC_PROJECT_KEY);
    EXPECT_EQ(childProjects[1].id(), 3);
    EXPECT_EQ(childProjects[1].key(), AUTO_PROJECT_KEY);
}

TEST_F(Fixture, test_idm_roles)
{
    auto txn = txnHandle();
    auto mrcProjectRoles = ProjectRoleGateway{*txn}.load(
        table::ProjectRole::projectId == 2);
    EXPECT_EQ(mrcProjectRoles.size(), 2u);

    TIds roleIds;
    for (auto& pr : mrcProjectRoles) {
        roleIds.push_back(pr.roleId());
    }
    auto mrcRoles = RoleGateway{*txn}.load(
        table::Role::id.in(roleIds),
        sql_chemistry::orderBy(table::Role::id));
    ASSERT_EQ(mrcRoles.size(), 2u);

    const auto& developer = mrcRoles[0];
    EXPECT_EQ(developer.id(), 1);
    EXPECT_EQ(developer.key(), ROLE_DEVELOPER);
    EXPECT_EQ(developer.roleSet(), ROLE_DEVELOPER);

    const auto& manager = mrcRoles[1];
    EXPECT_EQ(manager.id(), 2);
    EXPECT_EQ(manager.key(), ROLE_MANAGER);
    EXPECT_EQ(manager.roleSet(), std::nullopt);
}

TEST_F(Fixture, test_idm_add_remove_user_roles)
{
    auto txn = txnHandle();
    ProjectGateway projetcGtw{*txn};
    ProjectRoleGateway projetcRoleGtw{*txn};

    auto mrcProject = projetcGtw.loadOne(table::Project::key == MRC_PROJECT_KEY);
    auto autoProject = projetcGtw.loadOne(table::Project::key == AUTO_PROJECT_KEY);
    auto mrcProjectRoles = projetcRoleGtw.load(
        table::ProjectRole::projectId == mrcProject.id());
    Roles allRoles = RoleGateway{*txn}.load();

    std::map<std::string, TId> projectRoleByKey;
    for (const auto& role : allRoles) {
        auto itr = std::find_if(
            mrcProjectRoles.begin(),
            mrcProjectRoles.end(),
            [&](const ProjectRole& pr) { return pr.roleId() == role.id(); });
        if (itr != mrcProjectRoles.end()) {
            projectRoleByKey.emplace(role.key(), itr->id());
        }
    }

    EXPECT_TRUE(projectRoleByKey.find(ROLE_DEVELOPER) != projectRoleByKey.end());
    EXPECT_TRUE(projectRoleByKey.find(ROLE_MANAGER) != projectRoleByKey.end());

    UserRoles mrcUserRoles{
         {"neo", projectRoleByKey[ROLE_DEVELOPER]},
         {"morpheus", projectRoleByKey[ROLE_MANAGER]}
    };
    UserRoleGateway{*txn}.insert(mrcUserRoles);
    txn->commit();

    {
        // Load roles in mrc project
        auto txn = txnHandle();
        auto userRoles = UserRoleGateway{*txn}.load(
            table::UserRole::projectRoleId == table::ProjectRole::id &&
            table::ProjectRole::projectId == mrcProject.id());
        ASSERT_EQ(userRoles.size(), 2u);
        EXPECT_EQ(userRoles, mrcUserRoles);
    }

    // Load project role Ids
    TId mrcDeveloperRoleId{0};
    TId autoDeveloperRoleId{0};
    {
        auto txn = txnHandle();
        auto projectRoles = ProjectRoleGateway{*txn}.load(
            table::ProjectRole::roleId == table::Role::id &&
            table::Role::key == ROLE_DEVELOPER);
        for (const auto& pr : projectRoles) {
            if (pr.projectId() == mrcProject.id()) {
                mrcDeveloperRoleId = pr.id();
            } else if (pr.projectId() == autoProject.id()) {
                autoDeveloperRoleId = pr.id();
            }
        }
    }

    {
        // Revoke developer role
        auto txn = txnHandle();
        UserRoleGateway{*txn}.remove(
            table::UserRole::login == "neo" &&
            table::UserRole::projectRoleId == mrcDeveloperRoleId);
        txn->commit();
    }

    {
        // Load roles in mrc project
        auto txn = txnHandle();
        auto userRoles = UserRoleGateway{*txn}.load(
            table::UserRole::projectRoleId == table::ProjectRole::id &&
            table::ProjectRole::projectId == mrcProject.id());
        ASSERT_EQ(userRoles.size(), 1u);
        EXPECT_EQ(userRoles[0], mrcUserRoles[1]);
    }

    UserRoles autoUserRoles{
        {"neo", autoDeveloperRoleId}
    };

    {
        // Add developer role in Auto project
        auto txn = txnHandle();
        UserRoleGateway{*txn}.insert(autoUserRoles);
        txn->commit();
    }

    {
        // Check roles
        auto txn = txnHandle();

        auto userRoles = UserRoleGateway{*txn}.load(
            table::UserRole::projectRoleId == table::ProjectRole::id &&
            table::ProjectRole::projectId == mrcProject.id());
        ASSERT_EQ(userRoles.size(), 1u);
        EXPECT_EQ(userRoles[0], mrcUserRoles[1]);

        userRoles = UserRoleGateway{*txn}.load(
            table::UserRole::projectRoleId == table::ProjectRole::id &&
            table::ProjectRole::projectId == autoProject.id());
       ASSERT_EQ(userRoles.size(), 1u);
       EXPECT_EQ(userRoles[0], autoUserRoles[0]);
    }
}

} // namespace maps::fw_updater::db::idm::tests
