#include "fixture.h"

#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/branch_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/firmware_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/rollout_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/rollout_history_gateway.h>

#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/introspection/include/stream_output.h>

#include <library/cpp/testing/gtest/gtest.h>

namespace maps::fw_updater::db {
using introspection::operator==;
using introspection::operator!=;
using introspection::operator<<;
} // namespace maps::fw_updater::db


namespace maps::fw_updater::db::tests {

namespace {

const std::string HARDWARE_1 = "nanopi-neo4";
const std::string HARDWARE_2 = "raspberry-pi-3b";

const std::string STABLE = "stable";
const std::string TESTING = "testing";

const std::string SEED = "seed";
const std::string USER = "login";

Firmware createFirmware(const std::string& hardwareId,
                        Slot slot,
                        const std::string& version)
{
    return Firmware(hardwareId,
                    slot,
                    version,
                    "s3.yandex.ru/key",
                    195716838,
                    "22c14bc29040b2b2e0a3e9ed02d99eeb");
}

} // namespace


TEST_F(Fixture, test_rollout)
{
    auto txn = txnHandle();

    Branches branches{{STABLE}, {TESTING}};
    BranchGateway{*txn}.insert(branches);

    Firmwares firmwares {
        createFirmware(HARDWARE_1, Slot::Rootfs, "1.0.0"),
        createFirmware(HARDWARE_1, Slot::Rootfs, "2.0.0"),
        createFirmware(HARDWARE_2, Slot::Rootfs, "1.0.0")
    };
    FirmwareGateway{*txn}.insert(firmwares);

    Rollouts rollouts {
        Rollout(STABLE, firmwares[0].id(), SEED),
        Rollout(TESTING, firmwares[1].id(), SEED),
        Rollout(TESTING, firmwares[2].id(), SEED)
    };
    RolloutGateway{*txn}.insert(rollouts);
    txn->commit();

    // Load
    {
        auto loadedRollout = RolloutGateway{*txnHandle()}.load(
            table::Rollout::branch == TESTING);
        ASSERT_EQ(loadedRollout.size(), 2u);
        EXPECT_EQ(loadedRollout[0], rollouts[1]);
        EXPECT_EQ(loadedRollout[1], rollouts[2]);
    }
}

TEST_F(Fixture, test_rollout_constraints)
{
    Firmwares firmwares;
    {
        auto txn = txnHandle();

        Branches branches{{STABLE}, {TESTING}};
        BranchGateway{*txn}.insert(branches);

        firmwares.push_back(createFirmware(HARDWARE_1, Slot::Rootfs, "1.0.0"));
        firmwares.push_back(createFirmware(HARDWARE_2, Slot::Rootfs, "1.0.0"));
        firmwares.push_back(createFirmware(HARDWARE_2, Slot::Rootfs, "2.0.0"));
        FirmwareGateway{*txn}.insert(firmwares);

        Rollout rollout(TESTING, firmwares[2].id(), SEED);
        RolloutGateway{*txn}.insert(rollout);
        txn->commit();
    }

    // Can't downgrade version_seq of firmware id for the same slot/branch combination
    {
        auto txn = txnHandle();
        Rollout rollout(TESTING, firmwares[1].id(), SEED);
        EXPECT_THROW(RolloutGateway{*txnHandle()}.insert(rollout),
                     pqxx::sql_error);

    }

    // For a different branch it is Ok to create rollout with smaller version_seq
    Rollout rollout(STABLE, firmwares[1].id(), SEED);
    {
        auto txn = txnHandle();
        RolloutGateway{*txnHandle()}.insert(rollout);
        txn->commit();
    }

    // Can't update rollout entry
    EXPECT_THROW(RolloutGateway{*txnHandle()}.update(rollout),
                 sql_chemistry::EditConflict);
}


class RolloutHistoryFixture : public Fixture
{
public:
    RolloutHistory addRolloutHistory(TId rolloutId, uint16_t percent, RolloutStatus status)
    {
        auto txn = txnHandle();
        RolloutHistory rolloutHistory(rolloutId, percent, status, USER);
        RolloutHistoryGateway{*txn}.insert(rolloutHistory);
        txn->commit();

        return rolloutHistory;
    }

    void checkRolloutHistories(size_t expectedTotalSize, const RolloutHistory& latestEntry)
    {
        auto entries = RolloutHistoryGateway{*txnHandle()}.load(
            sql_chemistry::orderBy(table::RolloutHistory::id));
        ASSERT_EQ(entries.size(), expectedTotalSize);
        EXPECT_EQ(entries[expectedTotalSize - 1], latestEntry);
    }
};


TEST_F(RolloutHistoryFixture, test_rollout_history_increase_percent)
{
    TId rolloutId = 0;

    auto txn = txnHandle();

    Branch branch{TESTING};
    BranchGateway{*txn}.insert(branch);

    Firmware firmware = createFirmware(HARDWARE_1, Slot::Rootfs, "1.0.0");
    FirmwareGateway{*txn}.insert(firmware);

    Rollout rollout(TESTING, firmware.id(), SEED);
    RolloutGateway{*txn}.insert(rollout);
    txn->commit();

    rolloutId = rollout.id();

    // Add initial experiment for 10%
    auto rolloutHistory = addRolloutHistory(rolloutId, 10, RolloutStatus::Active);
    checkRolloutHistories(1, rolloutHistory);
    EXPECT_TRUE(rolloutHistory.isExperimental());

    // Expand experiment to 15%
    rolloutHistory = addRolloutHistory(rolloutId, 15, RolloutStatus::Active);
    checkRolloutHistories(2, rolloutHistory);
    EXPECT_TRUE(rolloutHistory.isExperimental());

    // Expand experiment to 100%
    rolloutHistory = addRolloutHistory(rolloutId, 100, RolloutStatus::Active);
    checkRolloutHistories(3, rolloutHistory);
    EXPECT_FALSE(rolloutHistory.isExperimental());
}

TEST_F(RolloutHistoryFixture, test_rollout_history_stop_experiment)
{
    auto txn = txnHandle();

    Branch branch{TESTING};
    BranchGateway{*txn}.insert(branch);

    Firmwares firmwares {
        createFirmware(HARDWARE_1, Slot::Rootfs, "1.0.0"),
        createFirmware(HARDWARE_1, Slot::Rootfs, "2.0.0"),
        createFirmware(HARDWARE_1, Slot::Rootfs, "3.0.0")
    };
    FirmwareGateway{*txn}.insert(firmwares);

    Rollouts rollouts {
        {TESTING, firmwares[0].id(), SEED},
        {TESTING, firmwares[1].id(), SEED},
        {TESTING, firmwares[2].id(), SEED}
    };
    RolloutGateway{*txn}.insert(rollouts);

    txn->commit();

    // Roll-out initial version to 100%
    auto rolloutHistory = addRolloutHistory(rollouts[0].id(), 100, RolloutStatus::Active);
    checkRolloutHistories(1, rolloutHistory);

    // Add experiment to 10%
    rolloutHistory = addRolloutHistory(rollouts[1].id(), 10, RolloutStatus::Active);
    checkRolloutHistories(2, rolloutHistory);

    // Stop experiment
    rolloutHistory = addRolloutHistory(rollouts[1].id(), 10, RolloutStatus::Inactive);
    checkRolloutHistories(3, rolloutHistory);

    // Add another experiment
    rolloutHistory = addRolloutHistory(rollouts[2].id(), 5, RolloutStatus::Active);
    checkRolloutHistories(4, rolloutHistory);

    // Expand to 100%
    rolloutHistory = addRolloutHistory(rollouts[2].id(), 100, RolloutStatus::Active);
    checkRolloutHistories(5, rolloutHistory);
}

TEST_F(RolloutHistoryFixture, test_rollout_history_constraints)
{
    auto txn = txnHandle();

    Branch branch{TESTING};
    BranchGateway{*txn}.insert(branch);

    Firmwares firmwares {
        createFirmware(HARDWARE_1, Slot::Rootfs, "1.0.0"),
        createFirmware(HARDWARE_1, Slot::Rootfs, "2.0.0"),
        createFirmware(HARDWARE_1, Slot::Rootfs, "3.0.0")
    };
    FirmwareGateway{*txn}.insert(firmwares);

    Rollouts rollouts {
        {TESTING, firmwares[0].id(), SEED},
        {TESTING, firmwares[1].id(), SEED},
        {TESTING, firmwares[2].id(), SEED}
    };
    RolloutGateway{*txn}.insert(rollouts);

    txn->commit();

    // Add experiment to 10%
    auto rolloutHistory = addRolloutHistory(rollouts[0].id(), 10, RolloutStatus::Active);
    checkRolloutHistories(1, rolloutHistory);

    // Can't decrease experiment percent
    RolloutHistory rh1(rollouts[0].id(), 5, RolloutStatus::Active, USER);
    EXPECT_THROW(RolloutHistoryGateway{*txnHandle()}.insert(rh1),
                 pqxx::sql_error);

    // Can't have more than one active experiment
    RolloutHistory rh2(rollouts[1].id(), 10, RolloutStatus::Active, USER);
    EXPECT_THROW(RolloutHistoryGateway{*txnHandle()}.insert(rh2),
                 pqxx::sql_error);

    // Expand rollout0 to 100%
    rolloutHistory = addRolloutHistory(rollouts[0].id(), 100, RolloutStatus::Active);
    checkRolloutHistories(2, rolloutHistory);

    // Now can create a new experiment
    rolloutHistory = addRolloutHistory(rollouts[2].id(), 10, RolloutStatus::Active);
    checkRolloutHistories(3, rolloutHistory);

    // Expand rollout2 to 100%
    rolloutHistory = addRolloutHistory(rollouts[2].id(), 100, RolloutStatus::Active);
    checkRolloutHistories(4, rolloutHistory);

    // Can't restore rollout1 from the past
    RolloutHistory rh3(rollouts[1].id(), 10, RolloutStatus::Active, USER);
    EXPECT_THROW(RolloutHistoryGateway{*txnHandle()}.insert(rh3),
                 pqxx::sql_error);
}

TEST_F(RolloutHistoryFixture, test_rollout_history_with_multiple_hw_and_branches)
{
    Rollouts rollouts;
    {
        auto txn = txnHandle();

        Branches branches{{TESTING}, {STABLE}};
        BranchGateway{*txn}.insert(branches);

        Firmwares firmwares {
            createFirmware(HARDWARE_1, Slot::Rootfs, "1.0.0"),
            createFirmware(HARDWARE_1, Slot::Rootfs, "2.0.0"),
            createFirmware(HARDWARE_2, Slot::Rootfs, "1.0.0"),
        };
        FirmwareGateway{*txn}.insert(firmwares);

        rollouts.emplace_back(TESTING, firmwares[0].id(), SEED);
        rollouts.emplace_back(TESTING, firmwares[1].id(), SEED);
        rollouts.emplace_back(TESTING, firmwares[2].id(), SEED);
        rollouts.emplace_back(STABLE, firmwares[0].id(), SEED);
        RolloutGateway{*txn}.insert(rollouts);

        txn->commit();
    }

    auto& hw1TestingRolloutVer1 = rollouts[0];
    auto& hw1TestingRolloutVer2 = rollouts[1];
    auto& hw2TestingRolloutVer1 = rollouts[2];
    auto& hw1StableRolloutVer1 = rollouts[3];

    // Can have simultaneous experiments for different branches or hardwares
    addRolloutHistory(hw1TestingRolloutVer1.id(), 10, RolloutStatus::Active);
    addRolloutHistory(hw1StableRolloutVer1.id(), 10, RolloutStatus::Active);
    addRolloutHistory(hw2TestingRolloutVer1.id(), 10, RolloutStatus::Active);

    // Inactivate experiment for hw1TestingRolloutVer1
    addRolloutHistory(hw1TestingRolloutVer1.id(), 10, RolloutStatus::Inactive);

    // Activate new experiments in testing and stable
    addRolloutHistory(hw1TestingRolloutVer2.id(), 10, RolloutStatus::Active);
    addRolloutHistory(hw1StableRolloutVer1.id(), 20, RolloutStatus::Active);

    // Inactivate experiment for hw2
    addRolloutHistory(hw2TestingRolloutVer1.id(), 10, RolloutStatus::Inactive);

    // Activate same experiment for hw2 again
    addRolloutHistory(hw2TestingRolloutVer1.id(), 10, RolloutStatus::Active);
}

} // namespace maps::fw_updater::db::tests
