CREATE SCHEMA firmware;

CREATE TYPE firmware.slot_t AS ENUM (
    'rootfs'
);

CREATE TYPE firmware.rollout_status_t AS ENUM (
    'inactive',
    'active'
);

CREATE TABLE firmware.hardware (
    id text PRIMARY KEY,
    idm_project_id bigint NOT NULL REFERENCES idm.project(id),
    attrs jsonb
);

CREATE TABLE firmware.firmware_upload (
    id bigserial PRIMARY KEY,
    hardware_id text NOT NULL REFERENCES firmware.hardware(id),
    slot firmware.slot_t NOT NULL,
    version text NOT NULL,
    created_at timestamp NOT NULL,
    created_by text NOT NULL
);

CREATE TABLE firmware.firmware_upload_part (
    id bigint NOT NULL,
    upload_id bigint REFERENCES firmware.firmware_upload(id),
    content_size bigint NOT NULL CHECK (content_size > 0),
    content_md5 text NOT NULL,

    PRIMARY KEY(id, upload_id)
);

CREATE TABLE firmware.branch (
    name text PRIMARY KEY
);

CREATE TABLE firmware.firmware (
    id bigserial NOT NULL PRIMARY KEY,
    hardware_id text NOT NULL REFERENCES firmware.hardware(id),
    slot firmware.slot_t NOT NULL,
    version text NOT NULL,
    version_seq bigserial NOT NULL,
    url text NOT NULL,
    size bigint NOT NULL CHECK (size > 0),
    md5 text NOT NULL,

    UNIQUE(hardware_id, slot, version),
    UNIQUE(hardware_id, slot, version_seq)
);

CREATE TABLE firmware.rollout(
    id bigserial PRIMARY KEY,
    branch text NOT NULL REFERENCES firmware.branch(name),
    firmware_id bigint NOT NULL REFERENCES firmware.firmware(id),
    seed text NOT NULL,

    UNIQUE(branch, firmware_id)
);

CREATE TABLE firmware.rollout_history(
    id bigserial PRIMARY KEY,
    rollout_id bigint NOT NULL REFERENCES firmware.rollout(id),
    percent smallint NOT NULL CHECK (percent >= 0 AND percent <= 100),
    status firmware.rollout_status_t NOT NULL,
    created_at timestamp NOT NULL,
    created_by text NOT NULL
);

CREATE TABLE firmware.device_branch(
    hardware_id text NOT NULL REFERENCES firmware.hardware(id),
    device_id text NOT NULL,
    branch text NOT NULL REFERENCES firmware.branch(name),

    PRIMARY KEY(hardware_id, device_id)
);


CREATE OR REPLACE FUNCTION enforce_rollout_constraints() RETURNS TRIGGER AS $$
DECLARE
    hardware_id_ text;
    slot_ firmware.slot_t;
    version_seq_ bigint;
    max_version_seq bigint;
BEGIN
    IF (TG_OP = 'UPDATE') THEN
        RAISE EXCEPTION 'rollout entries are immutable';
    END IF;

    SELECT hardware_id, slot, version_seq
        INTO hardware_id_, slot_, version_seq_
        FROM firmware.firmware
        WHERE id = NEW.firmware_id;

    max_version_seq := (
        SELECT max(version_seq)
        FROM firmware.firmware f
        JOIN firmware.rollout r ON r.firmware_id = f.id
        WHERE f.hardware_id = hardware_id_
            AND f.slot = slot_
            AND r.branch = NEW.branch
    );

    IF max_version_seq IS NOT NULL AND version_seq_ <= max_version_seq THEN
        RAISE EXCEPTION 'cannot create rollout with downgraded firmware version_seq';
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER enforce_rollout_constraints_trigger
BEFORE INSERT OR UPDATE ON firmware.rollout
FOR EACH ROW EXECUTE PROCEDURE enforce_rollout_constraints();


CREATE OR REPLACE FUNCTION enforce_rollout_history_constraints() RETURNS TRIGGER AS $$
DECLARE
    hardware_id_ text;
    slot_ firmware.slot_t;
    branch_ text;
    max_active_percent smallint;
    latest_status firmware.rollout_status_t;
    max_active_rollout_id bigint;
BEGIN
    IF (TG_OP = 'UPDATE') THEN
        RAISE EXCEPTION 'rollout_history entries are immutable';
    END IF;

    SELECT hardware_id, slot, branch
        INTO hardware_id_, slot_, branch_
        FROM firmware.rollout r
        JOIN firmware.firmware f ON f.id = r.firmware_id
        WHERE r.id = NEW.rollout_id;

    -- Find greatest rollout_id which have ever been active,
    -- Then for this rollout_id find the most recent entry. Note that this entry
    -- has the maximum percent since percent may only increase.
    SELECT rollout_id, percent, status
        INTO max_active_rollout_id, max_active_percent, latest_status
        FROM firmware.rollout_history
        WHERE rollout_id = (
            SELECT max(rollout_id) FROM firmware.rollout_history rh
            JOIN firmware.rollout r ON r.id = rh.rollout_id
            JOIN firmware.firmware f ON f.id = r.firmware_id
            WHERE f.hardware_id = hardware_id_
                AND f.slot = slot_
                AND r.branch = branch_
                AND rh.status = 'active'
        )
        ORDER BY id DESC LIMIT 1;

    IF NEW.rollout_id < max_active_rollout_id THEN
        RAISE EXCEPTION 'cannot restore a rollout from the past';
    END IF;

    IF NEW.rollout_id > max_active_rollout_id AND max_active_percent < 100 AND latest_status = 'active' THEN
        RAISE EXCEPTION 'cannot have more than one active experiment for the same configuration';
    END IF;

    IF NEW.rollout_id = max_active_rollout_id AND NEW.percent < max_active_percent THEN
        RAISE EXCEPTION 'cannot decrease experiment percent';
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER enforce_rollout_history_constraints_trigger
BEFORE INSERT OR UPDATE ON firmware.rollout_history
FOR EACH ROW EXECUTE PROCEDURE enforce_rollout_history_constraints();

