#include "context.h"

#include <maps/libs/log8/include/log8.h>
#include <library/cpp/oauth/oauth.h>

#include <sstream>

namespace maps::fw_updater::storage::cli {

namespace {

std::string getOauthToken(const config::StorageConfig::Cli& cliConfig)
{
    TString id = cliConfig.oauthAppClientId().data();
    TString secret = cliConfig.oauthAppClientSecret().data();
    TOauthTokenParams params{id, secret};
    try {
        return GetOauthToken(params);
    } catch (const std::exception& e) {
        ERROR() << "Failed to get Oauth token: " << e.what();
        throw;
    }
}

} // namespace

Context::Context(const config::Config& config)
    : baseUrl_(config.storage().url())
    , oauthHeader_("OAuth " + getOauthToken(config.storage().cli()))
{
    httpClient_.setTimeout(std::chrono::minutes(30))
               .setConnectTimeout(std::chrono::seconds(30));

    retryPolicy_.setInitialCooldown(std::chrono::seconds(1))
        .setTryNumber(3)
        .setCooldownBackoff(2);
}

std::function<bool(const Expected<http::Response>&)> Context::responseValidator() const
{
    static const auto validator = [](const auto& maybeResponse) {
        return maybeResponse.valid()
            && maybeResponse.get().responseClass() != http::ResponseClass::ServerError;
    };
    return validator;
}

std::function<std::string(const http::Response&)> Context::errorReporter() const
{
    static const auto reporter = [](const http::Response& response) {
        return "Unexpected response status: " + std::to_string(response.status());
    };
    return reporter;
}

void addAuth(http::Request& request, const Context& ctx)
{
    request.addHeader("Authorization", ctx.oauthHeader());
}

std::string errorMessage(http::Response& response)
{
    std::ostringstream out;
    out << "Unexpected response status: " << response.status()
        << ", message: `" << response.readBody() << "`";
    return out.str();
}

} // namespace maps::fw_updater::storage::cli
