#pragma once

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/enum_io/include/enum_io_fwd.h>
#include <maps/libs/log8/include/log8.h>

namespace maps::fw_updater::storage::cli {

enum class Command {
    Firmware,
    FirmwareUpload,
    Rollout,
    DeviceBranch,
};

DECLARE_ENUM_IO(Command);

enum class FirmwareSubCommand {
    Upload,
    Delete,
    List
};

DECLARE_ENUM_IO(FirmwareSubCommand);

enum class FirmwareUploadSubCommand {
    List,
    Delete
};

DECLARE_ENUM_IO(FirmwareUploadSubCommand);

enum class RolloutSubCommand {
    Create,
    Expand,
    Pause,
    Resume,
    Target,
    History
};

DECLARE_ENUM_IO(RolloutSubCommand);

enum class DeviceBranchSubCommand {
    Add,
    Remove,
    List
};

DECLARE_ENUM_IO(DeviceBranchSubCommand);

/**
 * Convenient wrapper above cmdline::Parser which holds named options
 * specific to the tool and performs validity checks.
 */
struct Options
{
    Options(maps::cmdline::Parser& parser);

    template <typename T>
    struct NamedOption
    {
        std::string name;
        cmdline::Option<T> option;

        T operator()() const { return option; }

        std::optional<T> optional() const {
            std::optional<T> result;
            if (option.defined()) {
                result = option;
            }
            return result;
        }
    };

    void validate() const;

    static std::string helpText(const char* progName);

    Command command() const
    {
        return enum_io::fromString<Command>(parser.argv()[0]);
    }

    template <typename T>
    T subCommand() const
    {
        return enum_io::fromString<T>(parser.argv()[1]);
    }

    maps::cmdline::Parser& parser;
    NamedOption<std::string> branch;
    NamedOption<std::string> deviceId;
    NamedOption<std::string> filePath;
    NamedOption<std::string> hardware;
    NamedOption<uint16_t> percent;
    NamedOption<std::string> rolloutId;
    NamedOption<std::string> slot;
    NamedOption<std::string> uploadId;
    NamedOption<std::string> version;

    NamedOption<uint32_t> results;
    NamedOption<uint32_t> skip;

private:
    template <typename T>
    void requireOption(const NamedOption<T>& namedOption) const
    {
        if (!namedOption.option.defined()) {
            ERROR() << "Missing required option '" << namedOption.name << "'";
            parser.printUsageAndExit();
        }
    }

    template <typename... T>
    void requireOptions(const T&... namedOptions) const
    {
        ( requireOption(namedOptions) , ... );
    }
};

} // namespace maps::fw_updater::storage::cli
