#include "common.h"
#include "configuration.h"
#include "input_stream_adaptor.h"
#include "response.h"
#include "yacare_params.h"

#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/firmware_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/firmware_upload_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/firmware_upload_part_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/hardware_gateway.h>
#include <maps/infra/yacare/include/yacare.h>
#include <maps/infra/yacare/include/params/tvm.h>
#include <maps/infra/yacare/include/options.h>

#include <library/cpp/digest/md5/md5.h>
#include <yandex/maps/proto/firmware_updater/storage/upload.pb.h>

#include <algorithm>

namespace maps::fw_updater::storage {

namespace {

std::string makeKey(
    const std::string& hardware,
    db::Slot slot,
    const std::string& version)
{
    return hardware + "-" + std::string(toString(slot)) + "-" + version;
}

std::string computeMd5(s3::ObjectHolder& objectHolder)
{
    StdInputStreamAdaptor stream(objectHolder.getBody());
    char buf[33] = {0};
    return MD5().Stream(&stream, buf);
}

db::Firmware makeFirmwareFromUpload(
    s3::S3Client& s3Client,
    const db::FirmwareUpload& firmwareUpload,
    size_t contentSize)
{
    auto objectHolder = s3Client.getObject(firmwareUpload.s3Key());

    return db::Firmware(
        firmwareUpload.hardwareId(),
        firmwareUpload.slot(),
        firmwareUpload.version(),
        s3Client.makeReadingUrl(firmwareUpload.s3Key()),
        contentSize,
        computeMd5(objectHolder));
}

std::vector<FirmwareUploadWithParts> combineUploadsWithParts(
    db::FirmwareUploads uploads,
    db::FirmwareUploadParts parts)
{
    std::sort(uploads.begin(), uploads.end(),
        [](const auto& lhs, const auto& rhs) {
            return lhs.id() < rhs.id();
        });

    std::sort(parts.begin(), parts.end(),
        [](const auto& lhs, const auto& rhs) {
            return std::make_tuple(lhs.uploadId(), lhs.id()) <
                   std::make_tuple(rhs.uploadId(), rhs.id());
        });

    std::vector<FirmwareUploadWithParts> uploadsWithParts;
    uploadsWithParts.reserve(uploads.size());
    for (auto& upload : uploads) {
        uploadsWithParts.push_back({std::move(upload), {}});
    }

    auto uploadIt = uploadsWithParts.begin();
    auto partIt = parts.begin();

    while (uploadIt != uploadsWithParts.end()) {
        while (partIt != parts.end() && partIt->uploadId() == uploadIt->upload.id()) {
            uploadIt->parts.push_back(std::move(*partIt));
            ++partIt;
        }
        ++uploadIt;
    }
    ASSERT(partIt == parts.end());
    return uploadsWithParts;
}

} // namespace

YCR_RESPOND_TO("PUT /v2/firmware/upload/start", userInfo, hardware, slot, version)
{
    auto& s3Client = configuration()->s3Client();
    auto txn = configuration()->pool().masterWriteableTransaction();

    auto dbHardware = db::HardwareGateway{*txn}.tryLoadById(hardware);
    if (!dbHardware) {
        throw yacare::errors::BadRequest() << "Unknown hardware: " << hardware;
    }
    checkAcl(*txn, userInfo, *dbHardware);

    if (db::FirmwareGateway{*txn}.exists(
            db::table::Firmware::hardwareId == hardware &&
            db::table::Firmware::slot == slot &&
            db::table::Firmware::version == version)) {
        throw yacare::errors::Conflict() << "Firmware exists";
    }

    if (db::FirmwareUploadGateway{*txn}.exists(
            db::table::FirmwareUpload::hardwareId == hardware &&
            db::table::FirmwareUpload::slot == slot &&
            db::table::FirmwareUpload::version == version)) {
        throw yacare::errors::Conflict() << "Firmware upload exists";
    }

    db::FirmwareUpload fwUpload(userInfo.login(), hardware, slot, version);

    const auto key = makeKey(hardware, slot, version);
    const auto s3UploadId = s3Client.createMultipartUpload(key);
    fwUpload.setS3Key(key).setS3UploadId(s3UploadId);

    try {
        db::FirmwareUploadGateway{*txn}.insert(fwUpload);
        txn->commit();
    } catch (const std::exception& e) {
        ERROR() << "Failed to create multipart upload: " << e.what();
        s3Client.abortMultipartUpload(key, s3UploadId);
        throw;
    }

    makeFirmwareUploadResponse(response, fwUpload, db::FirmwareUploadParts{});
    response.setStatus(yacare::HTTPStatus::Created);
}


YCR_RESPOND_TO("PUT /v2/firmware/upload/add_part",
               YCR_USING(yacare::NginxLimitBody<yacare::ConfigScope::Endpoint>{}.maxSize(100_MB)),
               userInfo, id, part)
{
    auto& s3Client = configuration()->s3Client();
    auto txn = configuration()->pool().masterWriteableTransaction();

    auto fwUpload = db::FirmwareUploadGateway{*txn}.loadById(id);
    if (fwUpload.createdBy() != userInfo.login()) {
        throw yacare::errors::Forbidden();
    }

    const auto& body = request.body();
    std::stringbuf content(body);

    auto completedPart = s3Client.uploadPart(
        fwUpload.s3Key(), fwUpload.s3UploadId(), part, content);

    db::FirmwareUploadPart fwUploadPart(part, id, body.size(), completedPart.GetETag());
    db::FirmwareUploadPartGateway{*txn}.upsert(fwUploadPart);
    txn->commit();

    makeFirmwareUploadPartResponse(response, fwUploadPart);
}


YCR_RESPOND_TO("POST /v2/firmware/upload/finish", userInfo, id)
{
    auto& s3Client = configuration()->s3Client();
    auto txn = configuration()->pool().masterWriteableTransaction();
    db::FirmwareUploadGateway fwUploadGtw{*txn};
    db::FirmwareUploadPartGateway fwUploadPartGtw{*txn};

    auto fwUpload = fwUploadGtw.loadById(id);
    if (fwUpload.createdBy() != userInfo.login()) {
        throw yacare::errors::Forbidden();
    }

    auto parts = fwUploadPartGtw.load(db::table::FirmwareUploadPart::uploadId == id);
    if (parts.empty()) {
        throw yacare::errors::BadRequest() << "No parts have been added";
    }
    INFO() << "Found " << parts.size() << " uploaded parts, complete the upload";

    size_t size = 0;

    std::vector<Aws::S3::Model::CompletedPart> completedParts;
    completedParts.reserve(parts.size());
    for (const auto& part : parts) {
        completedParts.push_back(
            Aws::S3::Model::CompletedPart{}
                .WithPartNumber(part.id())
                .WithETag(part.contentEtag()));
        size += part.contentSize();
    }

    const std::string key = fwUpload.s3Key();
    s3Client.completeMultipartUpload(key, fwUpload.s3UploadId(), std::move(completedParts));

    try {
        INFO() << "Upload to S3 completed, make a firmware";
        auto firmware = makeFirmwareFromUpload(s3Client, fwUpload, size);
        db::FirmwareGateway{*txn}.insert(firmware);
        fwUploadPartGtw.remove(db::table::FirmwareUploadPart::uploadId == id);
        fwUploadGtw.removeById(id);
        txn->commit();

        makeFirmwareResponse(response, firmware);
    } catch (const std::exception& e) {
        ERROR() << "Failed to complete multipart upload for " << key << ": " << e.what();
        try {
            s3Client.deleteObject(key);
        } catch(const std::exception& e) {
            ERROR() << "Failed to delete object " << key << " from S3: " << e.what();
        }
        throw;
    }
}


YCR_RESPOND_TO("GET /v2/firmware/upload/list", userInfo, results = 0, skip = 0)
{
    auto txn = configuration()->pool().slaveTransaction();

    auto fwUploads = db::FirmwareUploadGateway{*txn}.load(
        db::table::FirmwareUpload::createdBy == userInfo.login(),
        pageOrderBy(db::table::FirmwareUpload::id, results, skip));

    if (fwUploads.empty()) {
        response.setStatus(yacare::HTTPStatus::NoContent);
        return;
    }

    db::TIds uploadIds = getIds(fwUploads);
    auto fwUploadParts = db::FirmwareUploadPartGateway{*txn}.load(
        db::table::FirmwareUploadPart::uploadId.in(uploadIds));

    auto uploadsWithParts = combineUploadsWithParts(std::move(fwUploads), std::move(fwUploadParts));
    makeFirmwareUploadsListResponse(response, uploadsWithParts);
}


YCR_RESPOND_TO("DELETE /v2/firmware/upload/delete", userInfo, id)
{
    auto& s3Client = configuration()->s3Client();
    auto txn = configuration()->pool().masterWriteableTransaction();
    db::FirmwareUploadGateway fwUploadGtw{*txn};
    db::FirmwareUploadPartGateway fwUploadPartGtw{*txn};

    auto fwUpload = fwUploadGtw.loadById(id);
    if (fwUpload.createdBy() != userInfo.login()) {
        throw yacare::errors::Forbidden();
    }

    try {
        s3Client.abortMultipartUpload(fwUpload.s3Key(), fwUpload.s3UploadId());
    } catch(const std::exception& e) {
        ERROR() << "Failed to delete ongoing upload " << fwUpload.s3UploadId()
                << " for key " << fwUpload.s3Key() << " from S3: " << e.what();
    }

    fwUploadPartGtw.remove(db::table::FirmwareUploadPart::uploadId == id);
    fwUploadGtw.removeById(id);
    txn->commit();
}

} //namespace maps::fw_updater::storage
