#include "role_tree.h"

#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/project_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/role_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/project_role_gateway.h>

namespace maps::fw_updater::storage::idm {

RoleTree RoleTree::load(pqxx::transaction_base& txn)
{
    RoleTree tree;

    for (auto& project : db::idm::ProjectGateway{txn}.load()) {
        auto id = project.id();
        tree.nodeById_.emplace(id, Node{std::move(project)});
    }

    // Fill tree edges
    for (const auto& [id, node] : tree.nodeById_) {
        if (node.parentId()) {
            auto& parent = tree.getNode(*node.parentId());
            parent.childIds_.push_back(id);
        } else {
            REQUIRE(tree.rootId_ == 0, "More than one root project found");
            tree.rootId_ = id;
        }
    }

    std::map<Id, std::shared_ptr<db::idm::Role>> roleMap;
    for (auto& role : db::idm::RoleGateway{txn}.load()) {
        auto id = role.id();
        roleMap.emplace(id, std::make_shared<db::idm::Role>(std::move(role)));
    }

    // Fill leafs and their edges
    for (auto& projectRole : db::idm::ProjectRoleGateway{txn}.load()) {
        const Id id = projectRole.id();
        auto role = roleMap.at(projectRole.roleId());

        auto& parentNode = tree.getNode(projectRole.projectId());
        parentNode.leafChildIds_.push_back(id);
        tree.leafNodeById_.emplace(id, LeafNode{std::move(projectRole), role});
    }

    REQUIRE(tree.rootId_ != 0, "No root project found");
    return tree;
}


const Node& RoleTree::getNode(Id id) const
{
    auto itr = nodeById_.find(id);
    REQUIRE(itr != nodeById_.end(), "Invalid project ID: " << id);
    return itr->second;
}

Node& RoleTree::getNode(Id id)
{
    return const_cast<Node&>(const_cast<const RoleTree*>(this)->getNode(id));
}

const LeafNode& RoleTree::getLeafNode(Id id) const
{
    auto itr = leafNodeById_.find(id);
    REQUIRE(itr != leafNodeById_.end(), "Invalid project role ID: " << id);
    return itr->second;
}

std::optional<Id> RoleTree::leafNodeIdBySlugPath(const SlugPath& path) const
{
    REQUIRE(!path.empty(), "Empty slug path");

    const Node* node = &root();
    for (size_t i = 0; i < path.size() - 1; ++i) {
        if (node->project().slug() != path[i].slug) {
            return std::nullopt;
        }
        auto itr = std::find_if(
            node->childIds().begin(),
            node->childIds().end(),
            [&](auto childId) {
                return getNode(childId).project().key() == path[i].key;
            });

        if (itr == node->childIds().end()) {
            return std::nullopt;
        }
        node = &getNode(*itr);
    }

    if (node->project().slug() != path.back().slug) {
        return std::nullopt;
    }

    // Search at the leaf level
    auto itr = std::find_if(
        node->leafChildIds().begin(),
        node->leafChildIds().end(),
        [&](auto childId) {
            return getLeafNode(childId).role().key() == path.back().key;
        });

    if (itr == node->leafChildIds().end()) {
        return std::nullopt;
    }
    return *itr;
}


SlugPath RoleTree::slugPathByLeafNodeId(Id leafNodeId) const
{
    SlugPath slugPath;
    const auto& leafNode = getLeafNode(leafNodeId);
    const auto* node = &getNode(leafNode.parentId());
    slugPath.push_back({.slug = node->project().slug(),
                        .key = leafNode.role().key()});

    while (node->parentId()) {
        auto key = node->project().key();
        node = &getNode(*node->parentId());
        slugPath.push_back({.slug = node->project().slug(),
                            .key = std::move(key)});
    }
    std::reverse(slugPath.begin(), slugPath.end());
    return slugPath;
}


void RoleTree::toJson(json::ObjectBuilder builder) const
{
    toJson(builder, root());
}


void RoleTree::toJson(
    json::ObjectBuilder builder,
    const Node& node) const
{
    const auto& project = node.project();

    // Don't add names for the root node
    if (node.parentId()) {
        builder["name"] << [&](json::ObjectBuilder bld) {
            bld["ru"] = project.nameRu();
            bld["en"] = project.nameEn();
        };
    }
    builder["roles"] << [&](json::ObjectBuilder bld) {
    bld["slug"] = project.slug();
        bld["name"] << [&](json::ObjectBuilder bld) {
            bld["ru"] = project.slugNameRu();
            bld["en"] = project.slugNameEn();
        };
        bld["values"] << [&](json::ObjectBuilder bld) {
            // Subprojects
            for (auto id : node.childIds()) {
                const auto& child = getNode(id);
                bld[child.project().key()] << [&](json::ObjectBuilder bld) {
                    toJson(bld, child);
                };
            }
            // Roles
            for (auto id : node.leafChildIds()) {
                const auto& leaf = getLeafNode(id);
                auto& role = leaf.role();
                bld[leaf.role().key()] << [&](json::ObjectBuilder bld) {
                    bld["name"] << [&](json::ObjectBuilder bld) {
                        bld["ru"] = leaf.role().nameRu();
                        bld["en"] = leaf.role().nameEn();
                    };
                    if (role.roleSet()) {
                        bld["set"] = *role.roleSet();
                    }
                };
            }
        };
    };
}

} //namespace maps::fw_updater::storage::idm
