#pragma once

#include "common.h"

#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/project.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/role.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/project_role.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/idm/user_role.h>
#include <maps/libs/json/include/builder.h>

#include <pqxx/transaction_base>

#include <map>

namespace maps::fw_updater::storage::idm {

/// Wrapper above db::idm::Project
/// which holds Project and references to child nodes
/// Invariant: either childIds or leafChildIds is empty
class Node {
public:
    explicit Node(db::idm::Project project)
        : project_(std::move(project))
    {}

    const db::idm::Project& project() const { return project_; }

    const Ids& childIds() const { return childIds_; }
    const Ids& leafChildIds() const { return leafChildIds_; }

    // Shortcuts
    Id id() const { return project_.id(); }
    const std::optional<Id>& parentId() const { return project_.parentId(); }

private:
    friend class RoleTree;

    db::idm::Project project_;
    Ids childIds_;
    Ids leafChildIds_;
};


/// Wrapper above db::idm::ProjectRole
/// which holds ProjectRole and reference to Role
class LeafNode {
public:
    LeafNode(db::idm::ProjectRole projectRole,
             std::shared_ptr<db::idm::Role> role)
        : projectRole_(std::move(projectRole))
        , role_(std::move(role))
    {
        REQUIRE(role_, "Broken project role id " << id());
    }

    const db::idm::Role& role() const { return *role_; }

    // Shortcuts
    Id id() const { return projectRole_.id(); }
    Id parentId() const { return projectRole_.projectId(); }

private:
    friend class RoleTree;

    db::idm::ProjectRole projectRole_;
    std::shared_ptr<db::idm::Role> role_;
};

/// Representation of Idm roles tree
/// where nodes are project and leaf nodes ar project roles
class RoleTree {
public:
    static RoleTree load(pqxx::transaction_base& txn);

    const Node& root() const { return getNode(rootId_); }
    const Node& getNode(Id id) const;
    const LeafNode& getLeafNode(Id id) const;

    std::optional<Id> leafNodeIdBySlugPath(const SlugPath& path) const;
    SlugPath slugPathByLeafNodeId(Id leafNodeId) const;

    void toJson(json::ObjectBuilder builder) const;

private:
    Node& getNode(Id id);

    void toJson(json::ObjectBuilder builder, const Node& node) const;

private:
    Id rootId_{0};
    std::map<Id, Node> nodeById_;
    std::map<Id, LeafNode> leafNodeById_;
};

} //namespace maps::fw_updater::storage::idm
