#pragma once

#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-core/include/aws/core/client/RetryStrategy.h>
#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-core/include/aws/core/client/AWSError.h>
#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-core/include/aws/core/client/CoreErrors.h>

#include <memory>

namespace maps::fw_updater::storage::s3 {

class RetryStrategy : public Aws::Client::RetryStrategy {
public:

    bool ShouldRetry(
        const Aws::Client::AWSError<Aws::Client::CoreErrors>& error,
        long attemptedRetries) const override
    {
        return attemptedRetries >= maxRetries_ ? false : error.ShouldRetry();
    }

    long CalculateDelayBeforeNextRetry(
        const Aws::Client::AWSError<Aws::Client::CoreErrors>& error,
        long attemptedRetries) const override
    {
        const auto getDelay = [=](unsigned long factor) {
            const auto delay = (1ul << static_cast<unsigned long>(attemptedRetries)) * factor;
            return static_cast<long>(delay);
        };

        if (error.GetErrorType() == Aws::Client::CoreErrors::SLOW_DOWN) {
            return getDelay(slowDownScaleFactor_);
        }
        if (attemptedRetries == 0) {
            return 0;
        }
        return getDelay(scaleFactor_);
    }

    long maxRetries() const { return maxRetries_; }
    unsigned long scaleFactor() const { return scaleFactor_; }
    unsigned long slowDownScaleFactor() const { return slowDownScaleFactor_; }

    RetryStrategy& setMaxRetries(long maxRetries)
    {
        maxRetries_ = maxRetries;
        return *this;
    }

    RetryStrategy& setScaleFactor(long factor)
    {
        scaleFactor_ = factor;
        return *this;
    }

    RetryStrategy& setSlowDownScaleFactor(long factor)
    {
        slowDownScaleFactor_ = factor;
        return *this;
    }

private:
    long maxRetries_ = 10;
    unsigned long scaleFactor_ = 25;
    unsigned long slowDownScaleFactor_ = 1000;
};

using RetryStrategyPtr = std::shared_ptr<RetryStrategy>;

inline RetryStrategyPtr defaultRetryStrategy()
{
    return std::make_shared<RetryStrategy>();
}

} // maps::fw_updater::storage::s3
