#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/firmware_upload_gateway.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/libs/common/include/exception.h>
#include <iostream>

using namespace maps::fw_updater;

namespace {

struct Stat {
    size_t errorCount{0};
    size_t warnCount{0};
    std::chrono::hours oldestAge{0};
};

Stat loadStatistic(pqxx::transaction_base& txn)
{
    static constexpr std::chrono::hours WARN_HOURS{36};
    static constexpr std::chrono::hours ERROR_HOURS{72};

    const auto now = maps::chrono::TimePoint::clock::now();

    auto ongoingUploads = db::FirmwareUploadGateway{txn}.load();

    Stat stat;
    for (const auto& upload : ongoingUploads) {
        auto age = std::chrono::duration_cast<std::chrono::hours>(now - upload.createdAt());
        if (age > ERROR_HOURS)
            stat.errorCount++;
        if (age > WARN_HOURS)
            stat.warnCount++;
        stat.oldestAge = std::max(stat.oldestAge, age);
    }
    return stat;
}

} // namespace

int main(int, const char**) try {
    maps::log8::setLevel(maps::log8::Level::FATAL);

    config::Config cfg = config::templateConfig();
    auto pool = cfg.makePoolHolder();
    auto txn = pool.pool().slaveTransaction();

    const Stat stat = loadStatistic(*txn);

    if (stat.errorCount) {
        std::cout << "2;Error: unfinished uploads: " << stat.errorCount
                  << ", the oldest is " << stat.oldestAge.count() << "h" << std::endl;
    } else if (stat.warnCount) {
        std::cout << "1;Warning: unfinished uploads: " << stat.warnCount
                  << ", the oldest is " << stat.oldestAge.count() << "h" << std::endl;
    } else {
        std::cout << "0;OK" << std::endl;
    }

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    std::cout << "2;Error: " << e.what() << std::endl;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    std::cout << "2;Error: " << e.what() << std::endl;
    return EXIT_FAILURE;
}
