#include "fixture.h"
#include "test_utils.h"

#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/branch_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/device_branch_gateway.h>

#include <maps/infra/yacare/include/test_utils.h>
#include <maps/libs/http/include/test_utils.h>

#include <yandex/maps/proto/firmware_updater/storage/branch.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

namespace proto = yandex::maps::proto::firmware_updater::storage;

namespace maps::fw_updater::storage::tests {

namespace {

const std::string LOGIN = "john";
const std::string HARDWARE_1 = "nanopi-neo4";
const std::string HARDWARE_2 = "raspberry-pi-3b";
const std::string DEVICE_A = "device-a";
const std::string DEVICE_B = "device-b";
const std::string DEVICE_C = "device-c";
const std::string DEVICE_D = "device-d";
const std::string STABLE = "stable";
const std::string TESTING = "testing";

const std::string URL_LIST_DEVICES = "http://localhost/v2/branch/list_devices";

void addUserRole(pqxx::transaction_base& txn)
{
    idm::addUserRole(txn, LOGIN, idm::parseSlugPath("project/one/subproject/one-A/role/manager"));
    idm::addUserRole(txn, LOGIN, idm::parseSlugPath("project/two/role/manager"));
    txn.commit();
}

http::MockResponse performAddDeviceBranchRequest(
    const std::string& hardware,
    const std::string& deviceId,
    const std::string& branch)
{
    auto request = http::MockRequest(http::POST,
        http::URL("http://localhost/v2/branch/add_device")
            .addParam("hardware", hardware)
            .addParam("deviceid", deviceId)
            .addParam("branch", branch)
        );
    return yacare::performTestRequest(request);
}

http::MockResponse performRemoveDeviceBranchRequest(
    const std::string& hardware,
    const std::string& deviceId)
{
    auto request = http::MockRequest(http::DELETE,
        http::URL("http://localhost/v2/branch/remove_device")
            .addParam("hardware", hardware)
            .addParam("deviceid", deviceId)
        );
    return yacare::performTestRequest(request);
}

inline bool areEqual(const proto::DeviceBranch& pb, const db::DeviceBranch& db)
{
    return pb.hardware() == db.hardwareId()
        && pb.deviceid() == db.deviceId()
        && pb.branch() == db.branch();
}

} // namespace

TEST_F(Fixture, test_add_device_branch_errors)
{
    // No userInfo provided
    auto response = performAddDeviceBranchRequest(HARDWARE_1, DEVICE_A, TESTING);
    EXPECT_EQ(response.status, 401);

    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));

    // No rights in IDM
    response = performAddDeviceBranchRequest(HARDWARE_1, DEVICE_A, TESTING);
    EXPECT_EQ(response.status, 403);

    addUserRole(*txnHandle());

    // No such branch
    response = performAddDeviceBranchRequest(HARDWARE_1, DEVICE_A, TESTING);
    EXPECT_EQ(response.status, 400);
}

TEST_F(Fixture, test_add_device_branch_success)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto txn = txnHandle();
    db::Branches branches {db::Branch(STABLE), db::Branch(TESTING)};
    db::BranchGateway{*txn}.insert(branches);
    txn->commit();

    {
        auto response = performAddDeviceBranchRequest(HARDWARE_1, DEVICE_A, TESTING);
        EXPECT_EQ(response.status, 200);
        EXPECT_EQ(response.headers.at("Content-Type"), PROTOBUF_MEDIA_TYPE);
        proto::DeviceBranch deviceBranch;
        ASSERT_TRUE(deviceBranch.ParseFromString(TString(response.body)));
        EXPECT_EQ(deviceBranch.hardware(), HARDWARE_1);
        EXPECT_EQ(deviceBranch.deviceid(), DEVICE_A);
        EXPECT_EQ(deviceBranch.branch(), TESTING);

        auto dbDeviceBranches = db::DeviceBranchGateway{*txnHandle()}.load();
        ASSERT_EQ(dbDeviceBranches.size(), 1u);
        EXPECT_TRUE(areEqual(deviceBranch, dbDeviceBranches[0]));
    }

    {
        // Update branch for existing device
        auto response = performAddDeviceBranchRequest(HARDWARE_1, DEVICE_A, STABLE);
        EXPECT_EQ(response.status, 200);
        proto::DeviceBranch deviceBranch;
        ASSERT_TRUE(deviceBranch.ParseFromString(TString(response.body)));
        EXPECT_EQ(deviceBranch.hardware(), HARDWARE_1);
        EXPECT_EQ(deviceBranch.deviceid(), DEVICE_A);
        EXPECT_EQ(deviceBranch.branch(), STABLE);

        auto dbDeviceBranches = db::DeviceBranchGateway{*txnHandle()}.load();
        ASSERT_EQ(dbDeviceBranches.size(), 1u);
        EXPECT_TRUE(areEqual(deviceBranch, dbDeviceBranches[0]));
    }
}

TEST_F(Fixture, test_list_device_branches)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto txn = txnHandle();
    db::Branches branches {db::Branch(STABLE), db::Branch(TESTING)};
    db::BranchGateway{*txn}.insert(branches);
    db::DeviceBranches deviceBranches {
        {HARDWARE_1, DEVICE_A, TESTING},
        {HARDWARE_1, DEVICE_B, STABLE},
        {HARDWARE_1, DEVICE_C, TESTING},
        {HARDWARE_2, DEVICE_D, TESTING}
    };
    db::DeviceBranchGateway{*txn}.insert(deviceBranches);
    txn->commit();

    {
        http::MockRequest request(http::GET,
            http::URL(URL_LIST_DEVICES).addParam("hardware", HARDWARE_2));
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);
        EXPECT_EQ(response.headers.at("Content-Type"), PROTOBUF_MEDIA_TYPE);
        proto::DeviceBranchesList deviceBranchesList;
        ASSERT_TRUE(deviceBranchesList.ParseFromString(TString(response.body)));
        ASSERT_EQ(deviceBranchesList.device_branches_size(), 1);
        EXPECT_TRUE(areEqual(deviceBranchesList.device_branches(0), deviceBranches[3]));
    }

    {
        http::MockRequest request(http::GET,
            http::URL(URL_LIST_DEVICES)
                .addParam("hardware", HARDWARE_1)
                .addParam("branch", TESTING)
            );
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);
        proto::DeviceBranchesList deviceBranchesList;
        ASSERT_TRUE(deviceBranchesList.ParseFromString(TString(response.body)));
        ASSERT_EQ(deviceBranchesList.device_branches_size(), 2);
        EXPECT_TRUE(areEqual(deviceBranchesList.device_branches(0), deviceBranches[0]));
        EXPECT_TRUE(areEqual(deviceBranchesList.device_branches(1), deviceBranches[2]));
    }

    {
        http::MockRequest request(http::GET,
            http::URL(URL_LIST_DEVICES)
                .addParam("hardware", HARDWARE_1)
                .addParam("deviceid", DEVICE_B)
            );
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);
        proto::DeviceBranchesList deviceBranchesList;
        ASSERT_TRUE(deviceBranchesList.ParseFromString(TString(response.body)));
        ASSERT_EQ(deviceBranchesList.device_branches_size(), 1);
        EXPECT_TRUE(areEqual(deviceBranchesList.device_branches(0), deviceBranches[1]));
    }

    // Unknown hardware
    {
        http::MockRequest request(http::GET,
            http::URL(URL_LIST_DEVICES).addParam("hardware", "unknown-hardware"));
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 400);
    }

    // Pagination
    {
        http::MockRequest request(http::GET,
            http::URL(URL_LIST_DEVICES)
                .addParam("hardware", HARDWARE_1)
                .addParam("results", 2)
                .addParam("skip", 0)
            );
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);
        proto::DeviceBranchesList pageOneDeviceBranches;
        ASSERT_TRUE(pageOneDeviceBranches.ParseFromString(TString(response.body)));
        ASSERT_EQ(pageOneDeviceBranches.device_branches_size(), 2);
        EXPECT_TRUE(areEqual(pageOneDeviceBranches.device_branches(0), deviceBranches[1]));
        EXPECT_TRUE(areEqual(pageOneDeviceBranches.device_branches(1), deviceBranches[0]));

        request = http::MockRequest(http::GET,
            http::URL(URL_LIST_DEVICES)
                .addParam("hardware", HARDWARE_1)
                .addParam("results", 2)
                .addParam("skip", 2)
            );
        response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);
        proto::DeviceBranchesList pageTwoDeviceBranches;
        ASSERT_TRUE(pageTwoDeviceBranches.ParseFromString(TString(response.body)));
        ASSERT_EQ(pageTwoDeviceBranches.device_branches_size(), 1);
        EXPECT_TRUE(areEqual(pageTwoDeviceBranches.device_branches(0), deviceBranches[2]));
    }
}


TEST_F(Fixture, test_delete_device_branch_unauthorized)
{
    auto txn = txnHandle();
    db::Branch branch(TESTING);
    db::BranchGateway{*txn}.insert(branch);
    db::DeviceBranch deviceBranch{HARDWARE_1, DEVICE_A, TESTING};
    db::DeviceBranchGateway{*txn}.insert(deviceBranch);
    txn->commit();

    // No userInfo provided
    auto response = performRemoveDeviceBranchRequest(HARDWARE_1, DEVICE_A);
    EXPECT_EQ(response.status, 401);

    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));

    // No rights in IDM
    response = performRemoveDeviceBranchRequest(HARDWARE_1, DEVICE_A);
    EXPECT_EQ(response.status, 403);
}


TEST_F(Fixture, test_delete_device_branch)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto txn = txnHandle();
    db::Branch branch(TESTING);
    db::BranchGateway{*txn}.insert(branch);
    db::DeviceBranch deviceBranch {HARDWARE_1, DEVICE_A, TESTING};
    db::DeviceBranchGateway{*txn}.insert(deviceBranch);
    txn->commit();

    auto response = performRemoveDeviceBranchRequest(HARDWARE_1, DEVICE_A);
    EXPECT_EQ(response.status, 200);

    // Check that device is removed
    http::MockRequest request(http::GET,
        http::URL(URL_LIST_DEVICES)
            .addParam("hardware", HARDWARE_1)
            .addParam("deviceid", DEVICE_A)
        );
    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 204);

    txn = txnHandle();
    EXPECT_TRUE(db::DeviceBranchGateway{*txn}.load().empty());

    // Remove non-existing device
    response = performRemoveDeviceBranchRequest(HARDWARE_1, DEVICE_A);
    EXPECT_EQ(response.status, 200);
}

} // namespace maps::fw_updater::storage::tests
