#include "fixture.h"
#include "test_utils.h"

#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/storage/lib/idm/idm_service.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/firmware_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/firmware_upload_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/firmware_upload_part_gateway.h>

#include <maps/infra/yacare/include/test_utils.h>
#include <maps/libs/http/include/test_utils.h>

#include <yandex/maps/proto/firmware_updater/storage/firmware.pb.h>
#include <yandex/maps/proto/firmware_updater/storage/upload.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

namespace proto = yandex::maps::proto::firmware_updater::storage;

namespace maps::fw_updater::storage::tests {

namespace {

constexpr size_t MB = 1024 * 1024;

const std::string LOGIN = "john";
const std::string HARDWARE = "nanopi-neo4";
const std::string ROOTFS = "rootfs";
const std::string VER_1 = "1.0.0";
const std::string VER_2 = "2.0.0";

const std::string URL_ADD_PART = "http://localhost/v2/firmware/upload/add_part";
const std::string URL_DELETE = "http://localhost/v2/firmware/upload/delete";
const std::string URL_FINISH = "http://localhost/v2/firmware/upload/finish";
const std::string URL_LIST = "http://localhost/v2/firmware/upload/list";
const std::string URL_START = "http://localhost/v2/firmware/upload/start";

void addUserRole(pqxx::transaction_base& txn)
{
    idm::addUserRole(txn, LOGIN, idm::parseSlugPath("project/one/subproject/one-A/role/manager"));
    txn.commit();
}

http::MockResponse performUploadStartRequest(
    const std::string& hardware,
    const std::string& slot,
    const std::string& version)
{
    http::MockRequest request(http::PUT,
        http::URL(URL_START)
            .addParam("hardware", hardware)
            .addParam("slot", slot)
            .addParam("version", version)
        );
    return yacare::performTestRequest(request);
}

} // namespace

TEST_F(Fixture, test_create_multipart_upload_bad_parameters)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));

    // Unknown hardware
    auto response = performUploadStartRequest("unknown-hardware", ROOTFS, VER_1);
    EXPECT_EQ(response.status, 400);

    // Unknown slot
    response = performUploadStartRequest(HARDWARE, "unknown-slot", VER_1);
    EXPECT_EQ(response.status, 400);
}


TEST_F(Fixture, test_create_multipart_upload_unauthorized)
{
    // No userInfo provided
    auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 401);

    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));

    // No rights in IDM
    response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 403);
}


TEST_F(Fixture, test_create_multipart_upload_succeses)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 201);
    EXPECT_EQ(response.headers.at("Content-Type"), PROTOBUF_MEDIA_TYPE);

    proto::Upload upload;
    ASSERT_TRUE(upload.ParseFromString(TString(response.body)));
}


TEST_F(Fixture, test_add_part_out_of_range)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 201);

    proto::Upload upload;
    ASSERT_TRUE(upload.ParseFromString(TString(response.body)));

    auto request = http::MockRequest(http::PUT,
        http::URL(URL_ADD_PART)
            .addParam("id", upload.id())
            .addParam("part", 10001) // invalid, max part number is 10000
        );
    request.body = std::string(5 * MB, 'a');

    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 400);
}


TEST_F(Fixture, test_add_part_wrong_user)
{
    std::string uploadId;
    {
        yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
        addUserRole(*txnHandle());

        auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
        EXPECT_EQ(response.status, 201);

        proto::Upload upload;
        ASSERT_TRUE(upload.ParseFromString(TString(response.body)));
        uploadId = upload.id();
    }

    {
        yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo("another"));

        http::MockRequest request(http::PUT,
            http::URL(URL_ADD_PART)
                .addParam("id", uploadId)
                .addParam("part", 1)
            );
        request.body = std::string(5 * MB, 'a');

        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 403);
    }
}


TEST_F(Fixture, test_add_part_success)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 201);

    proto::Upload upload;
    ASSERT_TRUE(upload.ParseFromString(TString(response.body)));

    http::MockRequest request(http::PUT,
        http::URL(URL_ADD_PART)
            .addParam("id", upload.id())
            .addParam("part", 1)
        );
    request.body = std::string(5 * MB, 'a');

    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 200);
    EXPECT_EQ(response.headers.at("Content-Type"), PROTOBUF_MEDIA_TYPE);

    proto::UploadPart uploadPart;
    ASSERT_TRUE(uploadPart.ParseFromString(TString(response.body)));
    EXPECT_EQ(uploadPart.part_id(), "1");
    EXPECT_EQ(uploadPart.upload_id(), upload.id());
    EXPECT_EQ(uploadPart.content_size(), 5 * MB);
}


TEST_F(Fixture, test_add_same_part_twice)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 201);

    proto::Upload upload;
    ASSERT_TRUE(upload.ParseFromString(TString(response.body)));

    http::MockRequest request(http::PUT,
        http::URL(URL_ADD_PART)
            .addParam("id", upload.id())
            .addParam("part", 1)
        );
    request.body = std::string(5 * MB, 'a');

    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 200);

    proto::UploadPart oldPart;
    ASSERT_TRUE(oldPart.ParseFromString(TString(response.body)));

    request = http::MockRequest(http::PUT,
        http::URL(URL_ADD_PART)
            .addParam("id", upload.id())
            .addParam("part", 1)
        );
    request.body = std::string(5 * MB + 1, 'b');

    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 200);

    proto::UploadPart newPart;
    ASSERT_TRUE(newPart.ParseFromString(TString(response.body)));

    // Check database record
    auto txn = txnHandle();
    auto dbParts = db::FirmwareUploadPartGateway{*txn}.load(
        db::table::FirmwareUploadPart::uploadId == std::stoul(upload.id()));
    ASSERT_EQ(dbParts.size(), 1u);
    EXPECT_EQ(dbParts[0].id(), 1u);
    EXPECT_EQ(dbParts[0].contentSize(), newPart.content_size());
    EXPECT_EQ(dbParts[0].contentEtag(), std::string(newPart.content_etag()));
}


TEST_F(Fixture, test_add_list_and_finish_upload)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 201);

    proto::Upload upload;
    ASSERT_TRUE(upload.ParseFromString(TString(response.body)));

    constexpr size_t NUM_PARTS = 3;
    for (size_t partNumber = 1; partNumber <= NUM_PARTS; ++partNumber) {
        http::MockRequest request(http::PUT,
            http::URL(URL_ADD_PART)
                .addParam("id", upload.id())
                .addParam("part", partNumber)
            );
        request.body = std::string(5 * MB, 'a' + partNumber);
        response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);
    }

    http::MockRequest request(http::GET, http::URL(URL_LIST));
    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 200);
    EXPECT_EQ(response.headers.at("Content-Type"), PROTOBUF_MEDIA_TYPE);

    proto::UploadsList uploadsList;
    ASSERT_TRUE(uploadsList.ParseFromString(TString(response.body)));
    ASSERT_EQ(uploadsList.uploads_size(), 1);
    EXPECT_EQ(uploadsList.uploads(0).id(), upload.id());
    ASSERT_EQ(uploadsList.uploads(0).parts_size(), 3);
    for (size_t partNumber = 1; partNumber <= NUM_PARTS; ++partNumber) {
        const auto& part = uploadsList.uploads(0).parts(partNumber - 1);
        EXPECT_EQ(part.upload_id(), upload.id());
        EXPECT_EQ(std::stoul(part.part_id()), partNumber);
        EXPECT_EQ(part.content_size(), 5 * MB);
        EXPECT_FALSE(part.content_etag().empty());
    }

    uint64_t uploadId = std::stoul(upload.id());

    request = http::MockRequest(http::POST,
         http::URL(URL_FINISH).addParam("id", uploadId));
    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.headers.at("Content-Type"), PROTOBUF_MEDIA_TYPE);
    EXPECT_EQ(response.status, 200);

    proto::Firmware firmware;
    ASSERT_TRUE(firmware.ParseFromString(TString(response.body)));
    EXPECT_EQ(firmware.hardware(), HARDWARE);
    EXPECT_EQ(firmware.slot(), ROOTFS);
    EXPECT_EQ(firmware.version(), VER_1);
    EXPECT_EQ(firmware.download_url(),
              "https://maps-mrc-drive-package-testing.localhost/nanopi-neo4-rootfs-1.0.0");
    EXPECT_EQ(firmware.content_size(), NUM_PARTS * 5 * MB);
    EXPECT_EQ(firmware.content_md5(), "063959d6c5bef2d58134b183ac09e622");

    // Check database records
    auto txn = txnHandle();
    auto dbFirmware = db::FirmwareGateway{*txn}.loadOne(
        db::table::Firmware::versionSeq == firmware.version_seq());
    EXPECT_TRUE(areEqual(firmware, dbFirmware));

    EXPECT_FALSE(db::FirmwareUploadGateway{*txn}.existsById(uploadId));
    EXPECT_FALSE(db::FirmwareUploadPartGateway{*txn}.exists(
        db::table::FirmwareUploadPart::uploadId == uploadId));

    // Check no active uploads
    request = http::MockRequest(http::GET, http::URL(URL_LIST));
    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 204);
}


TEST_F(Fixture, test_two_simultaneouts_uploads)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 201);
    proto::Upload uploadOne;
    ASSERT_TRUE(uploadOne.ParseFromString(TString(response.body)));

    response = performUploadStartRequest(HARDWARE, ROOTFS, VER_2);
    EXPECT_EQ(response.status, 201);
    proto::Upload uploadTwo;
    ASSERT_TRUE(uploadTwo.ParseFromString(TString(response.body)));

    constexpr size_t NUM_PARTS = 3;
    for (size_t partNumber = 0; partNumber < NUM_PARTS; ++partNumber) {
        http::MockRequest request(http::PUT,
            http::URL(URL_ADD_PART)
                .addParam("id", uploadOne.id())
                .addParam("part", 1 + partNumber)
            );
        request.body = std::string(5 * MB, 'a' + partNumber);
        response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);

        request = http::MockRequest(http::PUT,
            http::URL(URL_ADD_PART)
                .addParam("id", uploadTwo.id())
                .addParam("part", 1 + partNumber)
            );
        request.body = std::string(5 * MB, 'a' + partNumber);
        response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);
    }

    http::MockRequest request(http::GET, http::URL(URL_LIST));
    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 200);

    proto::UploadsList uploadsList;
    ASSERT_TRUE(uploadsList.ParseFromString(TString(response.body)));
    ASSERT_EQ(uploadsList.uploads_size(), 2);
    EXPECT_EQ(uploadsList.uploads(0).id(), uploadOne.id());
    EXPECT_EQ(uploadsList.uploads(1).id(), uploadTwo.id());

    // Check upload parts
    for (size_t i : {0, 1}) {
        EXPECT_EQ(uploadsList.uploads(i).parts_size(), int(NUM_PARTS));
        for (size_t partNumber = 1; partNumber <= NUM_PARTS; ++partNumber) {
            const auto& part = uploadsList.uploads(i).parts(partNumber - 1);
            EXPECT_EQ(part.upload_id(), uploadsList.uploads(i).id());
            EXPECT_EQ(std::stoul(part.part_id()), partNumber);
        }
    }

    request = http::MockRequest(http::POST,
        http::URL(URL_FINISH).addParam("id", uploadOne.id()));
    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 200);
    proto::Firmware firmwareOne;
    ASSERT_TRUE(firmwareOne.ParseFromString(TString(response.body)));

    request = http::MockRequest(http::POST,
        http::URL(URL_FINISH).addParam("id", uploadTwo.id()));
    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 200);
    proto::Firmware firmwareTwo;
    ASSERT_TRUE(firmwareTwo.ParseFromString(TString(response.body)));
}


TEST_F(Fixture, test_two_simultaneouts_uploads_conflict)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 201);
    proto::Upload uploadOne;
    ASSERT_TRUE(uploadOne.ParseFromString(TString(response.body)));

    response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 409);
}


TEST_F(Fixture, test_list_uploads_with_pagination)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    constexpr size_t NUM_UPLOADS = 5;
    std::vector<std::string> uploadIds;

    for (size_t i = 0; i < NUM_UPLOADS; ++i) {
        auto response = performUploadStartRequest(HARDWARE, ROOTFS, std::to_string(i));
        EXPECT_EQ(response.status, 201);
        proto::Upload upload;
        ASSERT_TRUE(upload.ParseFromString(TString(response.body)));
        uploadIds.push_back(upload.id());
    }

    http::MockRequest request(http::GET,
        http::URL(URL_LIST)
            .addParam("results", 3)
            .addParam("skip", 0)
        );
    auto response = yacare::performTestRequest(request);

    proto::UploadsList pageOne;
    ASSERT_TRUE(pageOne.ParseFromString(TString(response.body)));
    ASSERT_EQ(pageOne.uploads_size(), 3);
    EXPECT_EQ(pageOne.uploads(0).id(), uploadIds[0]);
    EXPECT_EQ(pageOne.uploads(1).id(), uploadIds[1]);
    EXPECT_EQ(pageOne.uploads(2).id(), uploadIds[2]);

    request = http::MockRequest(http::GET,
        http::URL(URL_LIST)
            .addParam("results", 3)
            .addParam("skip", 3)
        );
    response = yacare::performTestRequest(request);

    proto::UploadsList pageTwo;
    ASSERT_TRUE(pageTwo.ParseFromString(TString(response.body)));
    ASSERT_EQ(pageTwo.uploads_size(), 2);
    EXPECT_EQ(pageTwo.uploads(0).id(), uploadIds[3]);
    EXPECT_EQ(pageTwo.uploads(1).id(), uploadIds[4]);
}


TEST_F(Fixture, test_delete_upload)
{
    yacare::tests::UserInfoFixture userInfoFixture(makeUserInfo(LOGIN));
    addUserRole(*txnHandle());

    auto response = performUploadStartRequest(HARDWARE, ROOTFS, VER_1);
    EXPECT_EQ(response.status, 201);

    proto::Upload upload;
    ASSERT_TRUE(upload.ParseFromString(TString(response.body)));

    constexpr size_t NUM_PARTS = 3;
    for (size_t partNumber = 0; partNumber < NUM_PARTS; ++partNumber) {
        http::MockRequest request(http::PUT,
            http::URL(URL_ADD_PART)
                .addParam("id", upload.id())
                .addParam("part", 1 + partNumber)
            );
        request.body = std::string(5 * MB, 'a' + partNumber);
        response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);
    }

    http::MockRequest request(http::DELETE,
        http::URL(URL_DELETE).addParam("id", upload.id()));
    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 200);

    // Check database records
    auto txn = txnHandle();
    EXPECT_TRUE(db::FirmwareGateway{*txn}.load().empty());
    EXPECT_TRUE(db::FirmwareUploadGateway{*txn}.load().empty());
    EXPECT_TRUE(db::FirmwareUploadPartGateway{*txn}.load().empty());

    // Check no active uploads
    request = http::MockRequest(http::GET, http::URL(URL_LIST));
    response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 204);
}

} // namespace maps::fw_updater::storage::tests
