#include "fixture.h"

#include <library/cpp/testing/gtest/gtest.h>

#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/updater/lib/firmware_signature.h>

#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/enums.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/rollout_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/drive/firmware_updater/libs/db/include/rollout_history_gateway.h>

#include <maps/libs/http/include/test_utils.h>
#include <maps/libs/json/include/prettify.h>
#include <maps/libs/json/include/value.h>

#include <maps/infra/yacare/include/yacare.h>
#include <maps/infra/yacare/include/test_utils.h>

#include <util/string/hex.h>

#include <thread>


using namespace yacare::tests;
using namespace maps::fw_updater::db;
using Fixture = maps::fw_updater::tests::Fixture;

namespace maps::fw_updater::tests {

namespace {

const std::string HW_1 = "nanopi-neo4";
const std::string HW_2 = "raspberry-pi-3b";
const std::string HW_3 = "nanopi-neo2";
const std::string DEVICE_1 = "device1";
const std::string DEVICE_2 = "device2";
const std::string USER_AGENT = "mrc-drive-agent";
const std::string CLIENT_PATH = "maps/wikimap/mapspro/services/mrc/drive/firmware_updater/client/bin_test/bin_test";

std::string getSignatureFromClient(const std::string& url)
{
    std::string keyFile = "firmware.key";
    std::ostringstream programCall;
    const auto program = BinaryPath(CLIENT_PATH);
    programCall << program
                << " --key=\"" << SIGNATURE_KEY << "\""
                << " --url=\"" << url << "\""
                << " --user-agent=\"" << USER_AGENT << "\""
                << " > " << keyFile;

    std::system(programCall.str().c_str());
    std::ifstream input(keyFile);
    std::string signature;
    input >> signature;
    return signature;
}

http::MockRequest makeMockRequest(
    const std::string& hw,
    const std::string& deviceid,
    const std::string& slots
    )
{
    http::URL url("http://localhost/firmware/1.x/target_state");
    url.addParam("hardware", hw)
       .addParam("deviceid", deviceid)
       .addParam("slots", slots);

    http::MockRequest request(http::GET, url);
    request.headers.emplace("User-Agent", USER_AGENT);
    request.headers.emplace("X-YFirmware-Signature", getSignatureFromClient(url.toString()));

    return request;
}

void prepareActivityTests(maps::pgpool3::TransactionHandle txn)
{
    RolloutGateway rolloutGtw{*txn};
    RolloutHistoryGateway rolloutHistoryGtw{*txn};

    Rollouts rollouts = {
        Rollout("testing", 1, "1234"),
        Rollout("testing", 2, "5678"),
        Rollout("stable", 4, "abcd")
    };
    rolloutGtw.insert(rollouts);

    RolloutHistories histories = {
        RolloutHistory(1, 100, RolloutStatus::Active, "mrc-user"),
        RolloutHistory(2, 100, RolloutStatus::Active, "mrc-user"),
        RolloutHistory(3, 100, RolloutStatus::Active, "mrc-user")
    };
    rolloutHistoryGtw.insert(histories);

    txn->commit();
}

void prepareInactivityTests(maps::pgpool3::TransactionHandle txn)
{
    RolloutGateway rolloutGtw{*txn};
    RolloutHistoryGateway rolloutHistoryGtw{*txn};

    Rollouts rollouts = {
        Rollout("testing", 1, "1234"),
        Rollout("testing", 2, "5678"),
        Rollout("stable",  4, "abcd"),
        Rollout("stable",  5, "abcd")
    };
    rolloutGtw.insert(rollouts);

    RolloutHistories histories = {
        RolloutHistory(1, 100, RolloutStatus::Inactive, "mrc-user"),
        RolloutHistory(2, 100, RolloutStatus::Active, "mrc-user"),
        RolloutHistory(3, 100, RolloutStatus::Inactive, "mrc-user"),
        RolloutHistory(4, 100, RolloutStatus::Inactive, "mrc-user")
    };
    rolloutHistoryGtw.insert(histories);

    txn->commit();
}

void prepareMultipleHistoryTests(maps::pgpool3::TransactionHandle txn)
{
    RolloutGateway rolloutGtw{*txn};
    RolloutHistoryGateway rolloutHistoryGtw{*txn};

    std::vector<Rollout> rollouts = {
        Rollout("testing", 1, "1234"),
        Rollout("testing", 2, "5678"),
        Rollout("stable", 4, "abcd")
    };
    rolloutGtw.insert(rollouts);

    RolloutHistories histories = {
        RolloutHistory(1, 100, RolloutStatus::Active,   "mrc-user"),
        RolloutHistory(1, 100, RolloutStatus::Inactive, "mrc-user"),
        RolloutHistory(2, 100, RolloutStatus::Inactive, "mrc-user"),
        RolloutHistory(2, 100, RolloutStatus::Active,   "mrc-user"),
        RolloutHistory(3, 1,   RolloutStatus::Active,   "mrc-user")
    };
    rolloutHistoryGtw.insert(histories);
    txn->commit();
}

} // namespace

TEST(firmware_signature, test_hmac_multithread)
{
    std::vector<std::thread> threads;
    constexpr size_t threadsNum = 100;

    const std::vector<std::string> stringsToEncode(2 * threadsNum, "somestring");
    std::vector<std::string> results(threadsNum);

    for (size_t threadId = 0; threadId < threadsNum; ++threadId) {
        threads.emplace_back([threadId, &stringsToEncode, &results](){
            const auto clientFirmwareSignature = fw_signature::hmacSha1(SIGNATURE_KEY, stringsToEncode);
            results[threadId] = HexEncode(clientFirmwareSignature.data(), clientFirmwareSignature.size());
        });
    }

    for (auto& thr: threads) {
        thr.join();
    }

    std::string hexAnswer = "F572C2A6BE4F70EF1BCC3CAF256EE98270929292";
    for (const auto& resultStr : results)
        EXPECT_EQ(resultStr, hexAnswer);
}

TEST_F(Fixture, test_active_rollouts)
{
    prepareActivityTests(txnHandle());

    {
        auto request = makeMockRequest(HW_1, DEVICE_1, "rootfs,appfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);

        auto json = json::Value::fromString(response.body);
        auto slots = json["slots"];
        ASSERT_EQ(slots.size(), 2u);
        for (const auto& slot : slots) {
            if (slot["name"].as<std::string>() == "rootfs") {
                EXPECT_EQ(slot["version"].as<std::string>(), "1.1");
                EXPECT_EQ(slot["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo4/1.1/arm64/rootfs.bundle");
                EXPECT_EQ(slot["md5"].as<std::string>(), "54230f6a78a50012dd9c2a8d6d9a21ef");
                EXPECT_EQ(slot["size"].as<size_t>(), 8702014u);
            } else if (slot["name"].as<std::string>() == "appfs") {
                EXPECT_EQ(slot["version"].as<std::string>(), "0.1");
                EXPECT_EQ(slot["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo4/app/0.1/arm64/app.zip");
                EXPECT_EQ(slot["md5"].as<std::string>(), "db87c566e122290ca57b2bafd0eff162");
                EXPECT_EQ(slot["size"].as<size_t>(), 335313u);
            } else {
                // No such slot
                FAIL();
            }
        }
    }

    {
        // 1 slot existing
        auto request = makeMockRequest(HW_2, DEVICE_2, "rootfs,appfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);

        auto json = json::Value::fromString(response.body);
        auto slots = json["slots"];
        ASSERT_EQ(slots.size(), 1u);
        auto rootfs = slots[0];
        EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
        EXPECT_EQ(rootfs["version"].as<std::string>(), "2.1");
        EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/raspberry-pi-3b/2.1/arm64/rootfs.bundle");
        EXPECT_EQ(rootfs["md5"].as<std::string>(), "a26b805acc8f8b983ec3bc69e39fb2d1");
        EXPECT_EQ(rootfs["size"].as<size_t>(), 71028319u);
    }

    {
        // Test non existing device request
        auto request = makeMockRequest(HW_1, DEVICE_2, "rootfs,appfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 404);
    }

    {
        // Test non existing slot request
        auto request = makeMockRequest(HW_2, DEVICE_2, "appfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 404);
    }
}

TEST_F(Fixture, test_inactive_requests)
{
    prepareInactivityTests(txnHandle());

    {
        // 1 active, 1 inactive rollout
        auto request = makeMockRequest(HW_1, DEVICE_1, "rootfs,appfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);

        auto json = json::Value::fromString(response.body);
        auto slots = json["slots"];
        ASSERT_EQ(slots.size(), 1u);
        auto appfs = slots[0];
        EXPECT_EQ(appfs["name"].as<std::string>(), "appfs");
        EXPECT_EQ(appfs["version"].as<std::string>(), "0.1");
        EXPECT_EQ(appfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo4/app/0.1/arm64/app.zip");
        EXPECT_EQ(appfs["md5"].as<std::string>(), "db87c566e122290ca57b2bafd0eff162");
        EXPECT_EQ(appfs["size"].as<size_t>(), 335313u);
    }

    {
        // Inactive 1 slot
        auto request = makeMockRequest(HW_1, DEVICE_1, "rootfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 204);
    }

    {
        // Inactive 2 slots
        auto request = makeMockRequest(HW_2, DEVICE_2, "rootfs,appfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 204);
    }
}

TEST_F(Fixture, test_multiple_histories)
{
    prepareMultipleHistoryTests(txnHandle());

    {
        // Last experiment is inactive
        auto request = makeMockRequest(HW_1, DEVICE_1, "rootfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 204);
    }

    {
        // Last experiment is active
        auto request = makeMockRequest(HW_1, DEVICE_1, "appfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 200);

        auto json = json::Value::fromString(response.body);
        auto slots = json["slots"];
        ASSERT_EQ(slots.size(), 1u);
        auto appfs = slots[0];
        EXPECT_EQ(appfs["name"].as<std::string>(), "appfs");

        EXPECT_EQ(appfs["name"].as<std::string>(), "appfs");
        EXPECT_EQ(appfs["version"].as<std::string>(), "0.1");
        EXPECT_EQ(appfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo4/app/0.1/arm64/app.zip");
        EXPECT_EQ(appfs["md5"].as<std::string>(), "db87c566e122290ca57b2bafd0eff162");
        EXPECT_EQ(appfs["size"].as<size_t>(), 335313u);
    }

    {
        // Low percentage experiment
        auto request = makeMockRequest(HW_2, DEVICE_2, "rootfs");
        auto response = yacare::performTestRequest(request);
        EXPECT_EQ(response.status, 404);
    }
}

TEST_F(Fixture, test_real_histories)
{
    // https://jing.yandex-team.ru/files/miplot/updater_experiments_logic-v2.png
    // 1) Experiment 0: initial firmware rollout: 100%
    // 2) Experiment 1: 20% active rollout
    // 3) Experiment 1: 100% active rollout
    // 4) Experiment 2: 20% active rollout
    // 5) Experiment 2: 20% inactive rollout
    // 6) Experiment 3: 40% active rollout
    // 7) Experiment 3: 100% active rollout

    {
        auto txn = txnHandle();
        RolloutGateway rolloutGtw{*txn};
        RolloutHistoryGateway rolloutHistoryGtw{*txn};

        std::vector<Rollout> rollouts = {
            Rollout("stable", 7,  "ver10"),
            Rollout("stable", 8,  "ver11"),
            Rollout("stable", 9,  "ver12"),
            Rollout("stable", 10, "ver13")
        };
        rolloutGtw.insert(rollouts);
        txn->commit();
    }

    {
        // Initial rollout, 100%
        auto txn = txnHandle();
        RolloutHistoryGateway rolloutHistoryGtw{*txn};
        rolloutHistoryGtw.insert(RolloutHistory(1, 100, RolloutStatus::Active, "mrc-user"));
        txn->commit();

        for (const std::string& deviceId : {"A", "B", "C", "D", "E", "F", "G", "H"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.0");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.0/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "f5e55d2d434e7198030007f3279021bf");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 1231612u);
        }
    }

    {
        // 1 experiment, 20%
        auto txn = txnHandle();
        RolloutHistoryGateway rolloutHistoryGtw{*txn};
        rolloutHistoryGtw.insert(RolloutHistory(2, 20, RolloutStatus::Active, "mrc-user"));
        txn->commit();

        for (const std::string& deviceId : {"A", "B", "C", "D", "F", "G"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.0");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.0/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "f5e55d2d434e7198030007f3279021bf");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 1231612u);
        }

         for (const std::string& deviceId : {"E", "H"}) {
             auto request = makeMockRequest(HW_3, deviceId, "rootfs");
             auto response = yacare::performTestRequest(request);
             EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.1");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.1/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "e799f4244af865ff6052fea52eb7b7c5");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 4719201u);
        }
    }

    {
        // 1 experiment, 100%
        auto txn = txnHandle();
        RolloutHistoryGateway rolloutHistoryGtw{*txn};
        rolloutHistoryGtw.insert(RolloutHistory(2, 100, RolloutStatus::Active, "mrc-user"));
        txn->commit();

        for (const std::string& deviceId : {"A", "B", "C", "D", "E", "F", "G", "H"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.1");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.1/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "e799f4244af865ff6052fea52eb7b7c5");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 4719201u);
        }
    }

    {
        // 2 experiment, 20%
        auto txn = txnHandle();
        RolloutHistoryGateway rolloutHistoryGtw{*txn};
        rolloutHistoryGtw.insert(RolloutHistory(3, 20, RolloutStatus::Active, "mrc-user"));
        txn->commit();

        for (const std::string& deviceId : {"B", "C", "D", "F", "G", "H"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.1");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.1/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "e799f4244af865ff6052fea52eb7b7c5");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 4719201u);
        }

        for (const std::string& deviceId : {"A", "E"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.2");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.2/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "21c6ed807edd0a959b3e52d3976f9d02");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 8172912u);
        }
    }

    {
        // 2 experiment, inactive
        auto txn = txnHandle();
        RolloutHistoryGateway rolloutHistoryGtw{*txn};
        rolloutHistoryGtw.insert(RolloutHistory(3, 20, RolloutStatus::Inactive, "mrc-user"));
        txn->commit();

        for (const std::string& deviceId : {"A", "E"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 204);
        }

        for (const std::string& deviceId : {"B", "C", "D", "F", "G", "H"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.1");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.1/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "e799f4244af865ff6052fea52eb7b7c5");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 4719201u);
        }
    }

    {
        // 3 experiment: 40%
        auto txn = txnHandle();
        RolloutHistoryGateway rolloutHistoryGtw{*txn};
        rolloutHistoryGtw.insert(RolloutHistory(4, 40, RolloutStatus::Active, "mrc-user"));
        txn->commit();

        for (const std::string& deviceId : {"A"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 204);
        }

        for (const std::string& deviceId : {"B", "F", "G", "H"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.1");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.1/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "e799f4244af865ff6052fea52eb7b7c5");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 4719201u);
        }

        for (const std::string& deviceId : {"C", "D", "E"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.3");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.3/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "af8a493f5d47c9ce39e62a99855e614f");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 3012980u);
        }
    }

    {
        // 3 experiment: 100%
        auto txn = txnHandle();
        RolloutHistoryGateway rolloutHistoryGtw{*txn};
        rolloutHistoryGtw.insert(RolloutHistory(4, 100, RolloutStatus::Active, "mrc-user"));
        txn->commit();

        for (const std::string& deviceId : {"A", "B", "C", "D", "E", "F", "G", "H"}) {
            auto request = makeMockRequest(HW_3, deviceId, "rootfs");
            auto response = yacare::performTestRequest(request);
            EXPECT_EQ(response.status, 200);

            auto json = json::Value::fromString(response.body);
            auto slots = json["slots"];
            ASSERT_EQ(slots.size(), 1u);
            auto rootfs = slots[0];
            EXPECT_EQ(rootfs["name"].as<std::string>(), "rootfs");
            EXPECT_EQ(rootfs["version"].as<std::string>(), "1.3");
            EXPECT_EQ(rootfs["url"].as<std::string>(), "s3.yandex.ru/nanopi-neo2/1.3/arm64/rootfs.bundle");
            EXPECT_EQ(rootfs["md5"].as<std::string>(), "af8a493f5d47c9ce39e62a99855e614f");
            EXPECT_EQ(rootfs["size"].as<size_t>(), 3012980u);
        }
    }
}

TEST_F(Fixture, test_invalid_signature)
{
    prepareInactivityTests(txnHandle());

    auto request = makeMockRequest(HW_1, DEVICE_1, "rootfs");
    request.headers["X-YFirmware-Signature"] = "invalid_signature";

    auto response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 403);
}

} // maps::fw_updater::tests
