#include "power_manager.h"
#include "battery.h"
#include "power_statistics.h"

#include <maps/libs/log8/include/log8.h>

#include <string>
#include <iostream>
#include <chrono>
#include <thread>

namespace maps::mrc::drive {

using namespace std::chrono;

PowerManager::PowerManager(std::shared_ptr<Battery> data,
                           std::shared_ptr<PowerStatistics> statistics,
                           int pollSeconds,
                           int empty_loops,
                           int maxSecsOnBattery)
    : batteryData_(move(data))
    , powerStatistics_(move(statistics))
    , secsPollInterval_(pollSeconds)
    , maxEmptyLoops_(empty_loops)
    , maxSecsOnBattery_(maxSecsOnBattery)
{
}

void PowerManager::runTick()
{
    if (batteryData_->isChargerConnected()) {
        if (sourceName_.empty() || sourceName_ == "BAT") {
            powerStatistics_->beginOnCharger();
            INFO() << "Power supply restored";
        }

        if (!batteryData_->isCharging())
            batteryData_->updateBatteryLiveStatus();

        if ((batteryData_->isBatteryAlive() && !batteryData_->isCharging())
            && (sourceName_ == "BAT"
                || !isCharged_
                || batteryData_->needCharging())
            ) {
            isCharged_ = false;
            emptyLoops_ = 0;
            batteryData_->startCharging();
            powerStatistics_->beginCharging();
        }

        sourceName_ = "AC";
        powerStatistics_->updateTimeOnCharger();

        if (batteryData_->isCharging() && !isCharged_) {
            batteryData_->setChargingVoltage();
            if (batteryData_->batteryChargeCurrent() < 1e-12) {
                emptyLoops_++;
                if (emptyLoops_ >= maxEmptyLoops_) {
                    batteryData_->stopCharging();
                    INFO() << "Charging done in "
                              << powerStatistics_->secondsSinceCharging().count() / 60.
                              << " minutes.";
                    isCharged_ = true;
                }
            } else {
                emptyLoops_ = 0;
            }
        }
    } else {
        if (sourceName_.empty() || sourceName_ == "AC") {
            batteryData_->stopCharging();
            powerStatistics_->beginOnBattery();
            WARN() << "Alert: Power failure, running on battery, limit "
                      << maxSecsOnBattery_.count() << "s.";
        }

        sourceName_ = "BAT";
        powerStatistics_->updateTimeOnBattery();
        if (powerStatistics_->secondsOnBattery() > maxSecsOnBattery_ ) {
            WARN() << "Max on-battery time limit ("
                      << maxSecsOnBattery_.count() << "s"
                      << ") reached. Shutting down...";
            system("shutdown -h now");
        }

        emptyLoops_ = 0;
        isCharged_ = false;
    }

    std::this_thread::sleep_for(secsPollInterval_);
}

} // namespace maps::mrc::drive