#include "server.h"

#include "battery.h"
#include "power_statistics.h"

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/std.h>
#include <maps/libs/log8/include/log8.h>

#include <iostream>
#include <cstdio>
#include <sys/socket.h>
#include <sys/un.h>
#include <cstring>
#include <cerrno>
#include <sstream>

namespace maps::mrc::drive {

using namespace std::chrono;

Server::Server(std::shared_ptr<Battery> data,
               std::shared_ptr<PowerStatistics> statistics,
               std::string sock)
    : batteryData_(move(data))
    , powerStatistics_(move(statistics))
    , socketName_(move(sock))
{
}

void Server::start()
{
    std::remove(socketName_.c_str());

    int fdServer = 0;
    if ((fdServer = socket(AF_UNIX, SOCK_STREAM, 0)) == 0) {
        ERROR() << "Socket failed";
        exit(EXIT_FAILURE);
    }

    struct sockaddr_un address;
    memset(&address, 0, sizeof(struct sockaddr_un));

    int addrlen = sizeof(address);

    address.sun_family = AF_UNIX;
    strncpy(
        address.sun_path,
        socketName_.c_str(),
        sizeof(address.sun_path) - 1
    );

    if (bind(fdServer, (struct sockaddr *) &address, sizeof(struct sockaddr_un)) < 0) {
        ERROR() << "Bind failed: " << std::strerror(errno);
        exit(EXIT_FAILURE);
    }

    for (;;) {
        if (listen(fdServer, 1) < 0) {
            ERROR() << "Error listening: " << std::strerror(errno);
            break;
        }

        int connection = 0;
        if ((connection = accept(fdServer, (struct sockaddr *) &address,
                                 (socklen_t *) &addrlen)) < 0) {
            ERROR() << "Error in accept";
            break;
        }

        char buffer[1024];
        if (read(connection, buffer, 1024) < 0) {
            ERROR() << "Error in reading" << std::strerror(errno);
            break;
        }

        std::string bufferStr(buffer);

        std::ostringstream output;
        maps::json::Builder builder(output);
        bool needShutdown = false;
        if (bufferStr == "STATUS\n") {
            batteryData_->updateBatteryLiveStatus();
            generateStatusMsg(builder);
        } else if (bufferStr == "SHUTDOWN\n") {
            builder << [](maps::json::ObjectBuilder builder) {
                builder["result"] = true;
            };

            WARN() << "Shutdown requested via socket";
            needShutdown = true;
        } else if (bufferStr == "SOURCE\n") {
            std::string source = batteryData_->isChargerConnected()
                                 ? "AC"
                                 : "BAT";
            builder << [&](maps::json::ObjectBuilder builder) {
              builder["result"] = true;
              builder["source"] = source;
            };
        } else if (bufferStr == "UPS-ON\n") {
            batteryData_->enableBattery();
            builder << [&](maps::json::ObjectBuilder builder) {
              if (batteryData_->isBatteryOn()) {
                  builder["result"] = true;
              } else {
                  builder["result"] = false;
                  builder["reason"] = "no-battery";
              }
            };
        } else if (bufferStr == "UPS-OFF\n") {
            batteryData_->disableBattery();
            builder << [&](maps::json::ObjectBuilder builder) {
              builder["result"] = true;
            };
        } else {
            builder << [&](maps::json::ObjectBuilder builder) {
              builder["result"] = false;
              builder["reason"] = "unknown-request";
            };
        }

        send(connection, output.str().c_str(), output.str().size(), 0);
        if (needShutdown)
            system("shutdown -h now");

        close(connection);
    }

    close(fdServer);
    exit(EXIT_FAILURE);
}

void Server::generateStatusMsg(maps::json::Builder& builder) const
{
    try {
        bool batteryStatus = batteryData_->isCharging()
                          || batteryData_->isBatteryAlive();
        std::string powerSource = batteryData_->isChargerConnected()
                                  ? "AC"
                                  : "BAT";
        seconds chargingTime = batteryData_->isCharging()
                               ? powerStatistics_->secondsSinceCharging()
                               : 0s;

        builder << [&](maps::json::ObjectBuilder builder) {
            builder["result"] = true;
            builder["input"] = batteryData_->inputVoltage();
            builder["battery"] << [&](maps::json::ObjectBuilder builder) {
              builder["present"] = batteryStatus;
              builder["type"] = static_cast<int>(batteryData_->batteryType());
              builder["voltage"] = batteryData_->batteryVoltage();
              builder["is_charging"] = batteryData_->isCharging();
              builder["charge_current"] = batteryData_->batteryChargeCurrent();
              builder["discharge_current"] = batteryData_->batteryDischargeCurrent();
              builder["charging_time"] = chargingTime.count();
            };

            builder["time_on_battery"] = powerStatistics_->secondsOnBattery().count();
            builder["time_on_ac"] = powerStatistics_->secondsOnCharger().count();
            builder["ups_status"] = batteryData_->isBatteryOn();
        };
    }
    catch (const std::exception &e) {
        builder << [&](maps::json::ObjectBuilder builder) {
          builder["result"] = false;
          builder["reason"] = "system-exception";
          builder["error"] = e.what();
        };
    }
}

} // namespace maps::mrc::drive
