#include <maps/wikimap/mapspro/services/mrc/eye/lib/detect_sign/include/detect_sign.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detect_house_number/include/detect_house_number.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detect_traffic_light/include/detect_traffic_light.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detect_road_marking/include/detect_road_marking.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id_stream.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/load.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/secure_config.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/common.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <util/generic/yexception.h>

#include <chrono>
#include <string>

using namespace maps;
using namespace maps::mrc;
using namespace maps::mrc::eye;

namespace {

maps::mrc::yt::PoolType parseYtPoolType(const std::string& value)
{
    if (value == "ad_hoc") {
        return maps::mrc::yt::PoolType::AdHoc;
    } else if (value == "processing") {
        return maps::mrc::yt::PoolType::Processing;
    } else {
       throw maps::RuntimeError() << "Unsupported ytPoolType value '" << value << "'";
    }
}

} // namespace

int main(int argc, const char** argv) try {
    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser;

    auto syslog = parser.string("syslog-tag")
            .help("redirect log output to syslog with given tag");

    auto mrcConfigPath = parser.string("mrc-config")
            .help("path to mrc config");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto rework = parser.flag("rework")
            .help("run detection on already processed frames");

    auto frameIdsPath = parser.string("frame-ids")
            .help("path to frame ids file");

    auto loop = parser.flag("loop")
            .help("work in infinite loop");

    auto batchSize = parser.num("batch")
            .defaultValue(1000)
            .help("frame batch size (default 1000)");

    auto partitionSize = parser.num("partition-size")
            .defaultValue(70)
            .help("partition size");

    auto concurrency = parser.num("concurrency")
            .defaultValue(100)
            .help("number of parallel yt jobs");

    auto commit = parser.flag("commit")
            .help("commit results");

    auto ytCommit = parser.flag("yt-commit")
            .help("commit yt results (for debug)");

    auto ytPath = parser.string("yt-path")
            .help("yt root path (get from config by default)");

    auto ytPoolType = parser.string("yt-pool-type")
            .defaultValue("processing")
            .help("yt pool type ('ad_hoc' or 'processing')");

    auto useGpu = parser.flag("use-gpu")
            .help("use gpu for detection (do not use for detect panel)");

    auto lockFree = parser.flag("lock-free")
            .help("don't take pg lock (may be dangerous)");

    auto timeout = parser.num("timeout")
            .help("wait for db update in minutes (default 5)")
            .defaultValue(5);

    auto detectionType = parser.string("detect")
            .help("type of detection")
            .required();

    parser.parse(argc, const_cast<char**>(argv));

    if (syslog.defined()) {
        log8::setBackend(log8::toSyslog(syslog));
    }

    const auto mrcConfig = maps::mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);
    maps::mrc::eye::SecureConfig::initialize(mrcConfig);

    auto poolHolder = mrcConfig.makePoolHolder();
    const auto loader = FrameLoader::fromConfig(mrcConfig);
    const auto yt = mrcConfig.externals().yt().makeClient();

    std::unique_ptr<BaseMrcWorker> detect;

    auto initConfig = [&](auto&& config) {
        config.mrc.pool = &poolHolder.pool();
        config.mrc.lockFree = lockFree;
        config.mrc.commit = commit;
        config.yt.client = yt.Get();
        config.yt.frameLoader = &loader;
        config.yt.commit = ytCommit;
        config.yt.rootPath = ytPath.defined() ? ytPath : mrcConfig.externals().yt().path() + "/detect_" + detectionType;
        config.yt.partitionSize = partitionSize;
        config.yt.concurrency = concurrency;
        config.yt.useGpu = useGpu;
        config.yt.portoLayerPath = std::string(yt::DEFAULT_GPU_PORTO_LAYER);
        config.yt.poolType = parseYtPoolType(ytPoolType);
        config.rework = rework;

        return config;
    };

    if ("sign" == detectionType) {
        const auto config = initConfig(DetectSignConfig());
        detect.reset(new DetectSign(config));
    } else if ("house-number" == detectionType) {
        const auto config = initConfig(DetectHouseNumberConfig());
        detect.reset(new DetectHouseNumber(config));
    } else if ("traffic-light" == detectionType) {
        const auto config = initConfig(DetectTrafficLightConfig());
        detect.reset(new DetectTrafficLight(config));
    } else if ("road-marking" == detectionType) {
        const auto config = initConfig(DetectRoadMarkingConfig());
        detect.reset(new DetectRoadMarking(config));
    } else {
        throw maps::RuntimeError() << "Unknown object type: " << detectionType;
    }

    if (not loop) {
        if (frameIdsPath.defined()) {
            for (IdStream idStream(frameIdsPath); idStream.valid(); ) {
                detect->processBatch(idStream.readBatch(batchSize));
            }
        } else {
            WARN() << "No feature ids!";
        }

        return EXIT_SUCCESS;
    }

    INFO() << "Infinite loop mode...";
    if (frameIdsPath.defined()) {
        WARN() << "Ignore file '" << frameIdsPath << "'";
    }

    detect->runInLoopMode(batchSize, std::chrono::minutes(timeout));

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    FATAL() << e;
    return EXIT_FAILURE;
} catch (const yexception& e) {
    FATAL() << e.what();

    if (e.BackTrace()) {
        FATAL() << e.BackTrace()->PrintToString();
    }

    return EXIT_FAILURE;
} catch (const std::exception& e) {
    FATAL() << e.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Unknown error!";
    return EXIT_FAILURE;
}
