#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_panorama_frame/include/metadata.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/monitoring/include/check.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/monitoring/include/common.h>

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye_panorama_gateway.h>

#include <maps/libs/chrono/include/days.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/libs/common/include/exception.h>

#include <iostream>

using namespace maps::mrc;
using namespace maps::mrc::eye;

namespace {

juggler::Message checkImportQueueSize(ImportPanoramaFrameMetadataManager& metadata,
                                      db::EyePanoramaGateway& gateway)
{
    using table = db::table::EyePanorama;

    static constexpr int64_t WARN_QUEUE_SIZE = 10000;
    static constexpr int64_t ERROR_QUEUE_SIZE = 100000;

    const auto txnId = metadata.getTxnId();
    const auto sessionId = metadata.getSessionId();
    const auto orderNum = metadata.getOrderNum();

    const auto fromSameSession = table::txnId == txnId &&
                                 table::sessionId == sessionId &&
                                 table::orderNum > orderNum;

    const auto fromNextSession = table::txnId == txnId &&
                                 table::sessionId > sessionId;

    const int64_t size = gateway.count(fromSameSession ||
                                       fromNextSession ||
                                       table::txnId > txnId);

    const std::string message = std::to_string(size) + " panoramas in queue";

    if (size > ERROR_QUEUE_SIZE) {
        return juggler::error(message);
    } else if (size > WARN_QUEUE_SIZE) {
        return juggler::warn(message);
    } else {
        return juggler::ok(message);
    }
}

} // namespace

int main(int argc, char** argv) try {
    maps::log8::setLevel(maps::log8::Level::ERROR);

    auto config = maps::mrc::common::templateConfig();
    auto pool = config.makePoolHolder();
    auto txn = pool.pool().slaveTransaction();

    auto metadata = eye::importPanoramaFrameMetadata(*txn);
    db::EyePanoramaGateway eyePanoramaGateway(*txn);

    const MonitoringCheck checkQueue = std::bind(
        checkImportQueueSize,
        std::ref(metadata),
        std::ref(eyePanoramaGateway)
    );

    const MonitoringCheckByName checkByName {
        {"queue-size", checkQueue}
    };

    maps::cmdline::Parser parser;

    auto name = parser.string("check")
        .help("checks: " + collectNameList(checkByName))
        .required();

    parser.parse(argc, argv);

    REQUIRE(checkByName.count(name), "Unknown check " << name);
    std::cout << checkByName.at(name)() << std::endl;

    return EXIT_SUCCESS;
} catch (const std::exception& e) {
    std::cout << juggler::error(e.what()) << std::endl;
    return EXIT_FAILURE;
} catch (...) {
    std::cout << juggler::error("unknown exception") << std::endl;
    return EXIT_FAILURE;
}
