#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/include/object_manager.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id_stream.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/secure_config.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/position_matcher.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/position_clusterizer.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/superpoints_match.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/catboost_visibility_predictor.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <util/generic/yexception.h>

#include <chrono>
#include <memory>
#include <string>

using namespace maps;
using namespace maps::mrc;
using namespace maps::mrc::eye;

namespace {

maps::mrc::yt::PoolType parseYtPoolType(const std::string& value)
{
    if (value == "ad_hoc") {
        return maps::mrc::yt::PoolType::AdHoc;
    } else if (value == "processing") {
        return maps::mrc::yt::PoolType::Processing;
    } else {
       throw maps::RuntimeError() << "Unsupported ytPoolType value '" << value << "'";
    }
}

} // namespace

int main(int argc, const char** argv) try {
    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser;

    auto syslog = parser.string("syslog-tag")
            .help("redirect log output to syslog with given tag");

    auto mrcConfigPath = parser.string("mrc-config")
            .help("path to mrc config");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto detectionGroupIdsPath = parser.string("detection-group-ids")
            .help("path to detection group ids file");

    auto geoIdCoveragePathOpt = parser.string("geoid-coverage-path")
        .help("path to coverage5-geoid mms");

    auto loop = parser.flag("loop")
            .help("work in infinite loop");

    auto batchSize = parser.num("batch")
            .defaultValue(1000)
            .help("detection batch size (default 1000)");

    auto commit = parser.flag("commit")
            .help("commit results");

    auto lockFree = parser.flag("lock-free")
            .help("don't take pg lock (may be dangerous)");

    auto timeout = parser.num("timeout")
            .help("wait for db update in minutes (default 5)")
            .defaultValue(5);

    auto useGpu = parser.flag("use-gpu")
            .help("use gpu for detection (do not use for detect panel)");

    auto ytConcurrency = parser.num("yt-concurrency")
            .defaultValue(100)
            .help("number of parallel yt jobs");

    auto ytPoolType = parser.string("yt-pool-type")
            .defaultValue("processing")
            .help("yt pool type ('ad_hoc' or 'processing')");

    auto enableSign = parser.flag("enable-sign")
            .help("enable sign managment");

    auto enableTrafficLight = parser.flag("enable-traffic-light")
            .help("enable traffic light managment");

    auto enableHouseNumber = parser.flag("enable-house-number")
            .help("enable house number managment");

    auto enableRoadMarking = parser.flag("enable-road-marking")
            .help("enable road marking managment");

    auto rework = parser.flag("rework")
            .help("rework already processed detections");

    parser.parse(argc, const_cast<char**>(argv));

    if (syslog.defined()) {
        log8::setBackend(log8::toSyslog(syslog));
    }

    const auto config = mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);
    maps::mrc::eye::SecureConfig::initialize(config);

    std::vector<db::eye::DetectionType> types;
    if (enableSign) {
        types.push_back(db::eye::DetectionType::Sign);
    }

    if (enableTrafficLight) {
        types.push_back(db::eye::DetectionType::TrafficLight);
    }

    if (enableHouseNumber) {
        types.push_back(db::eye::DetectionType::HouseNumber);
    }

    if (enableRoadMarking) {
        types.push_back(db::eye::DetectionType::RoadMarking);
    }

    auto poolHolder = config.makePoolHolder();
    const auto loader = FrameLoader::fromConfig(config);
    const auto yt = config.externals().yt().makeClient();

    ObjectManagerConfig workerConfig;
    workerConfig.mrc.pool = &poolHolder.pool();
    workerConfig.mrc.commit = commit;
    workerConfig.mrc.lockFree = lockFree;
    workerConfig.rework = rework;
    workerConfig.detectionTypes = types;
    workerConfig.frameMatcher = std::make_shared<FrameSuperpointsMatcher>(
        yt, loader, useGpu, parseYtPoolType(ytPoolType), ytConcurrency
    );
    workerConfig.detectionMatcher = std::make_shared<PositionDetectionMatcher>();
    workerConfig.clusterizer = std::make_shared<PositionDetectionClusterizer>();
    workerConfig.visibilityPredictor = std::make_shared<CatboostClusterVisibilityPredictor>();

    const std::string coveragePath =
        geoIdCoveragePathOpt.defined()
            ? geoIdCoveragePathOpt
            : config.externals().geoIdCoveragePath();
    workerConfig.geoIdProvider = privacy::makeGeoIdProvider(coveragePath);
    workerConfig.verificationRules = loadVerificationRules();

    ObjectManager manager(workerConfig);

    if (not loop) {
        if (detectionGroupIdsPath.defined()) {
            for (IdStream idStream(detectionGroupIdsPath); idStream.valid(); ) {
                manager.processBatch(idStream.readBatch(batchSize));
            }
        } else {
            WARN() << "No detection ids!";
        }

        return EXIT_SUCCESS;
    }

    INFO() << "Infinite loop mode...";
    if (detectionGroupIdsPath.defined()) {
        WARN() << "Ignore detection ids '" << detectionGroupIdsPath << "'";
    }

    manager.runInLoopMode(batchSize, std::chrono::minutes(timeout));

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    FATAL() << e;
    return EXIT_FAILURE;
} catch (const yexception& e) {
    FATAL() << e.what();

    if (e.BackTrace()) {
        FATAL() << e.BackTrace()->PrintToString();
    }

    return EXIT_FAILURE;
} catch (const std::exception& e) {
    FATAL() << e.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Unknown error!";
    return EXIT_FAILURE;
}
