#include <maps/wikimap/mapspro/services/mrc/eye/lib/sync_detection/include/sync_detection.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id_stream.h>

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <util/generic/yexception.h>

#include <chrono>
#include <string>

using namespace maps;
using namespace maps::mrc;
using namespace maps::mrc::eye;

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser;

    auto syslog = parser.string("syslog-tag")
            .help("redirect log output to syslog with given tag");

    auto mrcConfigPath = parser.string("mrc-config")
            .help("path to mrc config");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto frameIdsPath = parser.string("frame-ids")
            .help("path to frame ids file");

    auto loop = parser.flag("loop")
            .help("work in infinite loop");

    auto batchSize = parser.num("batch")
            .defaultValue(1000)
            .help("frame batch size (default 1000)");

    auto commit = parser.flag("commit")
            .help("commit results");

    auto lockFree = parser.flag("lock-free")
            .help("don't take pg lock (may be dangerous)");

    auto timeout = parser.num("timeout")
            .help("wait for db update in minutes (default 5)")
            .defaultValue(5);

    auto enableSign = parser.flag("enable-sign")
            .help("enable sign detection sync");

    auto enableTrafficLight = parser.flag("enable-traffic-light")
            .help("enable traffic light detection sync");

    auto enableHouseNumber = parser.flag("enable-house-number")
            .help("enable house number detection sync");

    auto enableRoadMarking = parser.flag("enable-road-marking")
            .help("enable road marking detection sync");

    parser.parse(argc, const_cast<char**>(argv));

    if (syslog.defined()) {
        log8::setBackend(log8::toSyslog(syslog));
    }

    const auto config = mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);

    std::vector<db::eye::DetectionType> detectionTypes;

    if (enableSign) {
        detectionTypes.push_back(db::eye::DetectionType::Sign);
    }

    if (enableTrafficLight) {
        detectionTypes.push_back(db::eye::DetectionType::TrafficLight);
    }

    if (enableHouseNumber) {
        detectionTypes.push_back(db::eye::DetectionType::HouseNumber);
    }

    if (enableRoadMarking) {
        detectionTypes.push_back(db::eye::DetectionType::RoadMarking);
    }

    auto poolHolder = config.makePoolHolder();

    SyncDetectionConfig workerConfig;
    workerConfig.mrc.pool = &poolHolder.pool();
    workerConfig.mrc.commit = commit;
    workerConfig.mrc.lockFree = lockFree;
    workerConfig.detectionTypes = detectionTypes;

    SyncDetection sync(workerConfig);

    if (not loop) {
        if (frameIdsPath.defined()) {
            for (IdStream idStream(frameIdsPath); idStream.valid(); ) {
                sync.processBatch(idStream.readBatch(batchSize));
            }
        } else {
            WARN() << "No frame ids!";
        }

        return EXIT_SUCCESS;
    }

    INFO() << "Infinite loop mode...";
    if (frameIdsPath.defined()) {
        WARN() << "Ignore frame ids '" << frameIdsPath << "'";
    }

    sync.runInLoopMode(batchSize, std::chrono::minutes(timeout));

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    FATAL() << e;
    return EXIT_FAILURE;
} catch (const yexception& e) {
    FATAL() << e.what();

    if (e.BackTrace()) {
        FATAL() << e.BackTrace()->PrintToString();
    }

    return EXIT_FAILURE;
} catch (const std::exception& e) {
    FATAL() << e.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Unknown error!";
    return EXIT_FAILURE;
}
