import json
import cv2
import argparse
import numpy as np
import base64


def parse_base64_image(base64_image):
    data = np.frombuffer(base64.b64decode(base64_image), dtype=np.uint8)
    return cv2.imdecode(data, cv2.IMREAD_COLOR)


def load_images(path):
    data = json.load(open(path))

    result = {}
    for feature_image in data['features_image']:
        feature_id = feature_image['feature_id']
        image = feature_image['image']
        result[feature_id] = image
    return result


class BBox:
    def __init__(self, data):
        self.x_min = min(data[0][0], data[1][0])
        self.x_max = max(data[0][0], data[1][0])
        self.y_min = min(data[0][1], data[1][1])
        self.y_max = max(data[0][1], data[1][1])

    def draw(self, image):
        tl = (self.x_min, self.y_min)
        br = (self.x_max, self.y_max)
        cv2.rectangle(image, tl, br, (0, 0, 255), 3)


class Object:
    def __init__(self, feature_id, data):
        self.feature_id = feature_id
        self.id = data['object_id']
        self.type = data['type']
        self.bbox = BBox(data['bbox'])

    def draw(self, image):
        self.bbox.draw(image)


def load_objects(path):
    data = json.load(open(path))

    result = {}
    count = 0
    for feature_objects in data['features_objects']:
        feature_id = feature_objects['feature_id']

        if feature_id not in result:
            result[feature_id] = {}

        for object_data in feature_objects['objects']:
            object = Object(feature_id, object_data)
            result[feature_id][object.id] = object
            count += 1

    return result, count


def load_clusters(path):
    data = json.load(open(path))

    return data['clusters']


class Viewer:
    def __init__(self, images, objects, clusters):
        self.images = images
        self.objects = objects
        self.clusters = clusters

        assert len(self.clusters) > 0, 'There is no cluster'

        self.cluster_indx = 0
        self.object_indx = 0

    def next_cluster(self):
        clusters_count = len(self.clusters)
        self.cluster_indx = (self.cluster_indx + 1) % clusters_count
        self.object_indx = 0

    def prev_cluster(self):
        clusters_count = len(self.clusters)
        self.cluster_indx = (clusters_count + self.cluster_indx - 1) % clusters_count
        self.object_indx = 0

    def next_object(self):
        object_count = len(self.clusters[self.cluster_indx]['objects'])
        self.object_indx = (self.object_indx + 1) % object_count

    def prev_object(self):
        object_count = len(self.clusters[self.cluster_indx]['objects'])
        self.object_indx = (object_count + self.object_indx - 1) % object_count

    def render(self):
        cluster_objects = self.clusters[self.cluster_indx]['objects']
        cluster_object = cluster_objects[self.object_indx]
        object = self.objects[cluster_object['feature_id']][cluster_object['object_id']]
        image = parse_base64_image(self.images[object.feature_id])
        object.draw(image)
        font = cv2.FONT_HERSHEY_SIMPLEX
        cluster_str = 'cluster {}/{}'.format(self.cluster_indx + 1, len(self.clusters))
        object_str = 'object {}/{}'.format(self.object_indx + 1, len(cluster_objects))
        cv2.putText(image, cluster_str, (10, 50), font, 1, (0, 0, 255), 2, cv2.LINE_AA)
        cv2.putText(image, object_str, (10, 100), font, 1, (0, 0, 255), 2, cv2.LINE_AA)
        cv2.imshow('Viewer', image)

    def exit(self):
        exit()


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--images', help='Path to images', required=True)
    parser.add_argument('--objects', help='Path to objects', required=True)
    parser.add_argument('--clusters', help='Path to clusters', required=True)
    args = parser.parse_args()

    print('Loading images: {}'.format(args.images))
    images = load_images(args.images)
    print('Loaded {} images'.format(len(images)))

    print('Loading objects: {}'.format(args.objects))
    objects, objects_count = load_objects(args.objects)
    print('Loaded {} objects'.format(objects_count))

    print('Loading clusters: {}'.format(args.clusters))
    clusters = load_clusters(args.clusters)
    print('Loaded {} clusters'.format(len(clusters)))

    viewer = Viewer(images, objects, clusters)

    actions = {
        ord('n'): viewer.next_cluster, # Press 'n'
        ord('p'): viewer.prev_cluster, # Press 'p'
        ord('d'): viewer.next_object,  # Press 'd'
        ord('a'): viewer.prev_object,  # Press 'a'
        27:       viewer.exit          # Press 'Esc'
    }

    while True:
        viewer.render()
        key = cv2.waitKeyEx()
        if key in actions:
            actions[key]()


if __name__ == '__main__':
    main()
