#include "yt_serialization.h"
#include "util.h"

#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/serialization.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/yt_serialization.h>

#include <mapreduce/yt/interface/client.h>

#include <opencv2/opencv.hpp>

namespace maps::mrc::yt {

namespace {

geolib3::Line2 toGeolibLine(const cv::Point3f& line)
{
    auto point = std::abs(line.x) > geolib3::EPS
    ? geolib3::Point2(-line.z / line.x, 0)
    : geolib3::Point2(0, -line.z/ line.y);

    return geolib3::Line2(point, geolib3::Vector2(line.y, line.x));
}


cv::Point3f toCvPoint(const geolib3::Line2& line)
{
    return cv::Point3f{
        static_cast<float>(line.vector().y()),
        static_cast<float>(line.vector().x()),
        static_cast<float>(-line.vector().y() * line.anyPoint().x()
            - line.vector().x() * line.anyPoint().y())
    };
}

} // namespace

NYT::TNode serialize(const cv::Point3f& pnt)
{
    return NYT::TNode::CreateList()
        .Add(pnt.x)
        .Add(pnt.y)
        .Add(pnt.z);
}

cv::Point3f deserializePoint3f(const NYT::TNode& node)
{
    return cv::Point3f(
        node[0].AsDouble(),
        node[1].AsDouble(),
        node[2].AsDouble()
    );
}

NYT::TNode serialize(const eye::pov::ObjectMatchInfo& obj)  {
    return NYT::TNode::CreateMap()
        (TString(eye::pov::CLUSTER_ID_COLUMN), obj.clusterId)
        ("feature_id", obj.objectId.featureId)
        ("object_id", obj.objectId.detectionId)
        ("predicted_feature_id", obj.predictedFeatureId)
        ("frames_match", yt::serialize(obj.framesMatch))
        ("line", yt::serialize(toCvPoint(obj.epilines.centerEpiline)))
        ("line_lt", yt::serialize(toCvPoint(obj.epilines.topLeftEpiline)))
        ("line_rb", yt::serialize(toCvPoint(obj.epilines.bottomRightEpiline)));
}

eye::pov::ObjectMatchInfo deserializeToObjectMatchInfo(const NYT::TNode& node)
{
    return eye::pov::ObjectMatchInfo {
        .clusterId = node[eye::pov::CLUSTER_ID_COLUMN].AsInt64(),
        .objectId = eye::pov::ObjectId{
            .featureId = node["feature_id"].AsInt64(),
            .detectionId = node["object_id"].AsInt64()},
        .predictedFeatureId = node["predicted_feature_id"].AsInt64(),
        .framesMatch = yt::deserialize<eye::FramesMatchData>(node["frames_match"]),
        .epilines = eye::MatchEpilines{
            .centerEpiline = toGeolibLine(
                deserializePoint3f(node["line"])
            ),
            .topLeftEpiline = toGeolibLine(
                deserializePoint3f(node["line_lt"])
            ),
            .bottomRightEpiline = toGeolibLine(
                deserializePoint3f(node["line_rb"])
            )
        }
    };
}

} // namespace maps::mrc::yt
