from typing import NamedTuple, Literal, Union, Sequence
import vh3


class BuildArcadiaProjectOutput(NamedTuple):
    arcadia_project: vh3.Executable
    sandbox_task_id: vh3.Text


@vh3.decorator.external_operation("https://nirvana.yandex-team.ru/operation/dd4b5735-1ee7-497d-91fc-b81ba8b510fc")
@vh3.decorator.nirvana_output_names(arcadia_project="ARCADIA_PROJECT", sandbox_task_id="SANDBOX_TASK_ID")
def build_arcadia_project(
    *,
    targets: vh3.String,
    arts: vh3.String,
    arcadia_url: vh3.String = "arcadia:/arc/trunk/arcadia",
    arcadia_revision: vh3.Integer = None,
    checkout_arcadia_from_url: vh3.String = None,
    build_type: vh3.Enum[
        Literal["release", "debug", "profile", "coverage", "relwithdebinfo", "valgrind", "valgrind-release"]
    ] = "release",
    arts_source: vh3.String = None,
    result_single_file: vh3.Boolean = False,
    definition_flags: vh3.String = None,
    sandbox_oauth_token: vh3.Secret = None,
    arcadia_patch: vh3.String = None,
    owner: vh3.String = None,
    use_aapi_fuse: vh3.Boolean = True,
    use_arc_instead_of_aapi: vh3.Boolean = True,
    aapi_fallback: vh3.Boolean = False,
    kill_timeout: vh3.Integer = None,
    sandbox_requirements_disk: vh3.Integer = None,
    sandbox_requirements_ram: vh3.Integer = None,
    sandbox_requirements_platform: vh3.Enum[
        Literal[
            "Any",
            "darwin-20.4.0-x86_64-i386-64bit",
            "linux",
            "linux_ubuntu_10.04_lucid",
            "linux_ubuntu_12.04_precise",
            "linux_ubuntu_14.04_trusty",
            "linux_ubuntu_16.04_xenial",
            "linux_ubuntu_18.04_bionic",
            "osx",
            "osx_10.12_sierra",
            "osx_10.13_high_sierra",
            "osx_10.14_mojave",
            "osx_10.15_catalina",
            "osx_10.16_big_sur",
        ]
    ] = None,
    checkout: vh3.Boolean = False,
    clear_build: vh3.Boolean = True,
    strip_binaries: vh3.Boolean = False,
    lto: vh3.Boolean = False,
    thinlto: vh3.Boolean = False,
    musl: vh3.Boolean = False,
    use_system_python: vh3.Boolean = False,
    target_platform_flags: vh3.String = None,
    javac_options: vh3.String = None,
    ya_yt_proxy: vh3.String = None,
    ya_yt_dir: vh3.String = None,
    ya_yt_token_vault_owner: vh3.String = None,
    ya_yt_token_vault_name: vh3.String = None,
    result_rt: vh3.String = None,
    timestamp: vh3.Date = None,
    build_system: vh3.Enum[Literal["ya", "ya_force", "semi_distbuild", "distbuild"]] = "ya"
) -> BuildArcadiaProjectOutput:
    """
    Build Arcadia Project
    Launches YA_MAKE task in Sandbox for provided target and downloads requested artifact.
    :param targets: Target
      [[Multiple targets with ";" are not allowed]]
    :param arts: Build artifact
      [[Multiple artifacts with ";" and custom destination directory with "=" are not allowed]]
    :param arcadia_url: Svn url for arcadia
      [[Should not contain revision]]
    :param arcadia_revision: Arcadia Revision
    :param checkout_arcadia_from_url: Full SVN url for arcadia (Overwrites base URL and revision, use @revision to fix revision)
    :param build_type: Build type
    :param arts_source: Source artifacts (semicolon separated pairs path[=destdir])
      [[Какие файлы из Аркадии поместить в отдельный ресурс (формат тот же, что и у build artifacts)]]
    :param result_single_file: Result is a single file
    :param definition_flags: Definition flags
      [[For example "-Dkey1=val1 ... -DkeyN=valN"]]
    :param sandbox_oauth_token: Sandbox OAuth token
      [[To run task on behalf of specific user]]
      Name of the secret containing oauth token of user the sandbox task should be launched from
    :param arcadia_patch: Apply patch
      [[Diff file rbtorrent, paste.y-t.ru link or plain text. Doc: https://nda.ya.ru/3QTTV4]]
    :param owner: Custom sandbox task owner (should be used only with OAuth token)
      [[OAuth token owner should be a member of sandbox group]]
    :param use_aapi_fuse: Use arcadia-api fuse
    :param use_arc_instead_of_aapi: Use arc fuse instead of aapi
    :param aapi_fallback: Fallback to svn/hg if AAPI services are temporary unavailable
    :param kill_timeout: Kill Timeout (seconds)
    :param sandbox_requirements_disk: Disk requirements in Mb
    :param sandbox_requirements_ram: RAM requirements in Mb
    :param sandbox_requirements_platform: Platform
    :param checkout: Run ya make with --checkout
    :param clear_build: Clear build
    :param strip_binaries: Strip result binaries
    :param lto: Build with LTO
    :param thinlto: Build with ThinLTO
    :param musl: Build with musl-libc
    :param use_system_python: Use system Python to build python libraries
    :param target_platform_flags: Target platform flags (only for cross-compilation)
    :param javac_options: Javac options (semicolon separated)
    :param ya_yt_proxy: YT store proxy
    :param ya_yt_dir: YT store cypress path
    :param ya_yt_token_vault_owner: YT token vault owner
    :param ya_yt_token_vault_name: YT token vault name
    :param result_rt: Result resource type
    :param timestamp: Timestamp
    :param build_system: Build System
    """
    raise NotImplementedError("Write your local execution stub here")


@vh3.decorator.external_operation("https://nirvana.yandex-team.ru/operation/ad1716b8-c211-40af-8c1b-1b1d5c062c05")
@vh3.decorator.nirvana_output_names("output")
@vh3.decorator.nirvana_names_transformer(vh3.name_transformers.snake_to_dash, options=True)
def concatenate_mr_tables(
    *,
    yt_token: vh3.Secret = vh3.Factory(lambda: vh3.context.yt_token),
    mr_account: vh3.String = vh3.Factory(lambda: vh3.context.mr_account),
    tables: Sequence[vh3.MRTable],
    mr_output_path: vh3.String = None,
    ttl: vh3.Integer = 360,
    timestamp: vh3.String = None
) -> vh3.MRTable:
    """
    Concatenate MR Tables

    Concatenates multiple MapReduce tables and outputs a new table

    :param yt_token: YT Token:
      [[ID of Nirvana Secret with YT access token (https://nda.ya.ru/3RSzVU).
    Guide to Nirvana Secrets: https://nda.ya.ru/3RSzWZ]]
      YT OAuth Token.

        [Obtain access token](https://nda.ya.ru/3RSzVU), then [create a Nirvana secret](https://nda.ya.ru/3RSzWZ) and [use it here](https://nda.ya.ru/3RSzWb).
        You can [share the secret](https://nda.ya.ru/3RSzWd) with user(s) and/or a staff group.
    :param mr_account: MR Account:
      [[MR Account Name.
    By default, output tables and directories will be created in some subdirectory of home/<MR Account>/<workflow owner>/nirvana]]
      MR account name (e.g. `rank_machine`) used to build MR output path for this operation.

      See the `mr-output-path` option for more information
    :param tables:
      MapReduce tables, should be on the same cluster
    :param mr_output_path:
      [[Directory for output MR tables and directories.

    Limited templating is supported: `${param["..."]}`, `${meta["..."]}`, `${mr_input["..."]}` (path to input MR *directory*) and `${uniq}` (= unique path-friendly string).

    The default template for `mr-output-path` is

            home[#if param["mr-account"] != meta.owner]/${param["mr-account"]}[/#if]/${meta.owner}/nirvana/${meta.operation_uid}

    If output path does not exist, it will be created.

    Temporary directory, `${mr_tmp}`, is derived from output path in an unspecified way. It is ensured that:
      * It will exist before `job-command` is started
      * It need not be removed manually after execution ends. However, you **should** remove all temporary data created in `${mr_tmp}`, even if your command fails]]
      Directory for output MR tables and directories.

      Limited templating is supported: `${param["..."]}`, `${meta["..."]}`, `${mr_input["..."]}` (path to input MR *directory*) and `${uniq}` (= unique path-friendly string).

      The default template for `mr-output-path` is

              home[#if param["mr-account"] != meta.owner]/${param["mr-account"]}[/#if]/${meta.owner}/nirvana/${meta.operation_uid}

      If output path does not exist, it will be created.

      Temporary directory, `${mr_tmp}`, is derived from output path in an unspecified way. It is ensured that:
        * It will exist before `job-command` is started
        * It need not be removed manually after execution ends. However, you **should** remove all temporary data created in `${mr_tmp}`, even if your command fails
    """
    raise NotImplementedError("Write your local execution stub here")


class MrTableSplitOutput(NamedTuple):
    output1: vh3.MRTable
    output2: vh3.MRTable


@vh3.decorator.external_operation("https://nirvana.yandex-team.ru/operation/aab6a69b-794b-428a-afd9-5df83c28233e")
@vh3.decorator.nirvana_names_transformer(vh3.name_transformers.snake_to_dash, options=True)
def mr_table_split(
    *,
    yt_token: vh3.Secret = vh3.Factory(lambda: vh3.context.yt_token),
    mr_account: vh3.String = vh3.Factory(lambda: vh3.context.mr_account),
    seed: vh3.Integer,
    input: vh3.MRTable,
    yt_pool: vh3.String = None,
    ttl: vh3.Integer = 360,
    max_ram: vh3.Integer = 100,
    ratio: vh3.Number = 0.9
) -> MrTableSplitOutput:
    """
    MR table split

    **Назначение операции**

    Делит таблицу на YT на две. Пропорции указываются опцией **Output1 ratio**.

    **Описание входов**

    Путь к исходной таблице на YT в формате MR Table подается на вход "input".

    **Описание выходов**

    Ссылки на полученные таблицы возвращаются на выход "output1" и "output2".

    **Ограничения**

    :param yt_token: YT Token:
      [[ID of Nirvana Secret with YT access token (https://nda.ya.ru/3RSzVU).
    Guide to Nirvana Secrets: https://nda.ya.ru/3RSzWZ]]
      YT OAuth Token.

        [Obtain access token](https://nda.ya.ru/3RSzVU), then [create a Nirvana secret](https://nda.ya.ru/3RSzWZ) and [use it here](https://nda.ya.ru/3RSzWb).
        You can [share the secret](https://nda.ya.ru/3RSzWd) with user(s) and/or a staff group.
    :param mr_account: MR Account:
      [[MR Account Name.
    By default, output tables and directories will be created in some subdirectory of home/<MR Account>/<workflow owner>/nirvana]]
      MR account name (e.g. `rank_machine`) used to build MR output path for this operation.

      See the `mr-output-path` option for more information
    :param seed: Random seed
    :param yt_pool: YT Pool:
      [[Pool used by YT scheduler. Leave blank to use default pool.
    This option has no effect on YaMR.]]
      Pool used by [YT operation scheduler](https://nda.ya.ru/3Rk4af). Leave this blank to use default pool.
    :param ratio: Output1 ratio
    """
    raise NotImplementedError("Write your local execution stub here")


class CatBoostTrainOutput(NamedTuple):
    eval_result: vh3.TSV
    model_bin: vh3.Binary
    fstr: vh3.TSV
    learn_error_log: vh3.TSV
    test_error_log: vh3.TSV
    training_log_json: vh3.JSON
    plots_html: vh3.HTML
    snapshot_file: vh3.Binary
    tensorboard_log: vh3.Binary
    tensorboard_url: vh3.Text
    training_options_json: vh3.JSON
    borders: vh3.TSV
    plots_png: vh3.Image


@vh3.decorator.external_operation("https://nirvana.yandex-team.ru/operation/a4af82ab-80a6-4370-94f6-c61b4e8fb912")
@vh3.decorator.nirvana_names(
    bootstrap_type="bootstrap_type", od_type="od_type", yt_pool="yt_pool", model_metadata="model_metadata"
)
@vh3.decorator.nirvana_output_names(
    model_bin="model.bin",
    learn_error_log="learn_error.log",
    test_error_log="test_error.log",
    training_log_json="training_log.json",
    plots_html="plots.html",
    tensorboard_log="tensorboard.log",
    training_options_json="training_options.json",
    plots_png="plots.png",
)
@vh3.decorator.nirvana_names_transformer(vh3.name_transformers.snake_to_dash, options=True)
def cat_boost_train(
    *,
    learn: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text],
    ttl: vh3.Integer = 3600,
    gpu_type: vh3.Enum[Literal["NONE", "ANY", "CUDA_3_5", "CUDA_5_2", "CUDA_6_1", "CUDA_7_0", "CUDA_8_0"]] = "NONE",
    restrict_gpu_type: vh3.Boolean = False,
    slaves: vh3.Integer = None,
    cpu_guarantee: vh3.Integer = 1600,
    use_catboost_builtin_quantizer: vh3.Boolean = False,
    loss_function: vh3.Enum[
        Literal[
            "RMSE",
            "Logloss",
            "MAE",
            "CrossEntropy",
            "Quantile",
            "LogLinQuantile",
            "Poisson",
            "MAPE",
            "MultiClass",
            "MultiClassOneVsAll",
            "PairLogit",
            "YetiRank",
            "QueryRMSE",
            "QuerySoftMax",
            "YetiRankPairwise",
            "PairLogitPairwise",
            "QueryCrossEntropy",
            "LambdaMart",
            "StochasticFilter",
            "StochasticRank",
            "MultiRMSE",
            "Huber",
            "Lq",
            "Expectile",
        ]
    ] = "Logloss",
    loss_function_param: vh3.String = None,
    iterations: vh3.Integer = 1000,
    learning_rate: vh3.Number = None,
    ignored_features: vh3.String = None,
    l2_leaf_reg: vh3.Number = None,
    random_strength: vh3.Number = None,
    bootstrap_type: vh3.Enum[Literal["Poisson", "Bayesian", "Bernoulli", "MVS", "No"]] = None,
    bagging_temperature: vh3.Number = None,
    subsample: vh3.Number = None,
    sampling_unit: vh3.Enum[Literal["Object", "Group"]] = None,
    rsm: vh3.Number = None,
    leaf_estimation_method: vh3.Enum[Literal["Newton", "Gradient", "Exact"]] = None,
    leaf_estimation_iterations: vh3.Integer = None,
    depth: vh3.Integer = None,
    seed: vh3.Integer = 0,
    create_tensorboard: vh3.Boolean = False,
    use_best_model: vh3.Boolean = True,
    od_type: vh3.Enum[Literal["IncToDec", "Iter"]] = None,
    od_pval: vh3.Number = None,
    eval_metric: vh3.String = None,
    custom_metric: vh3.String = None,
    one_hot_max_size: vh3.Number = None,
    feature_border_type: vh3.Enum[
        Literal["Median", "GreedyLogSum", "UniformAndQuantiles", "MinEntropy", "MaxLogSum"]
    ] = None,
    per_float_feature_quantization: vh3.String = None,
    border_count: vh3.Integer = None,
    target_border: vh3.Number = None,
    has_header: vh3.Boolean = False,
    delimiter: vh3.String = None,
    ignore_csv_quoting: vh3.Boolean = False,
    cv_type: vh3.Enum[Literal["Classical", "Inverted"]] = None,
    cv_fold_index: vh3.Integer = None,
    cv_fold_count: vh3.Integer = None,
    fstr_type: vh3.Enum[Literal["PredictionValuesChange", "LossFunctionChange"]] = None,
    prediction_type: vh3.Enum[Literal["RawFormulaVal", "Probability", "Class"]] = None,
    output_columns: vh3.MultipleStrings = (),
    max_ctr_complexity: vh3.Integer = None,
    ctr_leaf_count_limit: vh3.Integer = None,
    simple_ctr: vh3.String = None,
    combinations_ctr: vh3.String = None,
    text_processing: vh3.String = None,
    args: vh3.String = None,
    yt_token: vh3.Secret = "fml_public_yt_token",
    yt_pool: vh3.String = None,
    model_metadata: vh3.MultipleStrings = (),
    inner_options_override: vh3.String = None,
    boost_from_average: vh3.Enum[Literal["True", "False"]] = "False",
    job_core_yt_token: vh3.Secret = "",
    force_distributed_fstr_evaluation: vh3.Boolean = True,
    test: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    cd: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    pairs: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    test_pairs: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    learn_group_weights: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    test_group_weights: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    catboost_binary: Union[vh3.Binary, vh3.Executable] = None,
    params_file: Union[vh3.JSON, vh3.Text] = None,
    snapshot_file: vh3.Binary = None,
    borders: Union[vh3.File, vh3.TSV, vh3.Text] = None,
    baseline_model: vh3.Binary = None,
    learn_baseline: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    test_baseline: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    pool_metainfo: vh3.JSON = None
) -> CatBoostTrainOutput:
    """
    CatBoost: Train

    CatBoost is a machine learning algorithm that uses gradient boosting on decision trees

    [Documentation](https://doc.yandex-team.ru/english/ml/catboost/doc/nirvana-operations/catboost__nirvana__train-catboost.html)

    [Change log](https://a.yandex-team.ru/arc/trunk/arcadia/ml/nirvana/catboost/train/change_log.md)

    :param learn:
      Training dataset. [Documentaion](https://tech.yandex.com/catboost/doc/dg/concepts/input-data_values-file-docpage/).
    :param restrict_gpu_type:
      [[Request gpu type exactly as specified]]
      [Doc](https://wiki.yandex-team.ru/nirvana/manual/createoperation/#jobprocessor)
    :param slaves: Number of slaves
      [[or null for auto selection]]
    :param use_catboost_builtin_quantizer: Use CatBoost builtin quantizer
      If checked, do not create quantized pool on YT
    :param loss_function: Loss function
    :param loss_function_param: Loss function param
      [[e.g. `border=0.7` for Logloss]]
    :param iterations: Iterations
    :param learning_rate: Learning rate
      [[0.0314 is a good choice]]
    :param ignored_features: Ignored features
      [[List feature indices to disregard, e.g. `0:5-13:2`]]
    :param l2_leaf_reg: L2 leaf regularization coeff
      [[Any positive value]]
    :param random_strength: Random strength
      [[Coeff at std deviation of score randomization.
    NOT supported for: QueryCrossEntropy, YetiRankPairwise, PairLogitPairwise]]
    :param bootstrap_type: Bootstrap type
      [[Method for sampling the weights of objects.]]
    :param bagging_temperature: Bagging temperature
      [[For Bayesian bootstrap]]
    :param subsample: Sample rate for bagging
    :param sampling_unit: Sampling unit
    :param rsm: Rsm
      [[CPU-only. A value in range (0; 1]]]
    :param leaf_estimation_method: Leaf estimation method
    :param leaf_estimation_iterations: Leaf estimation iterations
    :param depth: Max tree depth
      [[Suggested < 10 on GPU version]]
    :param seed: Random seed
    :param create_tensorboard: Create tensorboard
    :param use_best_model: Use best model
    :param od_type: Overfitting detector type
    :param od_pval: Auto stop PValue
      [[For IncToDec: a small value, [1e-12...1e-2], is a good choice.
    Do not use this this with Iter type overfitting detector.]]
    :param eval_metric: Eval metric (for OD and best model)
      [[Metric name with params, e.g. `Quantile:alpha=0.3`]]
    :param custom_metric: Custom metric
      [[Written during training. E.g. `Quantile:alpha=0.1`]]
    :param one_hot_max_size: One-hot max size
      [[Use one-hot encoding for cat features with number of values <= this size.
    Ctrs are not calculated for such features.]]
      If parameter is specified than features with no more than specified value different values will be converted to float
      features using one-hot encoding. No ctrs will be calculated on this features.
    :param feature_border_type: Border type for num-features
      "feature-border-type", "Should be one of: Median, GreedyLogSum, UniformAndQuantiles, MinEntropy, MaxLogSum"
    :param border_count: Border count for num-features
      [[Must be < 256 for GPU. May be large on CPU.]]
    :param target_border: Border for target binarization
    :param has_header: Has header
    :param delimiter: Delimiter
    :param cv_type: CV type
      [[Classical: train on many, test on 1, Inverted: train on 1, test on many]]
      [Doc](https://doc.yandex-team.ru/ml/catboost/doc/concepts/cli-reference_cross-validation.html)
    :param cv_fold_index: CV fold index
    :param cv_fold_count: CV fold count
    :param fstr_type: Feature importance calculation type
    :param prediction_type: Prediction type
      [[only CPU version]]
    :param output_columns: Output columns
      [[Columns for eval file, e.g. SampleId RawFormulaVal #5:FeatureN
    > https://nda.ya.ru/3VodMf]]
      [Doc](https://doc.yandex-team.ru/ml/catboost/doc/concepts/cli-reference_calc-model.html)
    :param max_ctr_complexity: Max ctr complexity
      [[Maximum number of cat-features in ctr combinations.]]
    :param ctr_leaf_count_limit: ctr leaf count limit (CPU only)
    :param text_processing:  text-processing
    :param args: additional arguments
    :param model_metadata: Model metadata key-value pairs
      [[Key-value pairs, quote key and value for reasonable results]]
    :param inner_options_override: Override options for inner train cube
      [[JSON string, e.g. {"master-max-ram":250000}]]
      Normally this option shall be null. Used to debug/override resource requirements
    :param job_core_yt_token:
      YT token to use when storing coredump

      More info: https://st.yandex-team.ru/YTADMINREQ-25127 https://wiki.yandex-team.ru/nirvana/vodstvo/processory/Job-processor/#sborcoredump-ov
        https://yt.yandex-team.ru/docs/problems/mapreduce_debug#poluchenie-cuda-core-dump-iz-dzhobov-operacii
    :param force_distributed_fstr_evaluation: Evaluate LossFunctionChange fstr on YT
    :param test:
      Testing dataset. [Documentaion](https://tech.yandex.com/catboost/doc/dg/concepts/input-data_values-file-docpage/).
    :param cd:
      Column description. [Documentation](https://tech.yandex.com/catboost/doc/dg/concepts/input-data_column-descfile-docpage/).
    :param pairs:
      Training pairs. [Documentation](https://tech.yandex.com/catboost/doc/dg/concepts/input-data_pairs-description-docpage/).
    :param test_pairs:
      Testing pairs. [Documentation](https://tech.yandex.com/catboost/doc/dg/concepts/input-data_pairs-description-docpage/).
    :param learn_group_weights:
      Training query weights.

      Format:
      ```
      <group_id>\t<weight>\n
      ```

      Example:
      ```
      id1\t0.5
      id2\t0.7
      ```
    :param test_group_weights:
      Testing query weights.

      Format:
      ```
      <group_id>\t<weight>\n
      ```

      Example:
      ```
      id1\t0.5
      id2\t0.7
      ```
    :param catboost_binary:
      CatBoost binary.
      Build command:
      ```
      ya make trunk/arcadia/catboost/app/ -r --checkout -DHAVE_CUDA=yes -DCUDA_VERSION=9.0
      ```
    :param params_file:
      JSON file that contains the training parameters, for example:
      ```
      {
          "loss_function": "Logloss",
          "iterations": 400
      }
      ```
    :param snapshot_file:
      Snapshot file. Used for recovering training after an interruption.
    :param borders:
      File with borders description.

      Format:
      ```
      <feature_index(zero-based)>\t<border_value>\n
      ```

      Example:
      ```
      0\t0.5
      0\t1.0
      1\t0.7
      ```

      Note: File should be sorted by first column.
    :param learn_baseline:
      Baseline for learn
    :param test_baseline:
      Baseline for test (the first one, if many)
    :param pool_metainfo:
      Metainfo file with feature tags.
    """
    raise NotImplementedError("Write your local execution stub here")


class CatBoostModelAnalysisOutput(NamedTuple):
    result: vh3.TSV
    output_plot_html: vh3.HTML
    plots: vh3.Image


@vh3.decorator.external_operation("https://nirvana.yandex-team.ru/operation/d246487d-edff-44a0-a456-dfb653986ef6")
@vh3.decorator.nirvana_names(yt_pool="yt_pool", model_bin="model.bin")
@vh3.decorator.nirvana_output_names(output_plot_html="output_plot.html")
@vh3.decorator.nirvana_names_transformer(vh3.name_transformers.snake_to_dash, options=True)
def cat_boost_model_analysis(
    *,
    pool: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text],
    model_bin: vh3.Binary,
    ttl: vh3.Integer = 3600,
    cpu_guarantee: vh3.Integer = 1600,
    analyse_mode: vh3.Enum[Literal["fstr", "ostr", "fstat", "prediction_diff"]] = "fstr",
    fstr_type: vh3.Enum[
        Literal[
            "FeatureImportance",
            "PredictionValuesChange",
            "LossFunctionChange",
            "InternalFeatureImportance",
            "Interaction",
            "InternalInteraction",
            "ShapValues",
        ]
    ] = "FeatureImportance",
    class_names: vh3.MultipleStrings = (),
    update_method: vh3.Enum[Literal["SinglePoint", "TopKLeaves", "AllPoints"]] = None,
    top: vh3.Integer = None,
    features_id_for_stats: vh3.String = None,
    docpair_ids: vh3.MultipleStrings = (),
    docpair_indexes: vh3.MultipleStrings = (),
    top_size: vh3.Integer = 5,
    has_header: vh3.Boolean = False,
    delimiter: vh3.String = None,
    ignore_csv_quoting: vh3.Boolean = False,
    yt_token: vh3.Secret = "fml_public_yt_token",
    yt_pool: vh3.String = None,
    max_ram: vh3.Integer = 5000,
    max_disk: vh3.Integer = 50000,
    force_distributed_fstr_evaluation: vh3.Boolean = True,
    test: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    cd: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    pairs: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    test_pairs: Union[vh3.Binary, vh3.File, vh3.MRTable, vh3.TSV, vh3.Text] = None,
    catboost_external_binary: Union[vh3.Binary, vh3.Executable] = None
) -> CatBoostModelAnalysisOutput:
    """
    CatBoost: Model Analysis

    [Documentation](https://doc.yandex-team.ru/english/ml/catboost/doc/nirvana-operations/catboost__nirvana__model-analysis.html).
    [Change log](https://a.yandex-team.ru/arc/trunk/arcadia/ml/nirvana/catboost/analyse_model/change_log.md).

    :param analyse_mode: Analysis mode
    :param fstr_type: Feature importance type (fstr only)
    :param class_names: Class names (fstr only)
      [[list of names for classes.]]
    :param update_method: Update method (ostr only)
    :param top: Top size in TopKLeaves update method (ostr only)
    :param features_id_for_stats: List of features indices for which you want to get statistics. (fstat only)
      [[List features indices to for which you want to get statistics, e.g. `0:5-13:2:Age`
    Pass null to get statistics for all features]]
    :param docpair_ids: docpair-ids (prediction_diff only)
    :param docpair_indexes: docpair-indexes (prediction_diff only)
    :param top_size: top-size (prediction_diff only)
    :param has_header: has header
    :param force_distributed_fstr_evaluation: Evaluate LossFunctionChange fstr on YT
    :param test:
      ostr only
    """
    raise NotImplementedError("Write your local execution stub here")
