import json
import argparse
import itertools


def load_clusters(path):
    data = json.load(open(path))
    clusters = []
    for cluster in data['clusters']:
        clusters.append(cluster['objects'])
    return clusters


def make_matches(clusters):
    matches = []
    for cluster in clusters:
        for object1, object2 in itertools.combinations(cluster, 2):
            if object1['feature_id'] == object2['feature_id']:
                raise Exception("There is match from one feature: " + str(object1) + " <-> " + str(object2))
            foid1 = (object1['feature_id'], object1['object_id'])
            foid2 = (object2['feature_id'], object2['object_id'])
            matches.append((foid1, foid2))
    return matches


def save_matches(matches, path):
    matches_by_feature_ids = {}
    for foid1, foid2 in matches:
        feature_id1, object_id1 = foid1
        feature_id2, object_id2 = foid2

        if feature_id1 < feature_id2:
            match_key = (feature_id1, feature_id2)
            match_value = (object_id1, object_id2)
        else:
            match_key = (feature_id2, feature_id1)
            match_value = (object_id2, object_id1)

        if match_key not in matches_by_feature_ids:
            matches_by_feature_ids[match_key] = []
        matches_by_feature_ids[match_key].append(match_value)

    json_matches = []
    for feature_ids, object_matches in matches_by_feature_ids.items():
        feature_id1, feature_id2 = feature_ids
        item = {}
        item['feature_id_1'] = feature_id1
        item['feature_id_2'] = feature_id2
        item['matches'] = []
        for object_id1, object_id2 in object_matches:
            item['matches'].append({
                'object_id_1': object_id1,
                'object_id_2': object_id2
            })
        json_matches.append(item)

    with open(path, 'w') as f:
        json.dump({'features_pairs': json_matches}, f, indent=2)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--clusters', required=True,
                        help='Input json file with clusters')
    parser.add_argument('--matches', required=True,
                        help='Output json file with matches')
    args = parser.parse_args()

    print('Load clusters from {}'.format(args.clusters))
    clusters = load_clusters(args.clusters)
    print('Loaded {} clusters'.format(len(clusters)))

    print('Create matches from clusters')
    matches = make_matches(clusters)
    print('Created {} matches'.format(len(matches)))

    print('Save matches to {}'.format(args.matches))
    save_matches(matches, args.matches)
    print('Done!')


if __name__ == '__main__':
    main()
