# -*- coding: utf-8 -*-

from .feature import Orientation, Rotation

import json


class Box:
    def __init__(self, min_x, min_y, max_x, max_y):
        assert min_x < max_x
        assert min_y < max_y

        self.min_x = min_x
        self.min_y = min_y
        self.max_x = max_x
        self.max_y = max_y

    @staticmethod
    def from_dict(dictionary):
        min_x = min(dictionary[0][0], dictionary[1][0])
        min_y = min(dictionary[0][1], dictionary[1][1])

        max_x = max(dictionary[0][0], dictionary[1][0])
        max_y = max(dictionary[0][1], dictionary[1][1])

        return Box(min_x, min_y, max_x, max_y)

    def to_list(self):
        return [[self.min_x, self.min_y], [self.max_x, self.max_y]]

    def rotate90cw(self, original_image_size):
        _, height = original_image_size
        return Box(height - self.max_y, self.min_x, height - self.min_y, self.max_x)

    def rotate180cw(self, original_image_size):
        width, height = original_image_size
        return Box(width - self.max_x, height - self.max_y, width - self.min_x, height - self.min_y)

    def rotate270cw(self, original_image_size):
        width, _ = original_image_size
        return Box(self.min_y, width - self.max_x, self.max_y, width - self.min_x)

    def max_size(self):
        return max(self.max_x - self.min_x, self.max_y - self.min_y)


class Detection:
    def __init__(self, id, box, type):
        self.id = id
        self.box = box
        self.type = type

    @staticmethod
    def from_dict(row):
        return Detection(row['object_id'], Box.from_dict(row['bbox']), row['type'])

    def to_dict(self):
        return {'object_id': self.id, 'bbox': self.box.to_list(), 'type': self.type}

    def transform_by_feature_orientation(self, feature):
        assert not feature.orientation.flip

        size = feature.original_size()
        rotation = feature.orientation.rotation
        box = self.box

        if rotation == Rotation.CW_90:
            box = box.rotate90cw(size)
        elif rotation == Rotation.CW_180:
            box = box.rotate180cw(size)
        elif rotation == Rotation.CW_270:
            box = box.rotate270cw(size)

        return Detection(self.id, box, self.type)

    def revert_by_feature_orientation(self, feature):
        assert not feature.orientation.flip

        size = feature.original_size()
        rotation = feature.orientation.rotation
        box = self.box

        if rotation == Rotation.CW_90:
            box = box.rotate270cw(size)
        elif rotation == Rotation.CW_180:
            box = box.rotate180cw(size)
        elif rotation == Rotation.CW_270:
            box = box.rotate90cw(size)

        return Detection(self.id, box, self.type)


# return right oriented detections
def load_detections(object_file, feature_by_id):
    detections_by_feature_id = {}

    for feature_objects in json.load(object_file)['features_objects']:
        feature_id = feature_objects['feature_id']
        if feature_id not in feature_by_id:
            continue

        feature = feature_by_id[feature_id]

        orientation = Orientation.from_exif(feature_objects['orientation'])
        assert orientation == feature.orientation

        detections = [Detection.from_dict(object) for object in feature_objects['objects']]

        if orientation != Orientation.default():
            detections = [detection.transform_by_feature_orientation(feature) for detection in detections]

        detections_by_feature_id[feature_id] = detections

    return detections_by_feature_id
