#include <maps/wikimap/mapspro/services/mrc/libs/unittest/include/yandex/maps/mrc/unittest/database_fixture.h>
#include <maps/wikimap/mapspro/services/mrc/libs/unittest/include/yandex/maps/mrc/unittest/unittest_config.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/load.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/playground/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/playground/include/playground.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/playground/include/geo_id_provider.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/secure_config.h>

#include <maps/libs/common/include/exception.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>

#include <fstream>
#include <optional>
#include <utility>

namespace db = maps::mrc::db;
namespace eye = maps::mrc::eye;
namespace unittest = maps::mrc::unittest;

int main(int argc, const char** argv) try {
    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser("playground");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto mrcConfigPath = parser.string("mrc-config")
            .help("path to mrc config");

    auto featureYtTablePath = parser.string("features")
            .required()
            .help("path to yt table with features");

    auto importDetectionsJsonPath = parser.string("import-detections")
            .help("path to import json file with sign detections");

    auto importMatchesJsonPath = parser.string("import-matches")
            .help("path to import json file with matches");

    auto exportDetectionJsonPath = parser.string("export-detections")
            .help("path to export json file with sign detections");

    auto exportClusterJsonPath = parser.string("export-clusters")
            .help("path to export json file with clusters");

    auto exportMatchJsonPath = parser.string("export-matches")
            .help("path to export json file with matches");

    auto exportClusterGeoJsonPath = parser.string("export-clusters-geojson")
            .help("path to export geojson file with clusters");

    parser.parse(argc, const_cast<char**>(argv));

    if (importMatchesJsonPath.defined()) {
        REQUIRE(importDetectionsJsonPath.defined(), "Required imported detections");
    }


    // Start local postgres
    unittest::WithUnittestConfig<unittest::DatabaseFixture> fixture;

    // For access to feature images
    const auto config = maps::mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);
    maps::mrc::eye::SecureConfig::initialize(config);

    auto yt = NYT::CreateClient("hahn");
    auto mds = config.makeMdsClient();
    auto loader = eye::FrameLoader::fromConfig(config);
    auto geoIdProvider = eye::makeFakeGeoIdProvider();

    eye::PlaygroundConfig playgroundConfig;
    playgroundConfig.mrcPool = std::addressof(fixture.pool());
    playgroundConfig.yt = yt.Get();
    playgroundConfig.mds = std::addressof(mds);
    playgroundConfig.frameLoader = std::addressof(loader);
    playgroundConfig.geoIdProvider = geoIdProvider;

    eye::Playground playground(playgroundConfig);

    playground.uploadMrcFeature(featureYtTablePath)
        .runImportMrc();

    if (importDetectionsJsonPath.defined()) {
        playground.uploadSignRecognition(importDetectionsJsonPath);
    } else {
        playground.runDetectSign();
    }

    playground.runSyncDetection();

    if (importMatchesJsonPath.defined()) {
        INFO() << "Use matches import";
        playground.runObjectManager(importDetectionsJsonPath, importMatchesJsonPath);
    } else {
        playground.runObjectManager();
    }

    if (exportDetectionJsonPath.defined()) {
        INFO() << "Export detections to json file " << exportDetectionJsonPath;
        playground.dumpDetectionsAsJson(exportDetectionJsonPath, db::eye::DetectionType::Sign);
    }

    if (exportClusterJsonPath.defined()) {
        INFO() << "Export clusters to json file " << exportClusterJsonPath;
        playground.dumpClustersAsJson(exportClusterJsonPath, db::eye::DetectionType::Sign);
    }

    if (exportMatchJsonPath.defined()) {
        INFO() << "Export matches to json file " << exportMatchJsonPath;
        playground.dumpMatchesAsJson(exportMatchJsonPath);
    }

    if (exportClusterGeoJsonPath.defined()) {
        INFO() << "Export clusters to " << exportClusterGeoJsonPath;
        playground.dumpClustersAsGeoJson(exportClusterGeoJsonPath, db::eye::DetectionType::Sign);
    }

    return EXIT_SUCCESS;
} catch (const maps::Exception& ex) {
    FATAL() << "Failed: " << ex;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Failed: " << ex.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Unknown error!";
}
