#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/eye/experiments/signs_map/toloka/matches/lib/include/serialization.h>
#include <maps/wikimap/mapspro/services/mrc/eye/experiments/signs_map/toloka/matches/lib/tests/fixture.h>

namespace maps::mrc::eye::tests {

TEST_F(Fixture, object_to_json)
{
    const Object object{1, "sign", common::ImageBox(1, 2, 3, 4)};
    const json::Value expected = json::Value::fromString(R"({
        "object_id": 1,
        "type": "sign",
        "bbox": [[1, 2], [3, 4]]
    })");

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        toJson(object, b);
    };

    json::Value value = json::Value::fromStream(ss);

    EXPECT_EQ(value, expected);
}

TEST_F(Fixture, feature_objects_to_json)
{
    const Objects objects{
        common::ImageOrientation::fromExif(1),
        {{1, "sign", common::ImageBox(1, 2, 3, 4)}}
    };
    const json::Value expected = json::Value::fromString(R"({
        "feature_id":2,
        "url": "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/2/image?orientation=1",
        "heading": 90,
        "position": [10,50.00000000000523],
        "objects": [
            {
                "object_id":1, "type": "sign", "bbox": [[1,2],[3,4]]
            }
        ]
    })");

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        toJson(features[1], objects, b);
    };

    json::Value value = json::Value::fromStream(ss);

    EXPECT_EQ(value, expected);
}

TEST_F(Fixture, golden_task_to_json)
{
    const Objects objects1{
        common::ImageOrientation::fromExif(1),
        {{1, "sign1", common::ImageBox(1, 2, 3, 4)}}
    };
    const Objects objects2{
        common::ImageOrientation::fromExif(1),
        {{2, "sign2", common::ImageBox(2, 3, 4, 5)}}
    };
    const Matches matches{
        {1, 2},
        {2, 3}
    };
    const Task task(
        features[1], objects1,
        features[2], objects2,
        matches
    );

    const json::Value expected = json::Value::fromString(R"({
        "knownSolutions": [
            {
                "weight": 1,
                "outputValues": {
                    "status": "ok",
                    "matches": [
                        {
                            "object_id_1": 1,
                            "object_id_2": 2
                        },
                        {
                            "object_id_1": 2,
                            "object_id_2": 3
                        }
                    ]
                }
            }
        ],
        "inputValues": {
            "features": [
                {
                    "feature_id": 2,
                    "url": "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/2/image?orientation=1",
                    "heading":90,
                    "position": [10,50.00000000000523],
                    "objects": [
                        {
                            "object_id":1, "type": "sign1", "bbox":[[1,2],[3,4]]
                        }
                    ]
                },
                {
                    "feature_id": 3,
                    "url": "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/3/image?orientation=1",
                    "heading": 67,
                    "position": [12.000000000000002,86.99999999999957],
                    "objects": [
                        {
                            "object_id": 2, "type": "sign2", "bbox": [[2,3],[4,5]]
                        }
                    ]
                }
            ]
        }
    })");

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        toJson(task, b);
    };

    json::Value value = json::Value::fromStream(ss);

    EXPECT_EQ(value, expected);
}

TEST_F(Fixture, real_task_to_json)
{
    const Objects objects1{
        common::ImageOrientation::fromExif(1),
        {{1, "sign1", common::ImageBox(1, 2, 3, 4)}}
    };
    const Objects objects2{
        common::ImageOrientation::fromExif(1),
        {{2, "sign2", common::ImageBox(2, 3, 4, 5)}}
    };
    const Task task(
        features[1], objects1,
        features[2], objects2
    );
    const json::Value expected = json::Value::fromString(R"({
        "inputValues": {
            "features": [
                {
                    "feature_id": 2,
                    "url": "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/2/image?orientation=1",
                    "heading":90,
                    "position": [10,50.00000000000523],
                    "objects": [
                        {
                            "object_id":1, "type": "sign1", "bbox":[[1,2],[3,4]]
                        }
                    ]
                },
                {
                    "feature_id": 3,
                    "url": "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/3/image?orientation=1",
                    "heading": 67,
                    "position": [12.000000000000002,86.99999999999957],
                    "objects": [
                        {
                            "object_id": 2, "type": "sign2", "bbox": [[2,3],[4,5]]
                        }
                    ]
                }
            ]
        }
    })");

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        toJson(task, b);
    };

    json::Value value = json::Value::fromStream(ss);

    EXPECT_EQ(value, expected);
}

TEST_F(Fixture, dump_tasks)
{
    const Objects objects1{
        common::ImageOrientation::fromExif(1),
        {{1, "sign1", common::ImageBox(1, 2, 3, 4)}}
    };
    const Objects objects2{
        common::ImageOrientation::fromExif(1),
        {{2, "sign2", common::ImageBox(2, 3, 4, 5)}}
    };
    const Objects objects3{
        common::ImageOrientation::fromExif(1),
        {{3, "sign3", common::ImageBox(3, 4, 5, 6)}}
    };
    const Tasks tasks{
        Task(
            features[1], objects1,
            features[2], objects2
        ),
        Task(
            features[2], objects2,
            features[3], objects3
        ),
    };
    const json::Value expected = json::Value::fromString(R"([
        {
            "inputValues": {
                "features": [
                    {
                        "feature_id": 2,
                        "url": "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/2/image?orientation=1",
                        "heading": 90,
                        "position": [10,50.00000000000523],
                        "objects": [
                            {
                                "object_id": 1, "type": "sign1", "bbox": [[1,2],[3,4]]
                            }
                        ]
                    },
                    {
                        "feature_id": 3,
                        "url": "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/3/image?orientation=1",
                        "heading": 67,
                        "position": [12.000000000000002,86.99999999999957],
                        "objects": [
                            {
                                "object_id":2, "type": "sign2", "bbox":[[2,3],[4,5]]
                            }
                        ]
                    }
                ]
            }
        },
        {
            "inputValues": {
                "features": [
                    {
                        "feature_id": 3,
                        "url": "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/3/image?orientation=1",
                        "heading": 67,
                        "position": [12.000000000000002,86.99999999999957],
                        "objects": [
                            {
                                "object_id": 2, "type": "sign2", "bbox": [[2,3],[4,5]]
                            }
                        ]
                    },
                    {
                        "feature_id": 4,
                        "url": "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/4/image?orientation=1",
                        "heading": 67,
                        "position": [12.000000000000002,86.99999999999957],
                        "objects": [
                            {
                                "object_id":3, "type": "sign3", "bbox": [[3,4],[5,6]]
                            }
                        ]
                    }
                ]
            }
        }
    ])");

    std::stringstream ss;
    dumpTasks(tasks, ss);

    json::Value value = json::Value::fromStream(ss);

    EXPECT_EQ(value, expected);
}

} // namespace maps::mrc::eye::tests
