#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/io.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/schema.h>

#include <maps/libs/common/include/exception.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/sql_chemistry/include/batch_load.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id.h>

#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/builder.h>

#include <fstream>
#include <optional>
#include <utility>

using namespace maps;
using namespace maps::mrc;
using namespace maps::mrc::eye;

void dumpFeatures(
    pqxx::transaction_base& txn,
    NYT::IClientPtr client,
    db::TIds featureIds,
    const std::string& featureTableYtPath)
{
    auto filter = db::table::Feature::id.in(std::move(featureIds));
    sql_chemistry::BatchLoad<db::table::Feature> batch(1000, filter);

    client->Create(TString(featureTableYtPath), NYT::NT_TABLE,
        NYT::TCreateOptions()
            .Recursive(true)
            .Attributes(NYT::TNode()(
                "schema", yt::getSchemaOf<db::Feature>().ToNode())
            )
    );

    while (batch.next(txn)) {
        yt::saveToTable(
            *client,
            NYT::TRichYPath(TString(featureTableYtPath))
                .Append(true),
            batch.begin(), batch.end()
        );
    }
}

void dumpDetections(
    const db::IdTo<db::eye::Detections>& detectionsByFeatureId,
    const db::IdTo<db::eye::Frame>& frameByFeatureId,
    const std::string& detectionsPath)
{
    std::ofstream ofs(detectionsPath);

    json::Builder builder(ofs);

    builder << [&](json::ObjectBuilder b) {
        b["features_objects"] << [&](json::ArrayBuilder b) {
            for (const auto& featureIdAndDetections : detectionsByFeatureId) {
                db::TId featureId = featureIdAndDetections.first;
                const db::eye::Detections& detections = featureIdAndDetections.second;
                const db::eye::Frame& frame = frameByFeatureId.at(featureId);
                int orientation = static_cast<int>(frame.orientation());

                b << [&](json::ObjectBuilder b) {
                    b["feature_id"] = featureId;
                    b["orientation"] = orientation;

                    b["objects"] << [&](json::ArrayBuilder b) {
                        for (const db::eye::Detection& detection : detections) {
                            db::eye::DetectedSign attrs = detection.attrs<db::eye::DetectedSign>();
                            b << [&](json::ObjectBuilder b) {
                                b["object_id"] = detection.id();
                                b["type"] = traffic_signs::toString(attrs.type);
                                b["bbox"] << [&](json::ArrayBuilder b) {
                                    b << [&](json::ArrayBuilder b) {
                                        b << attrs.box.minX();
                                        b << attrs.box.minY();
                                    };
                                    b << [&](json::ArrayBuilder b) {
                                        b << attrs.box.maxX();
                                        b << attrs.box.maxY();
                                    };
                                };
                            };
                        }
                    };
                };
            }
        };
    };

    ofs.close();
}

int main(int argc, const char** argv) try {
    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser("Upload mrc features to yt table");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto mrcConfigPath = parser.string("mrc-config")
            .help("path to mrc config");

    auto detectionIdsPath = parser.string("detection-ids")
            .required()
            .help("path to txt file with detection ids");

    auto featureTableYTPath = parser.string("feature-yt-table")
            .required()
            .help("path to yt table with features");

    auto detectionsPath = parser.string("detections-path")
            .required()
            .help("path to json file with detections");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Loading detection ids";
    db::TIds detectionIds;
    std::ifstream ifs(detectionIdsPath);
    while (!ifs.eof()) {
        db::TId detectionId;
        ifs >> detectionId;

        detectionIds.push_back(detectionId);
    }
    INFO() << "Loaded " << detectionIds.size() << " detections";

    NYT::IClientPtr client = NYT::CreateClient("hahn");
    const auto config = maps::mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);
    wiki::common::PoolHolder mrc(config.makePoolHolder());
    auto txn = mrc.pool().slaveTransaction();

    INFO() << "Loading detections";
    db::eye::Detections detections = db::eye::DetectionGateway(*txn).loadByIds(detectionIds);
    INFO() << "Loaded " << detections.size() << " detections";

    db::TIds groupIds;
    db::IdTo<db::eye::Detections> detectionsByGroupId;
    for (const auto& detection : detections) {
        groupIds.push_back(detection.groupId());
        detectionsByGroupId[detection.groupId()].push_back(detection);
    }

    INFO() << "Loading detection groups";
    db::eye::DetectionGroups groups = db::eye::DetectionGroupGateway(*txn).loadByIds(groupIds);
    INFO() << "Loaded " << groups.size() << " groups";

    db::TIds frameIds;
    db::IdTo<db::eye::Detections> detectionsByFrameId;
    for (const auto& group : groups) {
        frameIds.push_back(group.frameId());
        detectionsByFrameId[group.frameId()] = detectionsByGroupId.at(group.id());
    }

    INFO() << "Loading frames";
    db::IdTo<db::eye::Frame> frameById = byId(
        db::eye::FrameGateway(*txn).loadByIds(frameIds)
    );

    INFO() << "Loading feature to frame";
    std::vector<db::eye::FeatureToFrame> featureToFrameVec
        = db::eye::FeatureToFrameGateway(*txn).load(
            db::eye::table::FeatureToFrame::frameId.in(frameIds)
        );
    INFO() << "Loaded " << featureToFrameVec.size() << " feature to frame";

    db::TIds featureIds;
    db::IdTo<db::eye::Detections> detectionsByFeatureId;
    db::IdTo<db::eye::Frame> frameByFeatureId;
    for (const auto& featureToFrame : featureToFrameVec) {
        featureIds.push_back(featureToFrame.featureId());
        detectionsByFeatureId[featureToFrame.featureId()]
            = detectionsByFrameId.at(featureToFrame.frameId());
        frameByFeatureId.emplace(
            featureToFrame.featureId(),
            frameById.at(featureToFrame.frameId())
        );
    }
    INFO() << "Loaded " << featureIds.size() << " feature ids";

    INFO() << "Dumping features: " << featureTableYTPath;
    dumpFeatures(*txn, client, featureIds, featureTableYTPath);
    INFO() << "Done";

    INFO() << "Dumping detections";
    dumpDetections(
        detectionsByFeatureId,
        frameByFeatureId,
        detectionsPath
    );
    INFO() << "Done";

    return EXIT_SUCCESS;
} catch (const maps::Exception& ex) {
    FATAL() << "Failed: " << ex;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Failed: " << ex.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Unknown error!";
}
