#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/std.h>
#include <maps/libs/ymapsdf/include/rd.h>
#include <maps/libs/common/include/exception.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/url.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/enums.h>

#include <opencv2/opencv.hpp>

using namespace maps::mrc;

namespace {

struct Edge {
    uint64_t edgeId;
    uint64_t edgeCommitId;

    int speedLimitF;
    int speedLimitT;
    int speedLimitTruckF;
    int speedLimitTruckT;
    std::optional<db::TId> speedLimitObjectIdF;
    std::optional<db::TId> speedLimitObjectIdT;
    std::optional<db::TId> speedLimitTruckObjectIdF;
    std::optional<db::TId> speedLimitTruckObjectIdT;
    std::vector<maps::geolib3::Point2> points;

    int oneWay = 3; //maps/libs/ymapsdf/include/rd.h enum class Direction ; Forward = fromNodeId -> toNodeId, Backward = toNodeId -> fromNodeId
    int fc = 0;
};
using Edges = std::vector<Edge>;
using MapEdges = std::map<uint64_t, Edge>;
struct Error {
    uint64_t edgeId;
    uint64_t edgeCommitId;

    int currentSpeedLimit;
    int correctSpeedLimit;

    maps::geolib3::Point2 geodeticPos;

    bool forward;
    bool truck;

    std::optional<db::TId> speedLimitObjectId;
};
using Errors = std::vector<Error>;

struct ManeuverDiff {
    uint64_t edgeInId;
    bool edgeInForward;
    uint64_t edgeOutId;
    bool edgeOutForward;

    double angle; // угол в радианах между входным и выходным ребром изменяется от -M_PI до M_PI

    std::optional<db::TId> objectId;
    std::optional<uint64_t> condId;
    bool onlyTruck;

    bool condValid;
    bool signValid;
};
using ManeuverDiffs = std::vector<ManeuverDiff>;

struct ImageDescription {
    std::string url;
    int width;
    int height;
};
struct ImageFeature {
    db::TId id;
    double heading;
    maps::geolib3::Point2 geodeticPos;

    ImageDescription previewImage;
    ImageDescription fullImage;

    maps::chrono::TimePoint time;
    cv::Rect bbox;
};
using ImageFeatures = std::vector<ImageFeature>;
struct ObjectContent {
    db::TId objectId = 0;
    maps::geolib3::Point2 geodeticPos;
    double heading;
    std::string signType;
    ImageFeatures imageFeatures;
};
using ObjectsMap = std::map<db::TId, ObjectContent>;

Edge loadEdge(const maps::json::Value& value) {
    Edge edge;
    edge.edgeId = value["id"].as<uint64_t>();
    edge.edgeCommitId = value["commit_id"].as<uint64_t>();
    edge.speedLimitF = value["speed_limit_f"].as<int>();
    edge.speedLimitT = value["speed_limit_t"].as<int>();
    edge.speedLimitTruckF = value["speed_limit_truck_f"].as<int>();
    edge.speedLimitTruckT = value["speed_limit_truck_t"].as<int>();

    maps::json::Value jsonPoints = value["points"];
    REQUIRE(jsonPoints.size() > 0, "Edge with empty array of points");
    for (size_t i = 0; i < jsonPoints.size(); i++) {
        edge.points.emplace_back(maps::geolib3::Point2(jsonPoints[i][0].as<double>(), jsonPoints[i][1].as<double>()));
    }

    if (value.hasField("speed_limit_object_id_f")) {
        edge.speedLimitObjectIdF = value["speed_limit_object_id_f"].as<int64_t>();
    }
    if (value.hasField("speed_limit_object_id_t")) {
        edge.speedLimitObjectIdT = value["speed_limit_object_id_t"].as<int64_t>();
    }
    if (value.hasField("speed_limit_truck_object_id_f")) {
        edge.speedLimitTruckObjectIdF = value["speed_limit_truck_object_id_f"].as<int64_t>();
    }
    if (value.hasField("speed_limit_truck_object_id_t")) {
        edge.speedLimitTruckObjectIdT = value["speed_limit_truck_object_id_t"].as<int64_t>();
    }
    if (value.hasField("oneway")) {
        edge.oneWay = value["oneway"].as<int>();
    } else {
        edge.oneWay = (int)maps::ymapsdf::rd::Direction::Both;
    }
    if (value.hasField("fc")) {
        edge.fc = value["fc"].as<int>();
    } else {
        edge.fc = 0;
    }
    return edge;
}

Edges loadEdges(const std::string& jsonPath) {
    Edges edges;
    maps::json::Value jsonFile = maps::json::Value::fromFile(jsonPath);
    maps::json::Value jsonEdges = jsonFile["rd_el"];
    for (size_t i = 0; i < jsonEdges.size(); i++) {
        edges.emplace_back(loadEdge(jsonEdges[i]));
    }
    return edges;
}

MapEdges loadMapEdges(const std::string& jsonPath) {
    MapEdges edges;
    maps::json::Value jsonFile = maps::json::Value::fromFile(jsonPath);
    maps::json::Value jsonEdges = jsonFile["rd_el"];
    for (size_t i = 0; i < jsonEdges.size(); i++) {
        const Edge edge = loadEdge(jsonEdges[i]);
        edges[edge.edgeId] = edge;
    }
    return edges;
}

ManeuverDiff loadManeuverDiff(const maps::json::Value& value) {
    ManeuverDiff diff;
    maps::json::Value edgeIn = value["edge_in"];
    diff.edgeInId = edgeIn["id"].as<uint64_t>();
    diff.edgeInForward = edgeIn["forward"].as<bool>();
    maps::json::Value edgeOut = value["edge_out"];
    diff.edgeOutId = edgeOut["id"].as<uint64_t>();
    diff.edgeOutForward = edgeOut["forward"].as<bool>();

    diff.angle = value["angle"].as<double>();
    diff.condValid = value["valid_cond"].as<bool>();
    diff.signValid = value["valid_sign"].as<bool>();
    diff.onlyTruck = value["only_truck"].as<bool>();

    if (value.hasField("sign_id")) {
        diff.objectId = value["sign_id"].as<int64_t>();
    }
    if (value.hasField("cond_id")) {
        diff.condId = value["cond_id"].as<int64_t>();
    }
    return diff;
}

ManeuverDiffs loadManeuverDiffs(const std::string& jsonPath) {
    ManeuverDiffs mdiffs;
    maps::json::Value jsonFile = maps::json::Value::fromFile(jsonPath);
    maps::json::Value jsonMDiffs = jsonFile["maneuvers"];
    std::vector<std::string> nodeIds = jsonMDiffs.fields();
    for (size_t i = 0; i < nodeIds.size(); i++) {
        maps::json::Value jsonNodeMDiffs = jsonMDiffs[nodeIds[i]];
        if (0 == jsonNodeMDiffs.size()) {
            continue;
        }
        for (size_t j = 0; j < jsonNodeMDiffs.size(); j++) {
            mdiffs.emplace_back(loadManeuverDiff(jsonNodeMDiffs[j]));
        }
    }
    return mdiffs;

}

db::TIds extractObjectIds(const Errors& errors)
{
    db::TIds objectIds;
    for (size_t i = 0; i < errors.size(); i++) {
        if (errors[i].speedLimitObjectId) {
            objectIds.emplace_back(*(errors[i].speedLimitObjectId));
        }
    }
    return objectIds;
}

db::TIds extractObjectIds(const ManeuverDiffs& mdiffs)
{
    db::TIds objectIds;
    for (size_t i = 0; i < mdiffs.size(); i++) {
        if (mdiffs[i].objectId) {
            objectIds.emplace_back(*(mdiffs[i].objectId));
        }
    }
    return objectIds;
}

double getHeading(const Eigen::Quaterniond& rotation) {
    return eye::decomposeRotation(rotation).heading.value();
}

std::map<db::TId, db::eye::ObjectLocation> loadLocations(
    pqxx::transaction_base& txn,
    const db::TIds& objectIds)
{
    db::eye::ObjectLocations locations = db::eye::ObjectLocationGateway(txn).load(
         db::eye::table::ObjectLocation::objectId.in(objectIds)
     );
    std::map<db::TId, db::eye::ObjectLocation> result;
    for (size_t i = 0; i < locations.size(); i++) {
        result.emplace(locations[i].objectId(), locations[i]);
    }
    return result;
}

std::map<db::TId, db::eye::Object> loadObjects(
    pqxx::transaction_base& txn,
    const db::TIds& objectIds)
{
    db::eye::Objects objects = db::eye::ObjectGateway(txn).load(
        db::eye::table::Object::id.in(objectIds)
    );
    std::map<db::TId, db::eye::Object> result;
    for (size_t i = 0; i < objects.size(); i++) {
        result.insert({objects[i].id(), objects[i]});
    }
    return result;
}

std::map<db::TId, ImageFeatures> loadImageFeatures(
    pqxx::transaction_base& txn,
    const eye::FrameUrlResolver& frameUrlResolver,
    const db::TIds& objectIds)
{
    db::eye::Objects objects = db::eye::ObjectGateway(txn).load(
        db::eye::table::Object::id.in(objectIds)
    );

    std::map<db::TId, db::TIds> detectionIdsByObjectId;
    std::map<db::TId, db::TId> objectIdByPrimaryDetectionId;
    db::TIds detectionIds;
    for (size_t i = 0; i < objects.size(); i++) {
        const db::eye::Object& object = objects[i];
        detectionIds.push_back(object.primaryDetectionId());
        detectionIdsByObjectId[object.id()].push_back(object.primaryDetectionId());
        objectIdByPrimaryDetectionId[object.primaryDetectionId()] = object.id();
    }
    db::eye::PrimaryDetectionRelations pdrelations = db::eye::PrimaryDetectionRelationGateway(txn).load(
        db::eye::table::PrimaryDetectionRelation::primaryDetectionId.in(detectionIds) &&
        ! db::eye::table::PrimaryDetectionRelation::deleted
    );
    for (size_t i = 0; i < pdrelations.size(); i++) {
        const db::eye::PrimaryDetectionRelation& pdrelation = pdrelations[i];
        detectionIds.push_back(pdrelation.detectionId());
        detectionIdsByObjectId[objectIdByPrimaryDetectionId[pdrelation.primaryDetectionId()]].push_back(pdrelation.detectionId());
    }

    db::eye::Detections detections = db::eye::DetectionGateway(txn).load(
        db::eye::table::Detection::id.in(detectionIds) &&
        ! db::eye::table::Detection::deleted
    );
    std::set<db::TId> groupIds;
    std::map<db::TId, db::TId> groupIdByDetectionId;
    std::map<db::TId, cv::Rect> bboxByDetectionId;
    for (size_t i = 0; i < detections.size(); i++) {
        const db::eye::Detection& detection = detections[i];
        groupIds.insert(detection.groupId());
        groupIdByDetectionId[detection.id()] = detection.groupId();
        bboxByDetectionId[detection.id()] = detection.box();
    }
    db::eye::DetectionGroups groups = db::eye::DetectionGroupGateway(txn).load(
        db::eye::table::DetectionGroup::id.in({ groupIds.begin(), groupIds.end()})
    );
    db::TIds frameIds;
    std::map<db::TId, db::TId> frameIdByGroupId;
    for (size_t i = 0; i < groups.size(); i++) {
        const db::eye::DetectionGroup& group = groups[i];
        frameIds.push_back(group.frameId());
        frameIdByGroupId[group.id()] = group.frameId();
    }
    db::eye::Frames frames = db::eye::FrameGateway(txn).load(
        db::eye::table::Frame::id.in(frameIds)
    );
    std::map<db::TId, db::eye::Frame> frameByFrameId;
    for (size_t i = 0; i < frames.size(); i++) {
        frameByFrameId.emplace(frames[i].id(), frames[i]);
    }

    std::map<db::TId, db::eye::Frame> frameByDetectionId;
    for (size_t i = 0; i < detections.size(); i++) {
        const db::TId detId = detections[i].id();
        const db::eye::Frame& frame = frameByFrameId.at(frameIdByGroupId[groupIdByDetectionId[detId]]);
        frameByDetectionId.emplace(detId, frame);
    }

    db::eye::FrameLocations frameLocations = db::eye::FrameLocationGateway(txn).load(
        db::eye::table::FrameLocation::frameId.in(frameIds)
    );
    std::map<db::TId, db::eye::FrameLocation> frameLocationByFrameId;
    for (size_t i = 0; i < frameLocations.size(); i++) {
        frameLocationByFrameId.emplace(frameLocations[i].frameId(), frameLocations[i]);
    }
    std::map<db::TId, db::eye::FrameLocation> frameLocationByDetectionId;
    for (size_t i = 0; i < detections.size(); i++) {
        const db::TId detId = detections[i].id();
        const db::eye::FrameLocation& frameLocation = frameLocationByFrameId.at(frameIdByGroupId[groupIdByDetectionId[detId]]);
        frameLocationByDetectionId.emplace(detId, frameLocation);
    }

    std::map<db::TId, ImageFeatures> result;
    for (size_t i = 0; i < objectIds.size(); i++) {
        const db::TId objId = objectIds[i];
        const db::TIds detectionIds = detectionIdsByObjectId[objId];
        ImageFeatures ifeats;
        for (size_t j = 0; j < detectionIds.size(); j++) {
            const db::TId detId = detectionIds[j];
            const cv::Rect& bbox = bboxByDetectionId[detId];
            const db::eye::Frame& frame = frameByDetectionId.at(detId);
            const db::eye::FrameLocation& frameLocation = frameLocationByDetectionId.at(detId);
            ImageFeature ifeat;
            ifeat.id = frame.id();
            ifeat.heading = getHeading(frameLocation.rotation());
            ifeat.geodeticPos = frameLocation.geodeticPos();

            const common::Size thumbnailSize = common::getThumbnailSize(frame.size());
            ifeat.previewImage.url = frameUrlResolver.preview(frame, db::FeaturePrivacy::Public);
            ifeat.previewImage.width = thumbnailSize.width;
            ifeat.previewImage.height = thumbnailSize.height;

            ifeat.fullImage.url = frameUrlResolver.image(frame, db::FeaturePrivacy::Public);
            ifeat.fullImage.width = frame.size().width;
            ifeat.fullImage.height = frame.size().height;

            ifeat.time = frame.time();
            ifeat.bbox = common::transformByImageOrientation(bbox, frame.originalSize(), frame.orientation());
            ifeats.emplace_back(ifeat);
        }
        result.emplace(objId, ifeats);
    }
    return result;
}

ObjectsMap loadObjectsMap(
    maps::pgpool3::Pool& pool,
    const eye::FrameUrlResolver& frameUrlResolver,
    const db::TIds& objectIds)
{
    maps::pgpool3::TransactionHandle txn = pool.slaveTransaction();

    std::map<db::TId, db::eye::ObjectLocation> locationByObjectId = loadLocations(*txn, objectIds);
    std::map<db::TId, db::eye::Object> objectById = loadObjects(*txn, objectIds);

    std::map<db::TId, ImageFeatures> imageFeaturesByObjectId = loadImageFeatures(*txn, frameUrlResolver, objectIds);

    ObjectsMap result;
    for (size_t i = 0; i < objectIds.size(); i++) {
        const db::TId id = objectIds[i];
        const db::eye::ObjectLocation& location = locationByObjectId.at(id);
        const db::eye::Object& object = objectById.at(id);
        ObjectContent content;
        content.objectId = id;
        content.geodeticPos = location.geodeticPos();
        content.heading = getHeading(location.rotation());
        content.signType = traffic_signs::toString(object.attrs<db::eye::SignAttrs>().type);
        content.imageFeatures = imageFeaturesByObjectId[id];
        result.emplace(id, content);
    }
    return result;
};

Errors differentSpeedEdgeIds(const Edges& gtEdges, const MapEdges& genEdges)
{
    Errors errors;
    for (size_t i = 0; i < gtEdges.size(); i++) {
        const Edge& edge = gtEdges[i];
        const uint64_t id = edge.edgeId;
        auto it = genEdges.find(id);
        if (it == genEdges.end()) {
            continue;
        }

        if (it->second.speedLimitF != edge.speedLimitF) {
            Error error;
            error.edgeId = edge.edgeId;
            error.edgeCommitId = edge.edgeCommitId;
            error.currentSpeedLimit = edge.speedLimitF;
            error.correctSpeedLimit = it->second.speedLimitF;
            error.forward = true;
            error.truck = false;
            error.speedLimitObjectId = it->second.speedLimitObjectIdF;
            error.geodeticPos = maps::geolib3::convertMercatorToGeodetic(edge.points.front());
            errors.emplace_back(error);
        } else if (it->second.speedLimitTruckF != edge.speedLimitTruckF) {
            Error error;
            error.edgeId = edge.edgeId;
            error.edgeCommitId = edge.edgeCommitId;
            error.currentSpeedLimit = edge.speedLimitTruckF;
            error.correctSpeedLimit = it->second.speedLimitTruckF;
            error.forward = true;
            error.truck = true;
            if (it->second.speedLimitTruckObjectIdF) {
                error.speedLimitObjectId = it->second.speedLimitTruckObjectIdF;
            } else {
                error.speedLimitObjectId = it->second.speedLimitObjectIdF;
            }
            error.geodeticPos = maps::geolib3::convertMercatorToGeodetic(edge.points.front());
            errors.emplace_back(error);
        }
        if (it->second.speedLimitT != edge.speedLimitT) {
            Error error;
            error.edgeId = edge.edgeId;
            error.edgeCommitId = edge.edgeCommitId;
            error.currentSpeedLimit = edge.speedLimitT;
            error.correctSpeedLimit = it->second.speedLimitT;
            error.forward = false;
            error.truck = false;
            error.speedLimitObjectId = it->second.speedLimitObjectIdT;
            error.geodeticPos = maps::geolib3::convertMercatorToGeodetic(edge.points.back());
            errors.emplace_back(error);
        } else if (it->second.speedLimitTruckT != edge.speedLimitTruckT) {
            Error error;
            error.edgeId = edge.edgeId;
            error.edgeCommitId = edge.edgeCommitId;
            error.currentSpeedLimit = edge.speedLimitTruckT;
            error.correctSpeedLimit = it->second.speedLimitTruckT;
            error.forward = false;
            error.truck = true;
            if (it->second.speedLimitTruckObjectIdT) {
                error.speedLimitObjectId = it->second.speedLimitTruckObjectIdT;
            } else {
                error.speedLimitObjectId = it->second.speedLimitObjectIdT;
            }
            error.geodeticPos = maps::geolib3::convertMercatorToGeodetic(edge.points.back());
            errors.emplace_back(error);
        }
    }
    return errors;
}

void dumpImageDescription(
    const ImageDescription& imageDesc,
    maps::json::ObjectBuilder& builder)
{
    builder["height"] = imageDesc.height;
    builder["width"] = imageDesc.width;
    builder["url"] = imageDesc.url;
}

void dumpImageFeature(
    const ImageFeature& imageFeature,
    maps::json::ObjectBuilder& builder)
{
    builder["id"] = std::to_string(imageFeature.id);
    builder["heading"] = imageFeature.heading;
    builder["geometry"] << [&](maps::json::ObjectBuilder builder) {
        builder["type"] = "Point";
        builder["coordinates"] << [&](maps::json::ArrayBuilder builder) {
            builder << imageFeature.geodeticPos.x() << imageFeature.geodeticPos.y();
        };
    };
    builder["imagePreview"] << [&](maps::json::ObjectBuilder builder) {
        dumpImageDescription(imageFeature.previewImage, builder);
    };
    builder["imageFull"] << [&](maps::json::ObjectBuilder builder) {
        dumpImageDescription(imageFeature.fullImage, builder);
    };
    builder["box"] << [&](maps::json::ArrayBuilder builder) {
        builder << imageFeature.bbox.x
                << imageFeature.bbox.y
                << imageFeature.bbox.x + imageFeature.bbox.width - 1
                << imageFeature.bbox.y + imageFeature.bbox.height - 1;
    };
    builder["timestamp"] << maps::chrono::formatIsoDateTime(imageFeature.time);
}

void dumpImageFeatures(
    const ImageFeatures& imageFeatures,
    maps::json::ArrayBuilder& builder)
{
    constexpr size_t MAX_FEATURES = 10;
    std::vector<size_t> idx(imageFeatures.size());
    for (size_t i = 0; i < imageFeatures.size(); i++) {
        idx[i] = i;
    }
    if (imageFeatures.size() >= MAX_FEATURES) {
        std::sort(idx.begin(), idx.end(),
            [&](size_t l, size_t r) {
                return imageFeatures[l].time > imageFeatures[r].time;
            }
        );
        idx.resize(MAX_FEATURES);
    }
    for (size_t i = 0; i < idx.size(); i++) {
        builder << [&](maps::json::ObjectBuilder builder) {
            dumpImageFeature(imageFeatures[idx[i]], builder);
        };
    }
}

void dumpObjectContent(
    const ObjectContent& object,
    const std::string& type,
    maps::json::ObjectBuilder& builder)
{
    builder["id"] = std::to_string(object.objectId);
    builder["type"] = type;
    builder["imageFeatures"]  << [&](maps::json::ArrayBuilder builder) {
        dumpImageFeatures(object.imageFeatures, builder);
    };
    builder["sign"] <<  [&](maps::json::ObjectBuilder builder) {
        builder["type"] = object.signType;
        builder["id"] = std::to_string(object.objectId);
        builder["heading"] = object.heading;
        builder["geometry"] <<  [&](maps::json::ObjectBuilder builder) {
            builder["type"] = "Point";
            builder["coordinates"] << [&](maps::json::ArrayBuilder builder) {
                builder << object.geodeticPos.x() << object.geodeticPos.y();
            };
        };
    };
}

void generateHypothesis(const Error& error, const ObjectsMap& objects, std::ostream& outStream) {
    maps::json::Builder builder(outStream);
    auto it = objects.find(error.speedLimitObjectId.value_or(0));
    builder << [&](maps::json::ObjectBuilder builder) {
        builder["source"] = "experiment-speed-limits";
        builder["position"] << [&](maps::json::ObjectBuilder builder) {
            builder["type"] = "Point";
            builder["coordinates"] << [&](maps::json::ArrayBuilder builder) {
                if (it == objects.end()) {
                    builder << error.geodeticPos.x() << error.geodeticPos.y();
                } else {
                    builder << it->second.geodeticPos.x() << it->second.geodeticPos.y();
                }
            };
        };
        builder["objectId"] = std::to_string(error.edgeId);
        builder["commitId"] = std::to_string(error.edgeCommitId);
        builder["correctSpeedLimit"] = error.correctSpeedLimit;
        builder["currentSpeedLimit"] = error.currentSpeedLimit;
        builder["direction"] = (error.forward ? "F" : "T");
        builder["workflow"] = "task";
        builder["hidden"] = true;
        builder["truck"] = error.truck;
        builder["confirmingSignNotFound"] = (it == objects.end());
        if (it != objects.end()) {
            builder["sourceContext"] << [&](maps::json::ObjectBuilder builder) {
                builder["type"] = "mrc";
                builder["content"] << [&](maps::json::ObjectBuilder builder) {
                    dumpObjectContent(it->second, "wrong_speed_limit", builder);
                };
            };
        };
    };
}

void savePts(
    const std::vector<maps::geolib3::Point2>& pts,
    bool forward,
    maps::json::ArrayBuilder& builder)
{
    if (forward) {
        for (size_t i = 0; i < pts.size(); i++) {
            const maps::geolib3::Point2 pt = maps::geolib3::convertMercatorToGeodetic(pts[i]);
            builder << [&](maps::json::ArrayBuilder builder) {
                builder << pt.x() << pt.y();
            };
        }
    } else {
        for (int i = pts.size() - 1; 0 <= i; i--) {
            const maps::geolib3::Point2 pt = maps::geolib3::convertMercatorToGeodetic(pts[i]);
            builder << [&](maps::json::ArrayBuilder builder) {
                builder << pt.x() << pt.y();
            };
        }
    }
}

void generateHypothesis(const ManeuverDiff& diff, const ObjectsMap& objects, const MapEdges& edges, std::ostream& outStream) {
    REQUIRE((diff.condValid && !diff.signValid || !diff.condValid && diff.signValid),
            "Condition and sign valid flag is same, unable to generate hypothesis");

    maps::json::Builder builder(outStream);
    REQUIRE(diff.objectId.has_value() || diff.signValid, "Unable to generate hypothesis about sign without sign");
    auto objectIt = objects.end();
    if (!diff.signValid) {
        objectIt = objects.find(*diff.objectId);
        REQUIRE(objectIt != objects.end(), "Unable to generate hypothesis. Sign with id " << *diff.objectId << " not found");
    }
    const std::vector<maps::geolib3::Point2> edgeInPts = edges.at(diff.edgeInId).points;
    const std::vector<maps::geolib3::Point2> edgeOutPts = edges.at(diff.edgeOutId).points;

    builder << [&](maps::json::ObjectBuilder builder) {
        builder["source"] = "experiment-prohibited-path";
        builder["position"] << [&](maps::json::ObjectBuilder builder) {
            builder["type"] = "Point";
            builder["coordinates"] << [&](maps::json::ArrayBuilder builder) {
                if (!diff.signValid) {
                     builder << objectIt->second.geodeticPos.x() << objectIt->second.geodeticPos.y();
                } else {
                    const maps::geolib3::Point2 pt =
                        diff.edgeInForward
                            ? maps::geolib3::convertMercatorToGeodetic(edgeInPts.back())
                            : maps::geolib3::convertMercatorToGeodetic(edgeInPts.front());
                     builder << pt.x() << pt.y();
                }
            };
        };
        builder["workflow"] = "task";
        builder["hidden"] = true;
        builder["accessId"] = (int)(diff.onlyTruck ? maps::ymapsdf::rd::AccessId::Truck : maps::ymapsdf::rd::AccessId::Vehicles);
        builder["condType"] = diff.condValid ? "prohibited" : "allowed";
        builder["path"] << [&](maps::json::ObjectBuilder builder) {
            builder["type"] = "LineString";
            builder["coordinates"] << [&](maps::json::ArrayBuilder builder) {
                savePts(edgeInPts, diff.edgeInForward, builder);
                savePts(edgeOutPts, diff.edgeOutForward, builder);
            };
        };

        if (!diff.signValid) {
            builder["sourceContext"] << [&](maps::json::ObjectBuilder builder) {
                builder["type"] = "mrc";
                builder["content"] << [&](maps::json::ObjectBuilder builder) {
                    dumpObjectContent(objectIt->second, "prohibited_path", builder);
                };
            };
        }
    };
}

void reworkSpeedLimits(
    const std::string& gtJsonPath,
    const std::string& tstJsonPath,
    maps::pgpool3::Pool& pool,
    const eye::FrameUrlResolver& frameUrlResolver,
    const std::string& outFolderPath)
{
    Edges gtEdges = loadEdges(gtJsonPath);
    MapEdges genEdges = loadMapEdges(tstJsonPath);

    Errors errors = differentSpeedEdgeIds(gtEdges, genEdges);
    INFO() << "Errors: " << errors.size();

    ObjectsMap objectsMap = loadObjectsMap(pool, frameUrlResolver, extractObjectIds(errors));
    for (size_t i = 0; i < errors.size(); i++) {
        const Error& error = errors[i];
        std::ofstream ofs(outFolderPath + '/' + std::to_string(i) + ".json");
        generateHypothesis(error, objectsMap, ofs);
    }
}

void reworkManeuvers(
    const std::string& gtJsonPath,
    const std::string& maneuversDiffJsonPath,
    maps::pgpool3::Pool& pool,
    const eye::FrameUrlResolver& frameUrlResolver,
    const std::string& outFolderPath)
{
    MapEdges edgesMap = loadMapEdges(gtJsonPath);
    ManeuverDiffs mdiffs = loadManeuverDiffs(maneuversDiffJsonPath);
    ObjectsMap objectsMap = loadObjectsMap(pool, frameUrlResolver, extractObjectIds(mdiffs));
    size_t hypCount = 0;
    for (size_t i = 0; i < mdiffs.size(); i++) {
        const ManeuverDiff& mdiff = mdiffs[i];
        if (mdiff.condValid && mdiff.signValid || !mdiff.condValid && !mdiff.signValid) {
            INFO() << "Condition and sign valid flag is same, unable to generate hypothesis";
            continue;
        }
        hypCount++;
        std::ofstream ofs(outFolderPath + '/' + std::to_string(i) + ".json");
        generateHypothesis(mdiff, objectsMap, edgesMap, ofs);
    }
    INFO() << "Generated hypotheses: " << hypCount;
}

} //namespace

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser("Test");

    maps::cmdline::Option<std::string> gtJsonPath = parser.string("gt-json")
        .required()
        .help("Path to json file with data from maps");
    maps::cmdline::Option<std::string> hypType = parser.string("hyp-type")
        .help("Type of generated hypotheses (speed_limits, maneuvers");
    maps::cmdline::Option<std::string> tstJsonPath = parser.string("tst-json")
        .help("Path to json file with speed limits generated data");
    maps::cmdline::Option<std::string> maneuversDiffJsonPath = parser.string("maneuvers-json")
        .help("Path to json file with difference maneuvers");
    maps::cmdline::Option<std::string> outFolderPath = parser.string("out-folder")
        .required()
        .help("Path to output folder for generated hypopheses");
    maps::cmdline::Option<std::string> mrcConfigPath = parser.string("mrc-config")
        .required()
        .help("Path to mrc config");
    maps::cmdline::Option<std::string> secretVersion = parser.string("secret-version")
        .help("version for secrets from yav.yandex-team.ru");


    parser.parse(argc, const_cast<char**>(argv));
    common::Config mrcConfig =
        secretVersion.defined()
        ? common::Config(maps::vault_boy::loadContextWithYaVault(secretVersion), mrcConfigPath)
        : common::templateConfigFromCmdPath(mrcConfigPath);
    maps::wiki::common::PoolHolder mrc(mrcConfig.makePoolHolder());

    eye::FrameUrlResolver frameUrlResolver(
        mrcConfig.externals().mrcBrowserUrl(),
        mrcConfig.externals().mrcBrowserProUrl());

    if ("speed_limits" == hypType) {
        reworkSpeedLimits(gtJsonPath, tstJsonPath, mrc.pool(), frameUrlResolver, outFolderPath);
    } else if ("maneuvers" == hypType) {
        reworkManeuvers(gtJsonPath, maneuversDiffJsonPath, mrc.pool(), frameUrlResolver, outFolderPath);
    } else {
        REQUIRE(false, "Undefined type of hypotheses " << hypType);
    }
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
