#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/url.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>

namespace maps::mrc::eye::tests {

TEST(url, public_mrc)
{
    db::eye::Frame frame {
        1,
        common::ImageOrientation::fromExif(3),
        db::eye::MrcUrlContext{1, "1", "1"},
        {1280, 720},
        chrono::TimePoint()
    };

    FrameUrlResolver makeUrl(
        "https://mrc-browser.maps.yandex.ru",
        "https://mrc-browser-pro.maps.yandex.ru");

    EXPECT_EQ(
        makeUrl.image(frame, db::FeaturePrivacy::Public),
        "https://mrc-browser.maps.yandex.ru/feature/1/image?orientation=3"
    );
}

TEST(url, private_mrc)
{
    db::eye::Frame frame {
        1,
        common::ImageOrientation::fromExif(3),
        db::eye::MrcUrlContext{1, "1", "1"},
        {1280, 720},
        chrono::TimePoint()
    };

    FrameUrlResolver makeUrl(
        "https://mrc-browser.maps.yandex.ru",
        "https://mrc-browser-pro.maps.yandex.ru");

    EXPECT_EQ(
        makeUrl.image(frame, db::FeaturePrivacy::Secret),
        "https://mrc-browser-pro.maps.yandex.ru/feature/1/image?orientation=3"
    );
}


TEST(url, public_mrc_with_orientation)
{
    db::eye::Frame frame {
        1,
        common::ImageOrientation::fromExif(3),
        db::eye::MrcUrlContext{1, "1", "1"},
        {1280, 720},
        chrono::TimePoint()
    };

    FrameUrlResolver makeUrl(
        "https://mrc-browser.maps.yandex.ru",
        "https://mrc-browser-pro.maps.yandex.ru");

    EXPECT_EQ(
        makeUrl.image(frame, common::ImageOrientation::fromExif(5), db::FeaturePrivacy::Public),
        "https://mrc-browser.maps.yandex.ru/feature/1/image?orientation=5"
    );
}

TEST(url, public_thumbnail_mrc)
{
    db::eye::Frame frame {
        1,
        common::ImageOrientation::fromExif(3),
        db::eye::MrcUrlContext{1, "1", "1"},
        {1280, 720},
        chrono::TimePoint()
    };

    FrameUrlResolver makeUrl(
        "https://mrc-browser.maps.yandex.ru",
        "https://mrc-browser-pro.maps.yandex.ru");

    EXPECT_EQ(
        makeUrl.preview(frame, db::FeaturePrivacy::Public),
        "https://mrc-browser.maps.yandex.ru/feature/1/thumbnail"
    );
}

TEST(url, panorama)
{
    constexpr geolib3::Heading heading{180};
    constexpr geolib3::Degrees tilt{0};
    constexpr geolib3::Degrees hfov{70};
    constexpr common::Size size{1920u, 1080u};
    const std::string oid{"1-2-3"};

    db::eye::Frame frame{
        1,
        common::ImageOrientation::fromExif(1),
        db::eye::PanoramaUrlContext{
            .oid = oid,
            .heading = heading,
            .tilt = tilt,
            .horizontalFOV = hfov,
            .size = size},
        size,
        chrono::TimePoint()};

    FrameUrlResolver makeUrl(
        "https://mrc-browser.maps.yandex.ru",
        "https://mrc-browser-pro.maps.yandex.ru");

    const http::URL actual(makeUrl.image(frame, db::FeaturePrivacy::Public));
    EXPECT_EQ(actual.host(), "mrc-browser.maps.yandex.ru");
    EXPECT_EQ(actual.path(), "/v2/pano/" + oid + "/image");
    EXPECT_EQ(actual.param("heading"), std::to_string(heading.value()));
    EXPECT_EQ(actual.param("tilt"), std::to_string(tilt.value()));
    EXPECT_EQ(actual.param("hfov"), std::to_string(hfov.value()));
    EXPECT_EQ(actual.param("size"), "1920,1080");
}

TEST(url, thumbnail_panorama)
{
    constexpr geolib3::Heading heading{180};
    constexpr geolib3::Degrees tilt{0};
    constexpr geolib3::Degrees hfov{70};
    constexpr common::Size size{1920u, 1080u};
    const std::string oid{"1-2-3"};

    db::eye::Frame frame{
        1,
        common::ImageOrientation::fromExif(1),
        db::eye::PanoramaUrlContext{
            .oid = oid,
            .heading = heading,
            .tilt = tilt,
            .horizontalFOV = hfov,
            .size = size},
        size,
        chrono::TimePoint()};

    FrameUrlResolver makeUrl(
        "https://mrc-browser.maps.yandex.ru",
        "https://mrc-browser-pro.maps.yandex.ru");

    const http::URL actual(makeUrl.preview(frame, db::FeaturePrivacy::Public));
    EXPECT_EQ(actual.host(), "mrc-browser.maps.yandex.ru");
    EXPECT_EQ(actual.path(), "/v2/pano/" + oid + "/image");
    EXPECT_EQ(actual.param("heading"), std::to_string(heading.value()));
    EXPECT_EQ(actual.param("tilt"), std::to_string(tilt.value()));
    EXPECT_EQ(actual.param("hfov"), std::to_string(hfov.value()));

    const auto thumbnailSize =
        common::getThumbnailSize(common::Size{1920, 1080});
    const auto expectedSize = std::to_string(thumbnailSize.width)
                              + "," +
                              std::to_string(thumbnailSize.height) ;
    EXPECT_EQ(actual.param("size"), expectedSize);
}

} // namespace maps::mrc::eye::tests
