#include <maps/wikimap/mapspro/services/mrc/eye/lib/detect_panel/include/sequence.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/libs/sql_chemistry/include/order.h>

#include <algorithm>
#include <chrono>
#include <iterator>
#include <unordered_map>
#include <unordered_set>

namespace maps::mrc::eye {

namespace {

std::unordered_map<db::TId, std::vector<chrono::TimePoint>> groupTimeByDeviceId(
        pqxx::transaction_base& txn,
        const db::TIdSet& frameIds)
{
    std::unordered_map<db::TId, std::vector<chrono::TimePoint>> timeByDeviceId;

    for (const auto& frame: db::eye::FrameGateway(txn).loadByIds({frameIds.begin(), frameIds.end()})) {
        timeByDeviceId[frame.deviceId()].push_back(frame.time());
    }

    return timeByDeviceId;
}

struct TimeInterval {
    db::TId deviceId;
    chrono::TimePoint begin;
    chrono::TimePoint end;

    TimeInterval(
            db::TId deviceId,
            chrono::TimePoint begin,
            chrono::TimePoint end)
        : deviceId(deviceId)
        , begin(begin)
        , end(end)
    {}
};

std::vector<TimeInterval> defineTimeItervals(
        pqxx::transaction_base& txn,
        const db::TIdSet& frameIds,
        std::chrono::seconds margin)
{
    std::vector<TimeInterval> intervals;

    for (auto&& [deviceId, timePoints]: groupTimeByDeviceId(txn, frameIds)) {
        ASSERT(!timePoints.empty());

        std::sort(timePoints.begin(), timePoints.end());

        for (auto it = timePoints.begin(); it != timePoints.end(); ) {
            const auto last = std::adjacent_find(
                it, timePoints.end(),
                [&](const auto& lhs, const auto& rhs) {
                    return rhs - lhs > margin;
                }
            );

            const chrono::TimePoint start = it->time_since_epoch() >= margin
                ? *it - margin
                : chrono::TimePoint();

            if (last == timePoints.end()) {
                intervals.emplace_back(deviceId, start, timePoints.back());
                break;
            }

            intervals.emplace_back(deviceId, start, *last);
            it = last + 1;
        }
    }

    return intervals;
}

} // namespace

std::vector<db::eye::Frames> makeFrameSequences(
        pqxx::transaction_base& txn,
        const db::TIdSet& frameIds,
        size_t margin,
        std::chrono::seconds timeMargin)
{
    namespace table = db::eye::table;

    std::vector<db::eye::Frames> sequences;

    auto push = [&](db::eye::Frames& sequence) {
        if (not sequence.empty()) {
            sequences.push_back(std::move(sequence));
        }
    };

    for (const auto& interval: defineTimeItervals(txn, frameIds, timeMargin)) {
        db::eye::Frames sequence;

        const auto frames = db::eye::FrameGateway(txn).load(
            not table::Frame::deleted
            && table::Frame::deviceId == interval.deviceId
                && table::Frame::time >= interval.begin
                && table::Frame::time <= interval.end,
            sql_chemistry::orderBy(table::Frame::time)
        );

        auto begin = frames.begin();
        auto end = frames.begin();

        for (; begin != frames.end(); begin = end) {
            const auto it = std::find_if(
                begin, frames.end(),
                [&frameIds](const auto& frame) {
                    return frameIds.count(frame.id());
                }
            );

            if (it == frames.end()) {
                break;
            }

            if (it - end > int(margin)) {
                push(sequence);
                end = it + 1;
                sequence = {std::max(it - margin, frames.begin()), end};
                continue;
            }

            end = it + 1;
            std::copy(begin, end, std::back_inserter(sequence));
        }

        push(sequence);
    }

    return sequences;
}

} // namespace maps::mrc::eye
