#include <maps/wikimap/mapspro/services/mrc/eye/lib/detect_road_marking/tests/fixture.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/detect_road_marking/impl/batch.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>

#include <library/cpp/testing/unittest/registar.h>
#include <mapreduce/yt/tests/yt_unittest_lib/yt_unittest_lib.h>

namespace maps::mrc::eye::tests {

struct BatchFixture : public Fixture {
    BatchFixture() {
        auto txn = newTxn();

        devices = {
            {db::eye::MrcDeviceAttrs{"M1"}},
        };
        db::eye::DeviceGateway(*txn).insertx(devices);

        txn->commit();
    }
};

Y_UNIT_TEST_SUITE_F(batch_tests, BatchFixture)
{

Y_UNIT_TEST(empty)
{
    auto txn = newTxn();

    const db::TId beginFrameTxnId = 0u;
    const db::TId beginPanelTxnId = 0u;
    const size_t limit = 5u;
    DetectRoadMarkingBatch batch
        = loadDetectRoadMarkingBatch(
            *txn,
            beginFrameTxnId, beginPanelTxnId,
            limit
        );

    UNIT_ASSERT_EQUAL(batch.beginFrameTxnId, beginFrameTxnId);
    UNIT_ASSERT_EQUAL(batch.endFrameTxnId, beginFrameTxnId);
    UNIT_ASSERT_EQUAL(batch.beginPanelTxnId, beginPanelTxnId);
    UNIT_ASSERT_EQUAL(batch.endPanelTxnId, beginPanelTxnId);
    UNIT_ASSERT_EQUAL(batch.data.frameById.empty(), true);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.empty(), true);
}

Y_UNIT_TEST(frame_batch_from_one_txn)
{
    db::eye::Frames newFrames = {
        {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
    };

    db::TId frameTxnId;
    {
        auto txn = newTxn();
        frameTxnId = db::eye::FrameGateway(*txn).insertx(newFrames);
        txn->commit();
    }

    auto txn = newTxn();

    const db::TId beginFrameTxnId = 0u;
    const db::TId beginPanelTxnId = 0u;
    const size_t limit = 5u;
    DetectRoadMarkingBatch batch
        = loadDetectRoadMarkingBatch(
            *txn,
            beginFrameTxnId, beginPanelTxnId,
            limit
        );

    UNIT_ASSERT_EQUAL(batch.beginFrameTxnId, beginFrameTxnId);
    UNIT_ASSERT_EQUAL(batch.endFrameTxnId, frameTxnId + 1);
    UNIT_ASSERT_EQUAL(batch.beginPanelTxnId, beginPanelTxnId);
    UNIT_ASSERT_EQUAL(batch.endPanelTxnId, beginPanelTxnId);
    UNIT_ASSERT_EQUAL(batch.data.frameById.size(), 3u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.empty(), true);
}

Y_UNIT_TEST(frame_batch_from_few_txn)
{
    db::eye::Frames newFrames = {
        {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
    };

    db::TId frameTxnId;
    {
        auto txn = newTxn();
        db::eye::FrameGateway(*txn).insertx(newFrames[0]);
        txn->commit();
    }
    {
        auto txn = newTxn();
        db::eye::FrameGateway(*txn).insertx(newFrames[1]);
        frameTxnId = db::eye::FrameGateway(*txn).insertx(newFrames[2]);
        txn->commit();
    }

    auto txn = newTxn();

    const db::TId beginFrameTxnId = 0u;
    const db::TId beginPanelTxnId = 0u;
    const size_t limit = 2u;
    DetectRoadMarkingBatch batch
        = loadDetectRoadMarkingBatch(
            *txn,
            beginFrameTxnId, beginPanelTxnId,
            limit
        );

    UNIT_ASSERT_EQUAL(batch.beginFrameTxnId, beginFrameTxnId);
    UNIT_ASSERT_EQUAL(batch.endFrameTxnId, frameTxnId + 1);
    UNIT_ASSERT_EQUAL(batch.beginPanelTxnId, beginPanelTxnId);
    UNIT_ASSERT_EQUAL(batch.endPanelTxnId, beginPanelTxnId);
    UNIT_ASSERT_EQUAL(batch.data.frameById.size(), 3u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.empty(), true);
}

Y_UNIT_TEST(recognition_batch_from_one_txn)
{
    db::eye::Frames newFrames = {
        {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
    };
    db::TId frameTxnId;
    {
        auto txn = newTxn();
        frameTxnId = db::eye::FrameGateway(*txn).insertx(newFrames);
        txn->commit();
    }

    db::eye::Recognitions newRecognitions = {
        {
            newFrames[0].id(),
            newFrames[0].orientation(),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
        {
            newFrames[1].id(),
            common::ImageOrientation(common::Rotation::CW_180),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
        {
            newFrames[1].id(),
            newFrames[1].orientation(),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
    };

    db::TId panelTxnId;
    {
        auto txn = newTxn();
        panelTxnId = db::eye::RecognitionGateway(*txn).insertx(newRecognitions);
        txn->commit();
    }

    auto txn = newTxn();

    const db::TId beginFrameTxnId = frameTxnId + 1;
    const db::TId beginPanelTxnId = frameTxnId + 1;
    const size_t limit = 2u;
    DetectRoadMarkingBatch batch
        = loadDetectRoadMarkingBatch(
            *txn,
            beginFrameTxnId, beginPanelTxnId,
            limit
        );

    UNIT_ASSERT_EQUAL(batch.beginFrameTxnId, beginFrameTxnId);
    UNIT_ASSERT_EQUAL(batch.endFrameTxnId, beginFrameTxnId);
    UNIT_ASSERT_EQUAL(batch.beginPanelTxnId, beginPanelTxnId);
    UNIT_ASSERT_EQUAL(batch.endPanelTxnId, panelTxnId + 1);
    UNIT_ASSERT_EQUAL(batch.data.frameById.size(), 2u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.size(), 2u);

    UNIT_ASSERT_EQUAL(batch.data.frameById.count(newFrames[0].id()), 1u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.count(newFrames[0].id()), 1u);
    UNIT_ASSERT_EQUAL(
        batch.data.panelByFrameId.at(newFrames[0].id()).id(),
        newRecognitions[0].id()
    );

    UNIT_ASSERT_EQUAL(batch.data.frameById.count(newFrames[1].id()), 1u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.count(newFrames[1].id()), 1u);
    UNIT_ASSERT_EQUAL(
        batch.data.panelByFrameId.at(newFrames[1].id()).id(),
        newRecognitions[2].id()
    );
}

Y_UNIT_TEST(recognition_batch_from_few_txn)
{
    db::eye::Frames newFrames = {
        {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
    };
    db::TId frameTxnId;
    {
        auto txn = newTxn();
        frameTxnId = db::eye::FrameGateway(*txn).insertx(newFrames);
        txn->commit();
    }

    db::eye::Recognitions newRecognitions = {
        {
            newFrames[0].id(),
            newFrames[0].orientation(),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
        {
            newFrames[1].id(),
            common::ImageOrientation(common::Rotation::CW_180),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
        {
            newFrames[1].id(),
            newFrames[1].orientation(),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
    };

    db::TId panelTxnId;
    {
        auto txn = newTxn();
        db::eye::RecognitionGateway(*txn).insertx(newRecognitions[0]);
        txn->commit();
    }
    {
        auto txn = newTxn();
        db::eye::RecognitionGateway(*txn).insertx(newRecognitions[1]);
        panelTxnId = db::eye::RecognitionGateway(*txn).insertx(newRecognitions[2]);
        txn->commit();
    }

    auto txn = newTxn();

    const db::TId beginFrameTxnId = frameTxnId + 1;
    const db::TId beginPanelTxnId = frameTxnId + 1;
    const size_t limit = 2u;
    DetectRoadMarkingBatch batch
        = loadDetectRoadMarkingBatch(
            *txn,
            beginFrameTxnId, beginPanelTxnId,
            limit
        );

    UNIT_ASSERT_EQUAL(batch.beginFrameTxnId, beginFrameTxnId);
    UNIT_ASSERT_EQUAL(batch.endFrameTxnId, beginFrameTxnId);
    UNIT_ASSERT_EQUAL(batch.beginPanelTxnId, beginPanelTxnId);
    UNIT_ASSERT_EQUAL(batch.endPanelTxnId, panelTxnId + 1);
    UNIT_ASSERT_EQUAL(batch.data.frameById.size(), 2u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.size(), 2u);

    UNIT_ASSERT_EQUAL(batch.data.frameById.count(newFrames[0].id()), 1u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.count(newFrames[0].id()), 1u);
    UNIT_ASSERT_EQUAL(
        batch.data.panelByFrameId.at(newFrames[0].id()).id(),
        newRecognitions[0].id()
    );

    UNIT_ASSERT_EQUAL(batch.data.frameById.count(newFrames[1].id()), 1u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.count(newFrames[1].id()), 1u);
    UNIT_ASSERT_EQUAL(
        batch.data.panelByFrameId.at(newFrames[1].id()).id(),
        newRecognitions[2].id()
    );
}

Y_UNIT_TEST(two_queue_batch_from_one_txn)
{
    db::eye::Frames newFrames = {
        {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
    };

    db::TId frameTxnId;
    {
        auto txn = newTxn();
        frameTxnId = db::eye::FrameGateway(*txn).insertx(newFrames);
        txn->commit();
    }


    db::eye::Recognitions newRecognitions = {
        {
            newFrames[0].id(),
            newFrames[0].orientation(),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
        {
            newFrames[1].id(),
            common::ImageOrientation(common::Rotation::CW_180),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
        {
            newFrames[1].id(),
            newFrames[1].orientation(),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
    };

    db::TId panelTxnId;
    {
        auto txn = newTxn();
        panelTxnId = db::eye::RecognitionGateway(*txn).insertx(newRecognitions);
        txn->commit();
    }

    auto txn = newTxn();

    const db::TId beginFrameTxnId = 0;
    const db::TId beginPanelTxnId = 0;
    const size_t limit = 2u;
    DetectRoadMarkingBatch batch
        = loadDetectRoadMarkingBatch(
            *txn,
            beginFrameTxnId, beginPanelTxnId,
            limit
        );

    UNIT_ASSERT_EQUAL(batch.beginFrameTxnId, beginFrameTxnId);
    UNIT_ASSERT_EQUAL(batch.endFrameTxnId, frameTxnId + 1);
    UNIT_ASSERT_EQUAL(batch.beginPanelTxnId, beginPanelTxnId);
    UNIT_ASSERT_EQUAL(batch.endPanelTxnId, panelTxnId + 1);
    UNIT_ASSERT_EQUAL(batch.data.frameById.size(), 3u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.size(), 2u);

    UNIT_ASSERT_EQUAL(batch.data.frameById.count(newFrames[0].id()), 1u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.count(newFrames[0].id()), 1u);
    UNIT_ASSERT_EQUAL(
        batch.data.panelByFrameId.at(newFrames[0].id()).id(),
        newRecognitions[0].id()
    );

    UNIT_ASSERT_EQUAL(batch.data.frameById.count(newFrames[1].id()), 1u);
    UNIT_ASSERT_EQUAL(batch.data.panelByFrameId.count(newFrames[1].id()), 1u);
    UNIT_ASSERT_EQUAL(
        batch.data.panelByFrameId.at(newFrames[1].id()).id(),
        newRecognitions[2].id()
    );

    UNIT_ASSERT_EQUAL(batch.data.frameById.count(newFrames[2].id()), 1u);
}

Y_UNIT_TEST(input_data)
{
    db::eye::Frames newFrames = {
        {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
        {devices[0].id(), identical, makeUrlContext(4, "4"), {1200, 800}, time()},
    };

    db::TId frameTxnId;
    {
        auto txn = newTxn();
        frameTxnId = db::eye::FrameGateway(*txn).insertx(newFrames);
        txn->commit();
    }


    db::eye::Recognitions newRecognitions = {
        {
            newFrames[0].id(),
            newFrames[0].orientation(),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
        {
            newFrames[1].id(),
            common::ImageOrientation(common::Rotation::CW_180),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
        {
            newFrames[1].id(),
            newFrames[1].orientation(),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
        {
            newFrames[2].id(),
            common::ImageOrientation(common::Rotation::CW_180),
            db::eye::RecognitionType::DetectPanel,
            db::eye::RecognitionSource::Model,
            0, // version
            db::eye::DetectedPanel{}
        },
    };

    db::TId panelTxnId;
    {
        auto txn = newTxn();
        panelTxnId = db::eye::RecognitionGateway(*txn).insertx(newRecognitions);
        txn->commit();
    }

    auto txn = newTxn();

    const db::TIds frameIds{
        newFrames[0].id(),
        newFrames[1].id(),
        newFrames[2].id(),
        newFrames[3].id(),
    };
    DetectRoadMarkingInputData data = loadDetectRoadMarkingInputData(*txn, frameIds);

    UNIT_ASSERT_EQUAL(data.frameById.size(), 4u);
    UNIT_ASSERT_EQUAL(data.panelByFrameId.size(), 2u);

    UNIT_ASSERT_EQUAL(data.panelByFrameId.count(newFrames[0].id()), 1u);
    UNIT_ASSERT_EQUAL(
        data.panelByFrameId.at(newFrames[0].id()).id(),
        newRecognitions[0].id()
    );

    UNIT_ASSERT_EQUAL(data.panelByFrameId.count(newFrames[1].id()), 1u);
    UNIT_ASSERT_EQUAL(
        data.panelByFrameId.at(newFrames[1].id()).id(),
        newRecognitions[2].id()
    );
}

} // Y_UNIT_TEST_SUITE

} // namespace maps::mrc::eye::tests
