#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/match_candidates.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/match.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/location.h>

#include <maps/libs/geolib/include/distance.h>

namespace maps::mrc::eye {

namespace {

using AnyDetectionAttrs = std::variant<
    db::eye::DetectedHouseNumber,
    db::eye::DetectedSign,
    db::eye::DetectedTrafficLight,
    db::eye::DetectedRoadMarking
>;

AnyDetectionAttrs getDetectionAttrs(
    const db::eye::Detection& detection,
    db::eye::DetectionType type)
{
    switch (type) {
        case db::eye::DetectionType::HouseNumber:
            return detection.attrs<db::eye::DetectedHouseNumber>();
        case db::eye::DetectionType::Sign:
            return detection.attrs<db::eye::DetectedSign>();
        case db::eye::DetectionType::TrafficLight:
            return detection.attrs<db::eye::DetectedTrafficLight>();
        case db::eye::DetectionType::RoadMarking:
            return detection.attrs<db::eye::DetectedRoadMarking>();
        default:
            throw RuntimeError() << "Unknown detection type - " << type;
    };
}

bool match(
    const db::eye::DetectedHouseNumber& lhs,
    const db::eye::DetectedHouseNumber& rhs)
{
    return lhs.number == rhs.number;
}

bool match(
    const db::eye::DetectedSign& lhs,
    const db::eye::DetectedSign& rhs)
{
    return lhs.type == rhs.type
        && lhs.temporary == rhs.temporary;
}

constexpr bool match(
    const db::eye::DetectedTrafficLight&,
    const db::eye::DetectedTrafficLight&)
{
    return true;
}

bool match(
    const db::eye::DetectedRoadMarking& lhs,
    const db::eye::DetectedRoadMarking& rhs)
{
    return lhs.type == rhs.type;
}

bool matchAttrs(
    const AnyDetectionAttrs& attrs1,
    const db::eye::DetectionType& type,
    const db::eye::Detection& detection2)
{
    switch (type) {
        case db::eye::DetectionType::HouseNumber:
            return match(
                std::get<db::eye::DetectedHouseNumber>(attrs1),
                detection2.attrs<db::eye::DetectedHouseNumber>()
            );
        case db::eye::DetectionType::Sign:
            return match(
                std::get<db::eye::DetectedSign>(attrs1),
                detection2.attrs<db::eye::DetectedSign>()
            );
        case db::eye::DetectionType::TrafficLight:
            return match(
                std::get<db::eye::DetectedTrafficLight>(attrs1),
                detection2.attrs<db::eye::DetectedTrafficLight>()
            );
        case db::eye::DetectionType::RoadMarking:
            return match(
                std::get<db::eye::DetectedRoadMarking>(attrs1),
                detection2.attrs<db::eye::DetectedRoadMarking>()
            );
        default:
            throw RuntimeError() << "Unknown detection type - " << type;
    };
}

class DetectionSearcher {
public:
    DetectionSearcher(
        const DetectionStore& store,
        const db::TIdSet& detectionIds)
    {
        for (db::TId id : detectionIds) {
            searcher_.insert(&store.locationByDetectionId(id).mercatorPos(), id);
        }
        searcher_.build();
    }

    db::TIds find(const geolib3::Point2& pos, double radiusMerc) {
        geolib3::BoundingBox searchBox(pos, 2 * radiusMerc, 2 * radiusMerc);

        const auto searchResult = searcher_.find(searchBox);

        db::TIds detectionIds;
        for (auto it = searchResult.first; it != searchResult.second; it++) {
            detectionIds.push_back(it->value());
        }

        return detectionIds;
    }

private:
    geolib3::StaticGeometrySearcher<geolib3::Point2, db::TId> searcher_;
};

} // namespace

DetectionIdPairSet generateMatchCandidates(
    const DetectionStore& store,
    const db::TIdSet& detectionIds1,
    const db::TIdSet& detectionIds2,
    const MatchCandidatesParams& params)
{
    DetectionSearcher searcher(store, detectionIds2);

    DetectionIdPairSet detectionPairs;

    for (db::TId detectionId1 : detectionIds1) {
        const db::eye::Detection& detection1 = store.detectionById(detectionId1);
        const db::eye::DetectionGroup& group1 = store.groupByDetectionId(detectionId1);

        const db::eye::FrameLocation& location1 = store.locationByDetectionId(detectionId1);
        const geolib3::Point2 position1 = location1.mercatorPos();
        const auto [heading1, _, __] = decomposeRotation(location1.rotation());
        const double distance = geolib3::toMercatorUnits(params.distanceMeters, position1);
        const auto attrs1 = getDetectionAttrs(detection1, group1.type());

        for (db::TId detectionId2 : searcher.find(position1, distance)) {
            if (detectionId1 == detectionId2) {
                continue;
            }

            if(detectionPairs.count({detectionId2, detectionId1}) > 0) {
                continue;
            }

            const db::eye::DetectionGroup& group2 = store.groupByDetectionId(detectionId2);

            if (group1.type() != group2.type()) {
                continue;
            }

            const db::eye::Detection& detection2 = store.detectionById(detectionId2);

            if (group1.frameId() == group2.frameId()) {
                // Matches in the same frame are forbidden
                continue;
            }

            if (!matchAttrs(attrs1, group1.type(), detection2)) {
                continue;
            }

            const db::eye::FrameLocation& location2 = store.locationByDetectionId(detectionId2);
            const geolib3::Point2 position2 = location2.mercatorPos();
            const auto [heading2, _, __] = decomposeRotation(location2.rotation());

            if (geolib3::distance(position1, position2) < distance &&
                geolib3::angleBetween(heading1, heading2) < params.angleEpsilon)
            {
                detectionPairs.emplace(detectionId1, detectionId2);
            }
        }
    }

    return detectionPairs;
}

} // namespace maps::mrc::eye
