#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition.h>

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <library/cpp/iterator/filtering.h>
#include <library/cpp/iterator/iterate_keys.h>
#include <library/cpp/iterator/iterate_values.h>

#include <unordered_map>

namespace maps::mrc::eye {

struct SplitPassageParams;

class DetectionStore {

public:
    DetectionStore() = default;

    DetectionStore(
            db::IdTo<db::eye::DetectionGroup> groupById,
            db::IdTo<db::eye::Detection> detectionById,
            db::IdTo<db::eye::Frame> frameById,
            db::IdTo<db::eye::FrameLocation> frameLocationByFrameId,
            db::IdTo<db::eye::FramePrivacy> framePrivacyByFrameId,
            db::IdTo<db::eye::Device> deviceById);

    void extendByDetections(
        pqxx::transaction_base& txn,
        db::eye::Detections detections);

    void extendByDetectionIds(pqxx::transaction_base& txn, const db::TIds& detectionIds);
    void extendByDetectionIds(pqxx::transaction_base& txn, const db::TIdSet& detectionIds);

    // relationMap - связи между детекциями
    // ключ - главная детекция
    // значение - набор связанных детекций
    void extendByObjectRelations(
        pqxx::transaction_base& txn,
        const db::IdTo<db::TIdSet>& relationMap);

    bool hasDetection(db::TId detectionId) const { return detectionById_.count(detectionId); }

    const db::eye::Detection& detectionById(db::TId detectionId) const;

    auto range(db::eye::DetectionType type) const {
        return MakeFilteringRange(
            IterateValues(detectionById_),
            [this, type](const auto& detection) {
                return this->groupByDetectionId(detection.id()).type() == type;
            }
        );
    }

    db::TIdSet detectionIds() const;

    auto detectionIdRange() const { return IterateKeys(detectionById_); }

    const db::IdTo<db::eye::DetectionGroup>& groupById() const { return groupById_; }
    const db::eye::DetectionGroup& groupById(db::TId detectionId) const;
    const db::eye::DetectionGroup& groupByDetectionId(db::TId detectionId) const;

    // frame may be deleted
    const db::IdTo<db::eye::Frame>& frameById() const { return frameById_; }
    db::TId frameId(db::TId detectionId) const;
    const db::eye::Frame& frameById(db::TId frameId) const;
    const db::eye::Frame& frameByDetectionId(db::TId detectionId) const;

    const db::eye::FrameLocation& locationByFrameId(db::TId frameId) const;
    const db::eye::FrameLocation& locationByDetectionId(db::TId detectionId) const;

    const db::eye::FramePrivacy& privacyByFrameId(db::TId frameId) const;
    const db::eye::FramePrivacy& privacyByDetectionId(db::TId detectionId) const;

    const db::eye::Device& deviceById(db::TId deviceId) const;
    const db::eye::Device& deviceByFrameId(db::TId frameId) const;
    const db::eye::Device& deviceByDetectionId(db::TId detectionId) const;

    struct Slice {
        db::eye::DetectionGroups groups;
        db::eye::Detections detections;
        db::eye::Frames frames;
        db::eye::FrameLocations locations;
        db::eye::Devices devices;
    };

    Slice slice(const db::TIdSet& detectionIds) const;

    db::eye::DetectionType getDetectionType(db::TId detectionId) const;

private:
    db::IdTo<db::eye::DetectionGroup> groupById_;
    db::IdTo<db::eye::Detection> detectionById_;

    db::IdTo<db::eye::Frame> frameById_;
    db::IdTo<db::eye::FrameLocation> frameLocationByFrameId_;
    db::IdTo<db::eye::FramePrivacy> framePrivacyByFrameId_;

    db::IdTo<db::eye::Device> deviceById_;

    friend std::vector<DetectionStore> splitByDeviceId(DetectionStore store);

    friend std::vector<DetectionStore> splitByPassages(
        DetectionStore store,
        const SplitPassageParams& params);

    friend DetectionStore mergeDetectionStores(std::vector<DetectionStore> stores);

    friend std::vector<db::TIdSet> makePassages(
        pqxx::transaction_base& txn,
        DetectionStore* storePtr,
        const SplitPassageParams& params,
        const std::chrono::seconds timePad);
};

} // namespace maps::mrc::eye
