#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/tests/fixture.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/store.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>

#include <library/cpp/testing/unittest/registar.h>

namespace maps::mrc::eye::tests {

struct ExtendFixture: public Fixture {
    ExtendFixture();

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::FrameLocations locations;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;
};

ExtendFixture::ExtendFixture()
{
    devices =  insertx<db::eye::DeviceGateway>(
        db::eye::Devices {
            {db::eye::MrcDeviceAttrs{""}}
        }
    );

    frames = db::eye::Frames {
        /* 0 */ {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
        /* 1 */ {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
        /* 2 */ {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
        /* 3 */ {devices[0].id(), identical, makeUrlContext(4, "4"), {1200, 800}, time()},
        /* 4 */ {devices[0].id(), identical, makeUrlContext(5, "5"), {1200, 800}, time()},
        /* 5 */ {devices[0].id(), identical, makeUrlContext(6, "6"), {1200, 800}, time()},
        /* 6 */ {devices[0].id(), identical, makeUrlContext(7, "7"), {1200, 800}, time()},
        /* 7 */ {devices[0].id(), identical, makeUrlContext(8, "8"), {1200, 800}, time()},
    };

    frames[7].setDeleted(true);
    frames = insertx<db::eye::FrameGateway>(std::move(frames));

    locations = insertx<db::eye::FrameLocationGateway>(
        db::eye::FrameLocations {
            {frames[0].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(0), identical)},
            {frames[1].id(), geolib3::Point2{50, 50}, toRotation(geolib3::Heading(0), identical)},
            {frames[2].id(), geolib3::Point2{100, 100}, toRotation(geolib3::Heading(0), identical)},
            {frames[3].id(), geolib3::Point2{300, 300}, toRotation(geolib3::Heading(0), identical)},
            {frames[4].id(), geolib3::Point2{350, 350}, toRotation(geolib3::Heading(0), identical)},
            {frames[5].id(), geolib3::Point2{1000, 1000}, toRotation(geolib3::Heading(0), identical)},
            {frames[6].id(), geolib3::Point2{1020, 1010}, toRotation(geolib3::Heading(0), identical)},
            {frames[7].id(), geolib3::Point2{30, 30}, toRotation(geolib3::Heading(0), identical)}
        }
    );

    groups = insertx<db::eye::DetectionGroupGateway>(
        db::eye::DetectionGroups {
            /* 0 */ {frames[0].id(), db::eye::DetectionType::HouseNumber},
            /* 1 */ {frames[1].id(), db::eye::DetectionType::HouseNumber},
            /* 2 */ {frames[2].id(), db::eye::DetectionType::Sign},
            /* 3 */ {frames[3].id(), db::eye::DetectionType::Sign},
            /* 4 */ {frames[3].id(), db::eye::DetectionType::TrafficLight},
            /* 5 */ {frames[4].id(), db::eye::DetectionType::Sign},
            /* 6 */ {frames[5].id(), db::eye::DetectionType::TrafficLight},
            /* 7 */ {frames[6].id(), db::eye::DetectionType::TrafficLight},
        }
    );

    detections = db::eye::Detections {
        /* 0 */ {groups[0].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
        /* 1 */ {groups[1].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "13"}},
        /* 2 */ {groups[2].id(), db::eye::DetectedSign {
                                    {100, 100, 200, 200},
                                    traffic_signs::TrafficSign::ProhibitoryMaxSpeed80, 0.89,
                                    true, 0.86
                                }
        },
        /* 3 */ {groups[3].id(), db::eye::DetectedSign {
                                    {200, 100, 300, 200},
                                    traffic_signs::TrafficSign::ProhibitoryMaxSpeed80, 0.82,
                                    true, 0.97
                                }
        },
        /* 4 */ {groups[3].id(), db::eye::DetectedSign {
                                    {500, 100, 600, 200},
                                    traffic_signs::TrafficSign::ProhibitoryMaxSpeed80, 0.81,
                                    true, 0.95
                                }
        },
        /* 5 */ {groups[5].id(), db::eye::DetectedSign {
                                    {200, 100, 300, 200},
                                    traffic_signs::TrafficSign::ProhibitoryMaxSpeed80, 0.82,
                                    true, 0.97
                                }
        },
        /* 6 */ {groups[6].id(), db::eye::DetectedTrafficLight{{200, 100, 300, 200}, 0.9}},
        /* 7 */ {groups[6].id(), db::eye::DetectedTrafficLight{{700, 100, 800, 200}, 0.9}},
        /* 8 */ {groups[7].id(), db::eye::DetectedTrafficLight{{200, 100, 300, 200}, 0.9}},
    };

    detections[8].setDeleted();
    detections = insertx<db::eye::DetectionGateway>(std::move(detections));
};

Y_UNIT_TEST_SUITE_F(extend, ExtendFixture)
{

struct ObjectUtilsFixture: public Fixture {
    ObjectUtilsFixture() {
        auto txn = newTxn();

        devices = {
            {db::eye::MrcDeviceAttrs{"M1"}},
        };
        db::eye::DeviceGateway(*txn).insertx(devices);

        frames = {
            {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
            {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
            {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
            {devices[0].id(), identical, makeUrlContext(4, "4"), {1200, 800}, time()},
            {devices[0].id(), identical, makeUrlContext(5, "5"), {1200, 800}, time()},
            {devices[0].id(), identical, makeUrlContext(6, "6"), {1200, 800}, time()},
        };
        db::eye::FrameGateway(*txn).insertx(frames);

        frameLocations = {
            {frames[0].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(90), identical)},
            {frames[1].id(), geolib3::Point2{1, 0}, toRotation(geolib3::Heading(90), identical)},
            {frames[2].id(), geolib3::Point2{0, 2}, toRotation(geolib3::Heading(90), identical)},
            {frames[3].id(), geolib3::Point2{0, 2}, toRotation(geolib3::Heading(90), identical)},
            {frames[4].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(90), identical)},
            {frames[5].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(90), identical)},
        };
        db::eye::FrameLocationGateway(*txn).insertx(frameLocations);

        groups = {
            {frames[0].id(), db::eye::DetectionType::HouseNumber},
            {frames[1].id(), db::eye::DetectionType::HouseNumber},
            {frames[2].id(), db::eye::DetectionType::HouseNumber},
            {frames[3].id(), db::eye::DetectionType::HouseNumber},
            {frames[4].id(), db::eye::DetectionType::HouseNumber},
            {frames[5].id(), db::eye::DetectionType::HouseNumber},
        };
        db::eye::DetectionGroupGateway(*txn).insertx(groups);

        detections = {
            {groups[0].id(), db::eye::DetectedHouseNumber{{0, 0, 10, 10}, 1.0, "12"}},
            {groups[1].id(), db::eye::DetectedHouseNumber{{0, 0, 20, 20}, 1.0, "12"}},
            {groups[2].id(), db::eye::DetectedHouseNumber{{0, 0, 10, 10}, 1.0, "12"}},
            {groups[3].id(), db::eye::DetectedHouseNumber{{0, 0, 70, 70}, 1.0, "12"}},
            {groups[4].id(), db::eye::DetectedHouseNumber{{0, 0, 10, 10}, 1.0, "12"}},
            {groups[5].id(), db::eye::DetectedHouseNumber{{0, 0, 60, 60}, 1.0, "12"}},
        };
        db::eye::DetectionGateway(*txn).insertx(detections);

        objects = {
            {detections[0].id(), db::eye::HouseNumberAttrs{"12"}}
        };
        db::eye::ObjectGateway(*txn).insertx(objects);

        relations = {
            {detections[0].id(), detections[1].id()},
            {detections[0].id(), detections[2].id()},
            {detections[0].id(), detections[3].id()},
            {detections[0].id(), detections[4].id()},
            {detections[0].id(), detections[5].id()},
        };
        db::eye::PrimaryDetectionRelationGateway(*txn).insertx(relations);

        txn->commit();
    }

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::FrameLocations frameLocations;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;
    db::eye::Objects objects;
    db::eye::PrimaryDetectionRelations relations;
};

Y_UNIT_TEST_F(extend_by_object_relations, ObjectUtilsFixture)
{
    DetectionStore detectionStore;
    detectionStore.extendByDetections(*newTxn(), {detections[0], detections[1]});
    db::IdTo<db::TIdSet> relationMap{
        {
            detections[0].id(),
            {
                detections[1].id(),
                detections[2].id(),
                detections[3].id(),
                detections[4].id(),
                detections[5].id(),
            },
        },
    };

    detectionStore.extendByObjectRelations(*newTxn(), relationMap);

    db::TIdSet expected{
        detections[0].id(),
        detections[1].id(),
        detections[2].id(),
        detections[3].id(),
        detections[4].id(),
        detections[5].id(),
    };

    UNIT_ASSERT_EQUAL(detectionStore.detectionIds(), expected);
}

} // Y_UNIT_TEST_SUITE

} // namespace maps::mrc::eye::tests
