#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/tests/fixture.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/match_candidates.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>

#include <library/cpp/testing/gmock_in_unittest/gmock.h>
#include <library/cpp/testing/unittest/registar.h>

namespace maps::mrc::eye::tests {

struct MatchCandidatesFixture: public Fixture {
    MatchCandidatesFixture();

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::FrameLocations locations;
    db::eye::FramePrivacies privacies;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;

    std::unique_ptr<DetectionStore> store;
};

MatchCandidatesFixture::MatchCandidatesFixture()
{
    devices = insertx<db::eye::DeviceGateway>(
        db::eye::Devices {
            {db::eye::MrcDeviceAttrs{""}}
        }
    );

    frames = insertx<db::eye::FrameGateway>(
        db::eye::Frames {
            {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
            {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
            {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
            {devices[0].id(), identical, makeUrlContext(4, "4"), {1200, 800}, time()},
        }
    );

    locations = insertx<db::eye::FrameLocationGateway>(
        db::eye::FrameLocations {
            {frames[0].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(75), identical)},
            {frames[1].id(), geolib3::Point2{1, 0}, toRotation(geolib3::Heading(80), identical)},
            {frames[2].id(), geolib3::Point2{70, 0}, toRotation(geolib3::Heading(95), identical)},
            {frames[3].id(), geolib3::Point2{30, 0}, toRotation(geolib3::Heading(90), identical)},
        }
    );

    for (const auto& frame : frames) {
        privacies.emplace_back(frame.id(), db::FeaturePrivacy::Public);
    }
    privacies = insertx<db::eye::FramePrivacyGateway>(std::move(privacies));

    groups = insertx<db::eye::DetectionGroupGateway>(
        db::eye::DetectionGroups {
            {frames[0].id(), db::eye::DetectionType::HouseNumber},
            {frames[1].id(), db::eye::DetectionType::HouseNumber},
            {frames[2].id(), db::eye::DetectionType::HouseNumber},
            {frames[3].id(), db::eye::DetectionType::HouseNumber},
        }
    );

    detections = insertx<db::eye::DetectionGateway>(
        db::eye::Detections {
            {groups[0].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
            {groups[1].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
            {groups[2].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
            {groups[3].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
        }
    );

    store = std::make_unique<DetectionStore>(
        byId(groups),
        byId(detections),
        byId(frames),
        byId(locations),
        byId(privacies),
        byId(devices)
    );
};

Y_UNIT_TEST_SUITE_F(match_candidates, MatchCandidatesFixture)
{

Y_UNIT_TEST(two_empty_sets)
{
    MatchCandidatesParams params{
        .distanceMeters = 50,
        .angleEpsilon = geolib3::Degrees(90.)
    };

    DetectionIdPairSet detectionPairs
        = generateMatchCandidates(
            *store,
            {} /* detectionIds1 */,
            {} /* detectionIds2 */,
            params
        );

    EXPECT_EQ(detectionPairs.size(), 0u);
}

Y_UNIT_TEST(one_empty_set)
{
    MatchCandidatesParams params{
        .distanceMeters = 50,
        .angleEpsilon = geolib3::Degrees(90.)
    };

    DetectionIdPairSet detectionPairs
        = generateMatchCandidates(
            *store,
            {detections[0].id()} /* detectionIds1 */,
            {} /* detectionIds2 */,
            params
        );

    EXPECT_EQ(detectionPairs.size(), 0u);
}

Y_UNIT_TEST(two_equal_detections)
{
    MatchCandidatesParams params{
        .distanceMeters = 50,
        .angleEpsilon = geolib3::Degrees(90.)
    };

    DetectionIdPairSet detectionPairs
        = generateMatchCandidates(
            *store,
            {detections[0].id()} /* detectionIds1 */,
            {detections[0].id()} /* detectionIds2 */,
            params
        );

    EXPECT_EQ(detectionPairs.size(), 0u);
}

Y_UNIT_TEST(base)
{
    MatchCandidatesParams params{
        .distanceMeters = 50,
        .angleEpsilon = geolib3::Degrees(90.)
    };

    DetectionIdPairSet detectionPairs
        = generateMatchCandidates(
            *store,
            {detections[0].id(), detections[2].id()} /* detectionIds1 */,
            {detections[1].id(), detections[3].id()} /* detectionIds2 */,
            params
        );

    DetectionIdPairSet expected{
        {detections[0].id(), detections[1].id()},
        {detections[0].id(), detections[3].id()},
        {detections[2].id(), detections[3].id()},
    };

    EXPECT_EQ(detectionPairs.size(), 3u);

    EXPECT_TRUE(
        std::is_permutation(
            detectionPairs.begin(), detectionPairs.end(),
            expected.begin()
        )
    );
}

Y_UNIT_TEST(unique_pair)
{
    MatchCandidatesParams params{
        .distanceMeters = 50,
        .angleEpsilon = geolib3::Degrees(90.)
    };

    DetectionIdPairSet detectionPairs
        = generateMatchCandidates(
            *store,
            {detections[0].id(), detections[3].id()} /* detectionIds1 */,
            {detections[0].id(), detections[3].id()} /* detectionIds2 */,
            params
        );

    EXPECT_EQ(detectionPairs.size(), 1u);

    const auto& [detectionId1, detectionId2] = *detectionPairs.begin();

    EXPECT_TRUE(
        (detectionId1 == detections[0].id() && detectionId2 == detections[3].id())
        ||
        (detectionId1 == detections[3].id() && detectionId2 == detections[0].id())
    );
}

} // Y_UNIT_TEST_SUITE

} // namespace maps::mrc::eye::tests
