#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/tests/fixture.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/match.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/store_utils.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/gmock_in_unittest/gmock.h>

namespace maps::mrc::eye::tests {

Y_UNIT_TEST_SUITE(store_utils)
{

struct SplitByDeviceIdFixture: public Fixture {
    SplitByDeviceIdFixture() {
        devices = insertx<db::eye::DeviceGateway>(
            db::eye::Devices {
                {db::eye::MrcDeviceAttrs{"M1"}},
                {db::eye::MrcDeviceAttrs{"M2"}},
                {db::eye::MrcDeviceAttrs{"M3"}},
            }
        );

        frames = insertx<db::eye::FrameGateway>(
            db::eye::Frames {
                {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
                {devices[1].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
                {devices[2].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
                {devices[2].id(), identical, makeUrlContext(4, "4"), {1200, 800}, time()},
            }
        );

        locations = insertx<db::eye::FrameLocationGateway>(
            db::eye::FrameLocations {
                {frames[0].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(90), identical)},
                {frames[1].id(), geolib3::Point2{1, 0}, toRotation(geolib3::Heading(90), identical)},
                {frames[2].id(), geolib3::Point2{0, 2}, toRotation(geolib3::Heading(90), identical)},
                {frames[3].id(), geolib3::Point2{0, 2}, toRotation(geolib3::Heading(90), identical)},
            }
        );
        for (const auto& frame : frames) {
            privacies.emplace_back(frame.id(), db::FeaturePrivacy::Public);
        }
        privacies = insertx<db::eye::FramePrivacyGateway>(std::move(privacies));

        groups = insertx<db::eye::DetectionGroupGateway>(
            db::eye::DetectionGroups {
                {frames[0].id(), db::eye::DetectionType::HouseNumber},
                {frames[1].id(), db::eye::DetectionType::HouseNumber},
                {frames[2].id(), db::eye::DetectionType::HouseNumber},
                {frames[3].id(), db::eye::DetectionType::HouseNumber},
            }
        );

        detections = insertx<db::eye::DetectionGateway>(
            db::eye::Detections {
                {groups[0].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[1].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[2].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[3].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
            }
        );

        store = DetectionStore(
            byId(groups),
            byId(detections),
            byId(frames),
            byId(locations),
            byId(privacies),
            byId(devices)
        );
    }

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::FrameLocations locations;
    db::eye::FramePrivacies privacies;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;
    DetectionStore store;
};

Y_UNIT_TEST_F(split_by_device_id, SplitByDeviceIdFixture)
{
    auto txn = pool().slaveTransaction();

    auto stores = splitByDeviceId(std::move(store));

    EXPECT_EQ(stores.size(), 3u);

    std::vector<db::TIdSet> expected{
        {detections[0].id()},
        {detections[1].id()},
        {detections[2].id(), detections[3].id()},
    };

    std::vector<db::TIdSet> result;
    for (const auto& splittedStore : stores) {
        result.push_back(splittedStore.detectionIds());
    }

    EXPECT_THAT(result, ::testing::UnorderedElementsAreArray(expected));
}

struct SplitByRidesFixture: public Fixture {
    SplitByRidesFixture() {
        devices = insertx<db::eye::DeviceGateway>(
            db::eye::Devices {
                {db::eye::MrcDeviceAttrs{"M1"}},
                {db::eye::MrcDeviceAttrs{"M2"}},
                {db::eye::MrcDeviceAttrs{"M3"}},
            }
        );

        frames = insertx<db::eye::FrameGateway>(
            db::eye::Frames {
                {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, chrono::parseIsoDateTime("2016-06-11 10:30:00")},
                {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, chrono::parseIsoDateTime("2016-06-11 10:30:30")},
                {devices[1].id(), identical, makeUrlContext(3, "3"), {1200, 800}, chrono::parseIsoDateTime("2016-06-11 10:30:00")},
                {devices[1].id(), identical, makeUrlContext(4, "4"), {1200, 800}, chrono::parseIsoDateTime("2016-06-11 10:32:00")},
                {devices[2].id(), identical, makeUrlContext(5, "5"), {1200, 800}, chrono::parseIsoDateTime("2016-06-11 10:32:00")},
            }
        );

        locations = insertx<db::eye::FrameLocationGateway>(
            db::eye::FrameLocations {
                {frames[0].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(90), identical)},
                {frames[1].id(), geolib3::Point2{25, 0}, toRotation(geolib3::Heading(90), identical)},
                {frames[2].id(), geolib3::Point2{0, 2}, toRotation(geolib3::Heading(90), identical)},
                {frames[3].id(), geolib3::Point2{0, 100}, toRotation(geolib3::Heading(90), identical)},
                {frames[4].id(), geolib3::Point2{0, 100}, toRotation(geolib3::Heading(90), identical)},
            }
        );

        for (const auto& frame : frames) {
            privacies.emplace_back(frame.id(), db::FeaturePrivacy::Public);
        }
        privacies = insertx<db::eye::FramePrivacyGateway>(std::move(privacies));

        groups = insertx<db::eye::DetectionGroupGateway>(
            db::eye::DetectionGroups {
                {frames[0].id(), db::eye::DetectionType::HouseNumber},
                {frames[1].id(), db::eye::DetectionType::HouseNumber},
                {frames[2].id(), db::eye::DetectionType::HouseNumber},
                {frames[3].id(), db::eye::DetectionType::HouseNumber},
                {frames[4].id(), db::eye::DetectionType::HouseNumber},
            }
        );

        detections = insertx<db::eye::DetectionGateway>(
            db::eye::Detections {
                {groups[0].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[1].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[2].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[3].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[4].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[4].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "13"}},
            }
        );

        store = DetectionStore(
            byId(groups),
            byId(detections),
            byId(frames),
            byId(locations),
            byId(privacies),
            byId(devices)
        );
    }

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::FrameLocations locations;
    db::eye::FramePrivacies privacies;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;
    DetectionStore store;
};

Y_UNIT_TEST_F(split_by_rides, SplitByRidesFixture)
{
    SplitPassageParams params{
        .metersEpsilon = 50,
        .secondsEpsilon = std::chrono::seconds(60),
    };
    auto stores = splitByPassages(std::move(store), params);

    EXPECT_EQ(stores.size(), 4u);

    std::vector<db::TIdSet> expected{
        {detections[0].id(), detections[1].id()},
        {detections[2].id()},
        {detections[3].id()},
        {detections[4].id(), detections[5].id()},
    };

    std::vector<db::TIdSet> result;
    for (const auto& splittedStore : stores) {
        result.push_back(splittedStore.detectionIds());
    }

    EXPECT_THAT(result, ::testing::UnorderedElementsAreArray(expected));
}

struct MakePassagesFixture: public Fixture {
    MakePassagesFixture() {
        devices = insertx<db::eye::DeviceGateway>(
            db::eye::Devices {
                {db::eye::MrcDeviceAttrs{"M1"}},
            }
        );

        frames = insertx<db::eye::FrameGateway>(
            db::eye::Frames {
                {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, chrono::parseIsoDateTime("2016-06-11 10:30:00")},
                {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, chrono::parseIsoDateTime("2016-06-11 10:30:01")},
                {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, chrono::parseIsoDateTime("2016-06-11 10:30:02")},
            }
        );

        locations = insertx<db::eye::FrameLocationGateway>(
            db::eye::FrameLocations {
                {frames[0].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(90), identical)},
                {frames[1].id(), geolib3::Point2{25, 0}, toRotation(geolib3::Heading(90), identical)},
                {frames[2].id(), geolib3::Point2{0, 2}, toRotation(geolib3::Heading(90), identical)},
            }
        );

        for (const auto& frame : frames) {
            privacies.emplace_back(frame.id(), db::FeaturePrivacy::Public);
        }
        privacies = insertx<db::eye::FramePrivacyGateway>(std::move(privacies));

        groups = insertx<db::eye::DetectionGroupGateway>(
            db::eye::DetectionGroups {
                {frames[0].id(), db::eye::DetectionType::HouseNumber},
                {frames[1].id(), db::eye::DetectionType::HouseNumber},
                {frames[2].id(), db::eye::DetectionType::HouseNumber},
            }
        );

        detections = insertx<db::eye::DetectionGateway>(
            db::eye::Detections {
                {groups[0].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[1].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[2].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
            }
        );

    }

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::FrameLocations locations;
    db::eye::FramePrivacies privacies;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;
};

Y_UNIT_TEST_F(make_passages, MakePassagesFixture)
{
    const db::IdTo<db::eye::DetectionGroup> groupById{
        {groups[0].id(), groups[0]},
        {groups[2].id(), groups[2]}
    };
    const db::IdTo<db::eye::Detection> detectionById{
        {detections[0].id(), detections[0]},
        {detections[2].id(), detections[2]}
    };
    const db::IdTo<db::eye::Frame> frameById{
        {frames[0].id(), frames[0]},
        {frames[2].id(), frames[2]}
    };
    const db::IdTo<db::eye::FrameLocation> locationById{
        {frames[0].id(), locations[0]},
        {frames[2].id(), locations[2]}
    };
    const db::IdTo<db::eye::FramePrivacy> privacyById{
        {frames[0].id(), privacies[0]},
        {frames[2].id(), privacies[2]}
    };

    DetectionStore store(
        groupById,
        detectionById,
        frameById,
        locationById,
        privacyById,
        byId(devices)
    );

    const SplitPassageParams params{
        .metersEpsilon = 50,
        .secondsEpsilon = std::chrono::seconds(60)
    };

    const std::chrono::seconds timePad = std::chrono::seconds(10);

    std::vector<db::TIdSet> passages
        = makePassages(*newTxn(), &store, params, timePad);

    db::TIdSet expectedStoreIds{
        detections[0].id(), detections[1].id(), detections[2].id()
    };
    EXPECT_EQ(store.detectionIds(), expectedStoreIds);

    EXPECT_EQ(passages.size(), 1u);
    db::TIdSet expectedPassageIds{
        detections[0].id(), detections[1].id(), detections[2].id()
    };
    EXPECT_EQ(passages[0], expectedPassageIds);
}

struct NeighboringObjectsFixture: public Fixture {
    NeighboringObjectsFixture() {
        devices = insertx<db::eye::DeviceGateway>(
            db::eye::Devices {
                {db::eye::MrcDeviceAttrs{"M1"}},
            }
        );

        frames = insertx<db::eye::FrameGateway>(
            db::eye::Frames {
                {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
                {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
                {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
            }
        );

        frameLocations = insertx<db::eye::FrameLocationGateway>(
            db::eye::FrameLocations {
                {frames[0].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(90), identical)},
                {frames[1].id(), geolib3::Point2{0, 5}, toRotation(geolib3::Heading(90), identical)},
                {frames[2].id(), geolib3::Point2{0, 10}, toRotation(geolib3::Heading(90), identical)},
            }
        );
        for (const auto& frame : frames) {
            privacies.emplace_back(frame.id(), db::FeaturePrivacy::Public);
        }
        privacies = insertx<db::eye::FramePrivacyGateway>(std::move(privacies));

        groups = insertx<db::eye::DetectionGroupGateway>(
            db::eye::DetectionGroups {
                {frames[0].id(), db::eye::DetectionType::HouseNumber},
                {frames[1].id(), db::eye::DetectionType::HouseNumber},
                {frames[2].id(), db::eye::DetectionType::HouseNumber},
            }
        );

        detections = insertx<db::eye::DetectionGateway>(
            db::eye::Detections {
                {groups[0].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[1].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[2].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
            }
        );

        objects = insertx<db::eye::ObjectGateway>(
            db::eye::Objects {
                {detections[0].id(), db::eye::HouseNumberAttrs{"12"}},
                {detections[1].id(), db::eye::HouseNumberAttrs{"12"}},
                {detections[2].id(), db::eye::HouseNumberAttrs{"12"}},
            }
        );

        objectLocations = insertx<db::eye::ObjectLocationGateway>(
            db::eye::ObjectLocations {
                {objects[0].id(), {0, 0}, toRotation(geolib3::Heading(90), identical)},
                {objects[1].id(), {5, 12}, toRotation(geolib3::Heading(90), identical)},
                {objects[2].id(), {12, 15}, toRotation(geolib3::Heading(90), identical)},
            }
        );
    }

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::FrameLocations frameLocations;
    db::eye::FramePrivacies privacies;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;
    db::eye::Objects objects;
    db::eye::ObjectLocations objectLocations;
};

Y_UNIT_TEST_F(load_neighboring_objects, NeighboringObjectsFixture)
{
    DetectionStore store(
        byId(groups),
        byId(detections),
        byId(frames),
        byId(frameLocations),
        byId(privacies),
        byId(devices)
    );

    double distanceMeters = 10;
    auto objects = loadNeighboringObjects(*newTxn(), store, distanceMeters);

    EXPECT_EQ(objects.size(), 2u);

    db::TIdSet result;
    for (const auto& object : objects) {
        result.insert(object.id());
    }

    db::TIdSet expected{objects[0].id(), objects[1].id()};

    EXPECT_EQ(result, expected);
}

struct MergeDetectionStoresFixture: public Fixture {
    MergeDetectionStoresFixture() {
        devices = insertx<db::eye::DeviceGateway>(
            db::eye::Devices {
                {db::eye::MrcDeviceAttrs{"M1"}},
            }
        );

        frames = insertx<db::eye::FrameGateway>(
            db::eye::Frames {
                {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
                {devices[0].id(), identical, makeUrlContext(2, "2"), {1200, 800}, time()},
                {devices[0].id(), identical, makeUrlContext(3, "3"), {1200, 800}, time()},
            }
        );

        frameLocations = insertx<db::eye::FrameLocationGateway>(
            db::eye::FrameLocations {
                {frames[0].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(90), identical)},
                {frames[1].id(), geolib3::Point2{0, 5}, toRotation(geolib3::Heading(90), identical)},
                {frames[2].id(), geolib3::Point2{0, 10}, toRotation(geolib3::Heading(90), identical)},
            }
        );
        for (const auto& frame : frames) {
            privacies.emplace_back(frame.id(), db::FeaturePrivacy::Public);
        }
        privacies = insertx<db::eye::FramePrivacyGateway>(std::move(privacies));

        groups = insertx<db::eye::DetectionGroupGateway>(
            db::eye::DetectionGroups {
                {frames[0].id(), db::eye::DetectionType::HouseNumber},
                {frames[1].id(), db::eye::DetectionType::HouseNumber},
                {frames[2].id(), db::eye::DetectionType::HouseNumber},
            }
        );

        detections = insertx<db::eye::DetectionGateway>(
            db::eye::Detections {
                {groups[0].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[1].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
                {groups[2].id(), db::eye::DetectedHouseNumber{{100, 100, 200, 200}, 1.0, "12"}},
            }
        );
    }

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::FrameLocations frameLocations;
    db::eye::FramePrivacies privacies;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;
};

Y_UNIT_TEST_F(merge_detection_stores, MergeDetectionStoresFixture)
{
    auto txn = newTxn();
    DetectionStore store1;
    store1.extendByDetections(*txn, {detections[0]});
    DetectionStore store2;
    store2.extendByDetections(*txn, {detections[1]});
    DetectionStore store3;
    store3.extendByDetections(*txn, {detections[2]});

    DetectionStore store = mergeDetectionStores({store1, store2, store3});

    db::TIdSet detectionIds = store.detectionIds();

    EXPECT_EQ(
        detectionIds,
        db::TIdSet({detections[0].id(), detections[1].id(), detections[2].id()})
    );
}

Y_UNIT_TEST(RelevanceWithVerdict_ordering)
{
    EXPECT_THAT(RelevanceWithVerdict{.relevance = 0.},
        ::testing::Lt(RelevanceWithVerdict{.relevance = 1.}));

    EXPECT_THAT((RelevanceWithVerdict{.relevance = 0., .verdict = MatchedFrameDetection::Verdict::No}),
        ::testing::Lt(RelevanceWithVerdict{.relevance = 1., .verdict = MatchedFrameDetection::Verdict::No}));

    EXPECT_THAT((RelevanceWithVerdict{.relevance = 0., .verdict = MatchedFrameDetection::Verdict::Yes}),
        ::testing::Lt(RelevanceWithVerdict{.relevance = 1., .verdict = MatchedFrameDetection::Verdict::Yes}));

    EXPECT_THAT((RelevanceWithVerdict{.relevance = 0., .verdict = MatchedFrameDetection::Verdict::No}),
        ::testing::Lt(RelevanceWithVerdict{.relevance = 0., .verdict = MatchedFrameDetection::Verdict::Yes}));

    EXPECT_THAT((RelevanceWithVerdict{.relevance = 1., .verdict = MatchedFrameDetection::Verdict::No}),
        ::testing::Lt(RelevanceWithVerdict{.relevance = 0.}));

    EXPECT_THAT((RelevanceWithVerdict{.relevance = 1.,}),
        ::testing::Lt(RelevanceWithVerdict{.relevance = 0., .verdict = MatchedFrameDetection::Verdict::Yes}));

    EXPECT_FALSE((RelevanceWithVerdict{.relevance = 0., .verdict = MatchedFrameDetection::Verdict::No}) <
        (RelevanceWithVerdict{.relevance = 0., .verdict = MatchedFrameDetection::Verdict::No}));
}

} // Y_UNIT_TEST_SUITE

} // namespace maps::mrc::eye::tests
